package shop;

import isj.ISJUtil;
import java.awt.geom.Point2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import map.MapPanel;

/**
 * みずほ銀行の住所を取得するクラスです。
 * @author Kumano Tatsuo
 * 2006/03/12
 */
public class Mizuho {
	/**
	 * 最初のURL
	 */
	private static final String URL1 = "http://vip.mapion.co.jp/c/f?uc=21&admi3code=";

	/**
	 * 次のURL
	 */
	private static final String URL2 = "&bool=(%2Badmi3code)&grp=mizuho&oi=key6&ob=0&mx=230";

	/**
	 * エンコーディング
	 */
	private static final String ENCODING = "EUC-JP";

	/**
	 * キャッシュを保存するディレクトリ
	 */
	private static final String CACHE_DIR = ".map" + File.separator + "shops";

	/**
	 * キャッシュファイル名の接頭語
	 */
	private static final String PREFIX = "mizuho_";

	/**
	 * キャッシュファイル名の接尾語
	 */
	private static final String SUFFIX = ".csv";

	/**
	 * みずほ銀行の座標と店舗名の対応表をホームページ又はキャッシュから取得します。
	 * @param prefectureLabel 都道府県名
	 * @param cityID 市区町村コード
	 * @param isj 街区レベル位置参照情報
	 * @param panel 地図を描画するパネル
	 * @return 座標と店舗名の対応表
	 * @throws IOException 
	 */
	public static Map<Point2D, String> load(final String prefectureLabel, final String cityID,
		final Map<String, Point2D> isj, final MapPanel panel) throws IOException {
		final Map<Point2D, String> ret = new LinkedHashMap<Point2D, String>();
		if (!new File(Mizuho.CACHE_DIR).exists()) {
			new File(Mizuho.CACHE_DIR).mkdirs();
		}
		final String cacheFile = Mizuho.CACHE_DIR + File.separator + Mizuho.PREFIX + cityID
			+ Mizuho.SUFFIX;
		if (!new File(cacheFile).exists()) {
			final PrintWriter out = new PrintWriter(new File(cacheFile), "SJIS");
			ISJUtil.parseAddresses(Mizuho.getAddresses(prefectureLabel, cityID, panel), out, isj);
			out.close();
		}
		final Scanner scanner = new Scanner(new InputStreamReader(new FileInputStream(new File(
			cacheFile)), "SJIS"));
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			final String[] items = line.split(",");
			if (items.length == 4) {
				final double x = Double.parseDouble(items[2]);
				final double y = Double.parseDouble(items[3]);
				ret.put(new Point2D.Double(x, y), "みずほ");
			} else {
				System.out.println("WARNING: データ形式が不正です。" + line);
			}
		}
		scanner.close();
		return ret;
	}

	/**
	 * みずほ銀行の市区町村別ページを解析して住所の一覧を取得します。
	 * @param prefectureLabel 都道府県名
	 * @param cityID 市区町村コード
	 * @param panel 地図を描画するパネル
	 * @return 住所と店舗名の対応表
	 * @throws IOException 
	 * @throws UnsupportedEncodingException 
	 */
	public static Map<String, String> getAddresses(final String prefectureLabel,
		final String cityID, final MapPanel panel) throws IOException {
		final Map<String, String> ret = new LinkedHashMap<String, String>();
		try {
			final URL url = new URL(URL1 + cityID + URL2);
			panel.addMessage(url + "をダウンロードしています。");
			final Scanner scanner = new Scanner(new InputStreamReader(url.openStream(), ENCODING));
			String caption = null;
			final Pattern pattern = Pattern
				.compile("<font [^<>]+\"><a href=\"[^\"<>]+\">([^<>]+)</a></font>");
			final Pattern pattern2 = Pattern.compile("<font [^<>]+>([^<>]+)</font>");
			while (scanner.hasNextLine()) {
				final String line = scanner.nextLine();
				final Matcher matcher = pattern.matcher(line);
				if (matcher.find()) {
					if (matcher.group(1).endsWith("支店")) {
						caption = matcher.group(1);
					}
				}
				final Matcher matcher2 = pattern2.matcher(line);
				if (matcher2.find()) {
					if (caption != null) {
						final String address = matcher2.group(1);
						if (address.startsWith(prefectureLabel)) {
							ret.put(address, caption);
							caption = null;
						}
					}
				}
			}
			scanner.close();
			panel.removeMessage();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		return ret;
	}
}
