package shop;

import isj.ISJUtil;
import java.awt.geom.Point2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import map.MapPanel;

/**
 * am/pmの住所を取得するクラスです。
 * @author Kumano Tatsuo
 * @since 3.16
 * 2005/12/04
 */
public class AmPm {
	/**
	 * 最初のURL
	 */
	private static final String URL = "http://www3.ampm.co.jp/kensaku/?item=address&key=";

	/**
	 * エンコーディング
	 */
	private static final String ENCODING = "EUC-JP";

	/**
	 * キャッシュを保存するディレクトリ
	 */
	private static final String CACHE_DIR = ".map" + File.separator + "shops";

	/**
	 * キャッシュファイル名の接頭語
	 */
	private static final String PREFIX = "ampm_";

	/**
	 * キャッシュファイル名の接尾語
	 */
	private static final String SUFFIX = ".csv";

	/**
	 * am/pmの座標と店舗名の対応表をホームページ又はキャッシュから取得します。
	 * @param id 市区町村コード
	 * @param city 市区町村名
	 * @param prefecture 都道府県名
	 * @param isj 街区レベル位置参照情報
	 * @param panel 地図を描画するパネル
	 * @return 座標と店舗名の対応表
	 * @since 3.16
	 * @throws IOException 
	 */
	public static Map<Point2D, String> load(final String id, final String city,
			final String prefecture, final Map<String, Point2D> isj, final MapPanel panel) throws IOException {
		final Map<Point2D, String> ret = new LinkedHashMap<Point2D, String>();
		if (!new File(AmPm.CACHE_DIR).exists()) {
			new File(AmPm.CACHE_DIR).mkdirs();
		}
		final String cacheFile = AmPm.CACHE_DIR + File.separator + AmPm.PREFIX + id + AmPm.SUFFIX;
		if (!new File(cacheFile).exists()) {
			final PrintWriter out = new PrintWriter(new File(cacheFile), "SJIS");
			ISJUtil.parseAddresses(AmPm.getAddresses(city, prefecture, panel), out, isj);
			out.close();
		}
		final Scanner scanner = new Scanner(new InputStreamReader(new FileInputStream(new File(
				cacheFile)), "SJIS"));
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			final String[] items = line.split(",");
			if (items.length == 4) {
				final double x = Double.parseDouble(items[2]);
				final double y = Double.parseDouble(items[3]);
				ret.put(new Point2D.Double(x, y), "am/pm");
			} else {
				System.out.println("WARNING: データ形式が不正です。" + line);
			}
		}
		scanner.close();
		return ret;
	}

	/**
	 * am/pmの市区町村別ページを解析して住所の一覧を取得します。
	 * @param city 市区町村名
	 * @param prefecture 都道府県名
	 * @param panel 地図を描画するパネル
	 * @return 住所と店舗名の対応表
	 * @since 3.16
	 * @throws IOException 
	 * @throws UnsupportedEncodingException 
	 */
	public static Map<String, String> getAddresses(final String city, final String prefecture, final MapPanel panel)
			throws IOException {
		final Map<String, String> ret = new LinkedHashMap<String, String>();
		try {
			final URL url = new URL(URL + URLEncoder.encode(prefecture + city, AmPm.ENCODING));
			panel.addMessage(url + "をダウンロードしています。");
			final Scanner scanner = new Scanner(new InputStreamReader(url.openStream(), ENCODING));
			final Pattern pattern = Pattern.compile("<td [^<>]+><a [^<>]+>([^<>]+)</a></td><td [^<>]+>[^<>]+</td><td [^<>]+>([^<>]+)</td>");
			while (scanner.hasNextLine()) {
				final String line = scanner.nextLine();
				final Matcher matcher = pattern.matcher(line);
				if (matcher.find()) {
					ret.put(matcher.group(2), matcher.group(1));
				}
			}
			scanner.close();
			panel.removeMessage();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		return ret;
	}
}
