package shop;

import isj.ISJUtil;
import java.awt.geom.Point2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import map.MapPanel;

/**
 * 三菱東京UFJ銀行の住所を取得するプログラムです。
 * @author Kumano Tatsuo
 * 2005/12/04
 * @since 3.16
 */
public class UFJ {
	/**
	 * 最初のURL
	 */
	private static final String URL1 = "http://www.mapion.co.jp/c/f?uc=21&pg=3&bool=admi2code*benefit1&grp=bk_mufg&ob=u&oi=key6%252Cedit5&admi2=";

	/**
	 * 次のURL
	 */
	private static final String URL2 = "&benefit1=";

	/**
	 * 最後のURL
	 */
	private static final String URL3 = "&=%B8%A1%A1%A1%BA%F7";

	/**
	 * エンコーディング
	 */
	private static final String ENCODING = "SJIS";

	/**
	 * キャッシュを保存するディレクトリ
	 */
	private static final String CACHE_DIR = ".map" + File.separator + "shops";

	/**
	 * キャッシュファイル名の接頭語
	 */
	private static final String PREFIX = "ufj_";

	/**
	 * キャッシュファイル名の接尾語
	 */
	private static final String SUFFIX = ".csv";

	/**
	 * 三菱東京UFJ銀行の座標と店舗名の対応表をホームページ又はキャッシュから取得します。
	 * @param prefectureLabel 都道府県名
	 * @param cityLabel 市区町村名
	 * @param city 市区町村コード
	 * @param isj 街区レベル位置参照情報
	 * @param panel 地図を描画するパネル
	 * @return 座標と店舗名の対応表
	 * @throws IOException 
	 */
	public static Map<Point2D, String> load(final String prefectureLabel, final String cityLabel,
		final String city, final Map<String, Point2D> isj, final MapPanel panel) throws IOException {
		final Map<Point2D, String> ret = new LinkedHashMap<Point2D, String>();
		if (!new File(UFJ.CACHE_DIR).exists()) {
			new File(UFJ.CACHE_DIR).mkdirs();
		}
		final String cacheFile = UFJ.CACHE_DIR + File.separator + UFJ.PREFIX + city + UFJ.SUFFIX;
		if (!new File(cacheFile).exists()) {
			final PrintWriter out = new PrintWriter(new File(cacheFile), "SJIS");
			ISJUtil.parseAddresses(UFJ.getAddresses(prefectureLabel, cityLabel, city, panel), out, isj);
			out.close();
		}
		final Scanner scanner = new Scanner(new InputStreamReader(new FileInputStream(new File(
			cacheFile)), "SJIS"));
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			final String[] items = line.split(",");
			if (items.length == 4) {
				final double x = Double.parseDouble(items[2]);
				final double y = Double.parseDouble(items[3]);
				ret.put(new Point2D.Double(x, y), "MUFJ");
			} else {
				System.out.println("WARNING: データ形式が不正です。" + line);
			}
		}
		scanner.close();
		return ret;
	}

	/**
	 * 三菱東京UFJ銀行の市区町村別ページを解析して住所の一覧を取得します。
	 * @param prefectureLabel 都道府県名
	 * @param cityLabel 市区町村名
	 * @param city 市区町村コード
	 * @param panel 地図を描画するパネル
	 * @return 住所と店舗名の対応表
	 * @throws IOException 
	 * @throws UnsupportedEncodingException 
	 */
	public static Map<String, String> getAddresses(final String prefectureLabel,
		final String cityLabel, final String city, final MapPanel panel) throws IOException {
		final Map<String, String> ret = new LinkedHashMap<String, String>();
		try {
			final URL url = new URL(URL1 + city.substring(0, 2) + URL2
				+ URLEncoder.encode(cityLabel, "EUC-JP") + URL3);
			panel.addMessage(url + "をダウンロードしています。");
			final Scanner scanner = new Scanner(new InputStreamReader(url.openStream(), ENCODING));
			String caption = null;
			final Pattern pattern = Pattern
				.compile("<a href=\"/c/f\\?uc=[0-9]+&ino=[A-Z0-9]+&pg=[0-9]+&grp=bk_mufg\">([^<>]+)</a>");
			final Pattern pattern2 = Pattern
				.compile("<td bgcolor=\"#[a-z]+\" width=\"[0-9]+\"><font class=\"[a-z]+\">([^<>]+)</font></td>");
			while (scanner.hasNextLine()) {
				final String line = scanner.nextLine();
				final Matcher matcher = pattern.matcher(line);
				if (matcher.find()) {
					caption = matcher.group(1);
				}
				final Matcher matcher2 = pattern2.matcher(line);
				if (matcher2.find()) {
					if (caption != null) {
						if (!caption.startsWith("ATM")) {
							ret.put(prefectureLabel + matcher2.group(1), caption);
						}
					}
				}
			}
			scanner.close();
			panel.removeMessage();
		} catch (FileNotFoundException e) {
		}
		return ret;
	}
}
