package map;

import isj.ISJUtil;
import java.awt.Shape;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Scanner;
import emap.Emap;
import emap.Sej;
import emap.Smbc;
import famima.Famima;

/**
 * 国土数値情報の行政界・海岸線（面）から作成された1つの市区町村を管理するクラスです。
 * @author Kumano Tatsuo
 * 2005/11/11
 */
public class City {
	/**
	 * 数値地図2500（空間データ基盤）を読み込んだかどうか
	 */
	private boolean has2500;

	/**
	 * 外接長方形
	 */
	private final Rectangle2D bounds;

	/**
	 * 高精度の領域
	 */
	private Shape fineShape;

	/**
	 * 市区町村コード
	 */
	private final String id;

	/**
	 * 市区町村名
	 */
	private final String label;

	/**
	 * 都道府県名
	 */
	private final String prefecture;

	/**
	 * 領域
	 */
	private final Shape shape;

	/**
	 * 数値地図2500（空間データ基盤）のURL
	 */
	private final URL url;

	/**
	 * 街区レベル位置参照情報
	 */
	private Map<String, Point2D> isj;

	/**
	 * 街区レベル位置参照情報のラベル位置
	 */
	private Map<Point2D, String> isjLabels;

	/**
	 * 三井住友銀行の一覧
	 * @since 3.10
	 */
	private Collection<PointData> smbc;

	/**
	 * セブンイレブンの一覧
	 * @since 3.11
	 */
	private Collection<PointData> sej;

	/**
	 * ファミリーマートの一覧
	 * @since 3.13
	 */
	private Collection<PointData> famima;

	/**
	 * 市区町村を初期化します。
	 * @param shape 領域
	 * @param label 市区町村名
	 * @param id 市区町村コード
	 * @param url 数値地図2500（空間データ基盤）のURL
	 * @param prefecture 都道府県名
	 */
	public City(final Shape shape, final String label, final String id, final URL url,
			final String prefecture) {
		this.shape = shape;
		this.bounds = shape.getBounds2D();
		this.label = label;
		this.id = id;
		this.url = url;
		this.prefecture = prefecture;
		this.isjLabels = new HashMap<Point2D, String>();
	}

	/**
	 * 街区レベル位置参照情報のラベル位置を空にします。
	 */
	public void clearIsjLabels() {
		this.isjLabels.clear();
	}

	/**
	 * 高精度の領域を開放します。
	 */
	public void freeFineShape() {
		this.fineShape = null;
	}

	/**
	 * 街区レベル位置参照情報を開放します。
	 */
	public void freeIsj() {
		this.isj = null;
	}

	/**
	 * @return 外接長方形
	 */
	public Rectangle2D getBounds() {
		return this.bounds;
	}

	/**
	 * @return 高精度の領域
	 */
	public Shape getFineShape() {
		return this.fineShape;
	}

	/**
	 * @return 市区町村コード
	 */
	public String getId() {
		return this.id;
	}

	/**
	 * @return 街区レベル位置参照情報
	 */
	public Map<String, Point2D> getIsj() {
		return this.isj;
	}

	/**
	 * @return 街区レベル位置参照情報のラベル位置
	 */
	public Map<Point2D, String> getIsjLabels() {
		return this.isjLabels;
	}

	/**
	 * @return 市区町村名
	 */
	public String getLabel() {
		return this.label;
	}

	/**
	 * @return 領域
	 */
	public Shape getShape() {
		return this.shape;
	}

	/**
	 * @return 数値地図2500（空間データ基盤）のURL
	 */
	public URL getURL() {
		return this.url;
	}

	/**
	 * @return 数値地図2500（空間データ基盤）を読み込んだかどうか
	 */
	public boolean has2500() {
		return this.has2500;
	}

	/**
	 * @return 高精度の領域を持っているかどうか
	 */
	public boolean hasFineShape() {
		return this.fineShape != null;
	}

	/**
	 * @return 街区レベル位置参照情報を持っているかどうか
	 */
	public boolean hasIsj() {
		return this.isj != null;
	}

	/**
	 * 街区レベル位置参照情報をダウンロードし、読み込みます。
	 * @throws IOException 
	 */
	public void loadIsj() throws IOException {
		this.isj = ISJUtil.loadIsj(this.id);
	}

	/**
	 * @param shape 高精度の領域
	 */
	public void setFineShape(final Shape shape) {
		this.fineShape = shape;
	}

	/**
	 * @param has2500 数値地図2500（空間データ基盤）を読み込んだかどうか 
	 */
	public void setHas2500(final boolean has2500) {
		this.has2500 = has2500;
	}

	/**
	 * @since 3.10
	 * @return 三井住友銀行の一覧
	 */
	public Collection<PointData> getSmbc() {
		return this.smbc;
	}

	/**
	 * @since 3.11
	 * @return セブンイレブンの一覧
	 */
	public Collection<PointData> getSej() {
		return this.sej;
	}

	/**
	 * @since 3.13
	 * @return ファミリーマートの一覧
	 */
	public Collection<PointData> getFamima() {
		return this.famima;
	}

	/**
	 * @return 三井住友銀行の一覧を持っているかどうか
	 */
	public boolean hasSmbc() {
		return this.smbc != null;
	}

	/**
	 * @return セブンイレブンの一覧を持っているかどうか
	 */
	public boolean hasSej() {
		return this.sej != null;
	}

	/**
	 * @return ファミリーマートの一覧を持っているかどうか
	 */
	public boolean hasFamima() {
		return this.famima != null;
	}

	/**
	 * 三井住友銀行の一覧を読み込みます。
	 * @throws IOException 
	 * @throws MalformedURLException 
	 * @since 3.10
	 */
	public void loadSmbc() throws MalformedURLException, IOException {
		this.smbc = new ArrayList<PointData>();
		loadEmap(new Smbc(), this.smbc, Const.Smbc.CACHE_DIR, Const.Smbc.PREFIX, Const.Smbc.SUFFIX,
				"三井住友");
	}

	/**
	 * セブンイレブンの一覧を読み込みます。
	 * @throws IOException 
	 * @throws MalformedURLException 
	 * @since 3.10
	 */
	public void loadSej() throws MalformedURLException, IOException {
		this.sej = new ArrayList<PointData>();
		loadEmap(new Sej(), this.sej, Const.Sej.CACHE_DIR, Const.Sej.PREFIX, Const.Sej.SUFFIX,
				"セブイレ");
	}

	/**
	 * ファミリーマートの一覧を読み込みます。
	 * @throws IOException 
	 * @since 3.13
	 */
	public void loadFamima() throws IOException {
		this.famima = new ArrayList<PointData>();
		final Map<String, Point2D> tempIsj = new HashMap<String, Point2D>();
		for (final Map.Entry<String, Point2D> entry4 : this.isj.entrySet()) {
			tempIsj.put(entry4.getKey().replaceAll(",", ""), entry4.getValue());
		}
		final Map<Point2D, String> points = Famima.loadFamima(this.id, tempIsj);
		for (final Map.Entry<Point2D, String> entry : points.entrySet()) {
			final Point2D point = entry.getKey();
			final String attribute = entry.getValue();
			final PointData pointData = new PointData(attribute, PointData.CLASSIFICATION_UNKNOWN,
					point.getX(), point.getY());
			pointData.setAttribute(attribute);
			this.famima.add(pointData);
		}
	}

	/**
	 * 三井住友銀行、セブンイレブンの一覧を読み込みます。
	 * @param emap 解析エンジン
	 * @param points 結果を格納するオブジェクト
	 * @param cacheDir キャッシュディレクトリ
	 * @param prefix 接頭語
	 * @param suffix 接尾語
	 * @param attribute 表示する属性
	 * @throws MalformedURLException
	 * @throws IOException
	 * @throws FileNotFoundException
	 * @throws UnsupportedEncodingException
	 */
	private void loadEmap(final Emap emap, final Collection<PointData> points,
			final String cacheDir, final String prefix, final String suffix, final String attribute)
			throws MalformedURLException, IOException, FileNotFoundException,
			UnsupportedEncodingException {
		File csvFile = new File(cacheDir + File.separator + prefix + String.valueOf(this.id)
				+ suffix);
		if (!csvFile.exists()) {
			if (!new File(cacheDir).exists()) {
				new File(cacheDir).mkdir();
			}
			final Map<String, URL> prefectures = emap.getPrefectures();
			for (final Map.Entry<String, URL> entry : prefectures.entrySet()) {
				if (this.prefecture.contains(entry.getKey())) {
					System.out.println("DEBUG: getting " + entry.getValue());
					final Map<String, URL> cities = emap.getCities(entry.getValue());
					for (final Map.Entry<String, URL> entry2 : cities.entrySet()) {
						final String[] strings = entry2.getKey().split(",");
						if (strings.length == 2) {
							final String city = strings[1];
							if (city.equals(this.label)) {
								final PrintWriter out = new PrintWriter(new File(cacheDir
										+ File.separator + prefix + String.valueOf(this.id)
										+ suffix), "SJIS");
								final Map<String, String> addresses = emap.getAddresses(entry2
										.getValue());
								final Map<String, Point2D> tempIsj = new HashMap<String, Point2D>();
								for (final Map.Entry<String, Point2D> entry4 : this.isj.entrySet()) {
									tempIsj.put(entry4.getKey().replaceAll(",", ""), entry4
											.getValue());
								}
								ISJUtil.parseAddresses(addresses, out, tempIsj);
								out.close();
							}
						} else {
							System.out.println("WARNING: データ形式が不正です。" + entry2);
						}
					}
					break;
				}
			}
			csvFile = new File(cacheDir + File.separator + prefix + String.valueOf(this.id)
					+ suffix);
		}
		loadShops(points, attribute, csvFile);
	}

	/**
	 * ファイルからお店の座標を読み込んでPointDataのリストに加えます。
	 * @param points PointDataの一覧
	 * @param attribute 表示する属性
	 * @param csvFile ファイル
	 * @throws UnsupportedEncodingException
	 * @throws FileNotFoundException
	 * @throws NumberFormatException
	 */
	private void loadShops(final Collection<PointData> points, final String attribute, File csvFile)
			throws UnsupportedEncodingException, FileNotFoundException, NumberFormatException {
		if (csvFile.exists()) {
			final Scanner scanner = new Scanner(new InputStreamReader(new FileInputStream(csvFile),
					"SJIS"));
			while (scanner.hasNextLine()) {
				final String line = scanner.nextLine();
				final String[] items = line.split(",");
				if (items.length == 4) {
					final double x = Double.parseDouble(items[2]);
					final double y = Double.parseDouble(items[3]);
					final PointData pointData = new PointData(attribute,
							PointData.CLASSIFICATION_UNKNOWN, x, y);
					pointData.setAttribute(attribute);
					points.add(pointData);
				} else {
					System.out.println("WARNING: データ形式が不正です。" + line);
				}
			}
			scanner.close();
		}
	}

	/**
	 * @return 都道府県名
	 */
	public String getPrefecture() {
		return this.prefecture;
	}
}
