/*
 * This source code is written in UTF-8.
 * To compile, type 'javac -encoding UTF-8 *.java'.
 * To run, type 'java Main'.
 */

import java.io.File;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Scanner;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import cell.CellSearch;
import search.Search;
import map.BackgroundThread;
import map.Const;
import map.MapData;
import map.MainFrame;
import map.MapPanel;

/*
 * Map
 * 国土地理院の数値地図2500（空間データ基盤）を表示するプログラムです。
 * Copyright (C) 2003-2005 Kumano Tatsuo
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
/**
 * 地図を表示するプログラムです。
 * @author  Kumano Tatsuo
 */
public class Main {
    /**
     * プログラム本体です。
     * @param args コマンドライン引数
     * @throws Exception 例外
     */
    public static void main(String[] args) throws Exception {
        final Map<String, MapData> maps = new TreeMap<String, MapData>();
        final Search search = new CellSearch(2000, 1500);
        final MapPanel panel = new MapPanel();
        final MainFrame frame = new MainFrame(panel);
        frame.setVisible(true);
        System.out.println("Loading map infomation...");
        String baseDir = null;
        final List<String> list = new ArrayList<String>();
        if (Const.IS_OFFLINE) {
            baseDir = (args.length > 0) ? args[0] : "data";
            final String[] files = new File(baseDir).list();
            final Pattern pattern = Pattern.compile("[0-9][0-9][a-zA-Z][a-zA-Z][0-9][0-9][0-9]");
            for (int i = 0; i < files.length; i++) {
                if (pattern.matcher(files[i]).matches()) {
                    list.add(files[i].toLowerCase());
                }
            }
        } else {
            baseDir = (args.length > 0) ? Const.SERVER + args[0] : Const.SERVER + "kobe";
            final URL url = new URL(baseDir);
            final Scanner scanner = new Scanner(new InputStreamReader(url.openStream(), "SJIS"));
            while (scanner.hasNextLine()) {
                final String line = scanner.nextLine();
                final Pattern pattern = Pattern.compile("[0-9][0-9][a-zA-Z][a-zA-Z][0-9][0-9][0-9]");
                final Matcher matcher = pattern.matcher(line);
                if (matcher.find()){
                    list.add(matcher.group().toLowerCase());
                }
            }
            scanner.close();
        }
        for (final String mapName : list) {
                MapData map = new MapData(baseDir, mapName);
                maps.put(mapName, map);
                search.insert(map.getBounds(), mapName);
        }
        System.out.println("Map information loading finished.");
        panel.setMapData(maps);
        panel.setSearch(search);
        panel.zoomAutomaticaly();
        panel.moveToCenter();
        new BackgroundThread(maps, panel).start();
    }
}
