package map;

import java.awt.BasicStroke;
import java.awt.Cursor;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionAdapter;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.awt.print.PageFormat;
import java.awt.print.Printable;
import java.awt.print.PrinterException;
import java.io.File;
import java.io.IOException;
import java.util.ConcurrentModificationException;
import java.util.Map;
import javax.imageio.ImageIO;
import javax.swing.JPanel;
import psout.PSOut;
import search.Search;

/**
 * 地図をパネルに描画するクラスです。
 * @author Kumano Tatsuo
 */
public class MapPanel extends JPanel {
    /**
     * 地図
     */
    private Map<String, MapData> maps;

    /**
     * 地図を検索するためのデータ構造
     */
    private Search search;

    /**
     * 倍率が変更されたかどうか
     */
    boolean isZoomChanged; // 倍率が変更されたかどうか

    /**
     * 直前の高さ
     */
    private double lastHeight; // 直前の高さ

    /**
     * 直前のマウスカーソル座標
     */
    double lastMouseX; // 直前のマウスカーソル座標

    /**
     * 直前のマウスカーソル座ひょう
     */
    double lastMouseY; // 直前のマウスカーソル座標

    /**
     * 直前の幅
     */
    private double lastWidth; // 直前の幅

    /**
     * x座標の最大値
     */
    private double maxX;

    /**
     * y座標の最大値
     */
    private double maxY;

    /**
     * x座標の最小値
     */
    private double minX;

    /**
     * y座標の最小値
     */
    private double minY;

    /**
     * オフセット（実座標）
     */
    double offsetX; // オフセット(実座標)

    /**
     * オフセット（実座標）
     */
    double offsetY; // オフセット(実座標)

    /**
     * 表示倍率
     */
    double zoom; // 表示倍率

    /**
     * ユーザ操作がない状態かどうか
     */
    private boolean isIdle; // ユーザ操作がない状態かどうか

    /**
     * ユーザ操作を禁止している状態かどうか
     */
    boolean isBusy; // ユーザ操作ができない状態かどうか

    /**
     * 再描画が必要かどうか
     */
    private boolean needsRepaint; // 再描画が必要かどうか

    /**
     * 直前にマウスのボタンが押されたx座標
     */
    double lastMousePressedX;

    /**
     * 直前にマウスのボタンが押されたy座標
     */
    double lastMousePressedY;

    /**
     * マウスが動いた幅
     */
    double mouseMotionWidth;

    /**
     * マウスが動いた高さ
     */
    double mouseMotionHeight;

    /**
     * イメージ
     */
    private Image image;

    /**
     * 地図の設定
     */
    private MapPreferences mapPreferences; // 地図の設定

    /**
     * コンストラクタです。
     * フレームのサイズを設定し、初期化します。
     */
    public MapPanel() {
        this.mapPreferences = new DefaultMapPreferences();
        this.setBackground(this.mapPreferences.getBackGroundColor());
        this.offsetX = 0;
        this.offsetY = 0;
        this.zoom = 1;
        this.isZoomChanged = true;
        this.lastMouseX = this.offsetX;
        this.lastMouseY = this.offsetY;
        this.lastWidth = getWidth();
        this.lastHeight = getHeight();
        setIdle(true);
        setIsBusy(false);
        setNeedsRepaint(true);
        addMouseListener(new MouseAdapter() {
            public void mousePressed(MouseEvent e) {
                if (!MapPanel.this.isBusy) {
                    MapPanel.this.lastMouseX = e.getX();
                    MapPanel.this.lastMouseY = e.getY();
                    MapPanel.this.lastMousePressedX = e.getX();
                    MapPanel.this.lastMousePressedY = e.getY();
                    setIdle(false);
                }
            }

            public void mouseReleased(MouseEvent e) {
                if (!MapPanel.this.isBusy) {
                    MapPanel.this.mouseMotionWidth = 0;
                    MapPanel.this.mouseMotionHeight = 0;
                    setNeedsRepaint(true);
                    setIdle(true);
                }
            }
        });
        addMouseMotionListener(new MouseMotionAdapter() {
            public void mouseDragged(MouseEvent e) {
                if (MapPanel.this.isBusy) {
                    MapPanel.this.lastMouseX = e.getX();
                    MapPanel.this.lastMouseY = e.getY();
                    MapPanel.this.lastMousePressedX = e.getX();
                    MapPanel.this.lastMousePressedY = e.getY();
                } else {
                    MapPanel.this.offsetX -= (e.getX() - MapPanel.this.lastMouseX);
                    MapPanel.this.offsetY -= (e.getY() - MapPanel.this.lastMouseY);
                    MapPanel.this.lastMouseX = e.getX();
                    MapPanel.this.lastMouseY = e.getY();
                    MapPanel.this.mouseMotionWidth = e.getX() - MapPanel.this.lastMousePressedX;
                    MapPanel.this.mouseMotionHeight = e.getY() - MapPanel.this.lastMousePressedY;
                    repaint();
                }
            }
        });
        addMouseWheelListener(new MouseWheelListener() {
            public void mouseWheelMoved(MouseWheelEvent e) {
                double newZoom = MapPanel.this.zoom * (1 + ((double) e.getWheelRotation() / 50));
                double newX = ((MapPanel.this.offsetX + e.getX()) / MapPanel.this.zoom * newZoom)
                        - e.getX();
                double newY = ((MapPanel.this.offsetY + e.getY()) / MapPanel.this.zoom * newZoom)
                        - e.getY();
                MapPanel.this.offsetX = newX;
                MapPanel.this.offsetY = newY;
                MapPanel.this.zoom = newZoom;
                MapPanel.this.isZoomChanged = true;
                setNeedsRepaint(true);
            }
        });
    }

    /**
     * ユーザ操作ができない状態かどうかを設定します。
     * @param b ユーザ操作ができない状態かどうか
     */
    void setIsBusy(boolean b) {
        this.isBusy = b;
        if (b) {
            setCursor(new Cursor(Cursor.WAIT_CURSOR));
        } else {
            setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
        }
    }

    /**
     * ユーザ操作ができない状態かどうかを取得します。
     * @return ユーザ操作ができない状態かどうか
     */
    boolean isBusy() {
        return this.isBusy;
    }

    /**
     * 再描画が必要かどうかを設定します。
     * @param b 再描画が必要かどうか
     */
    void setNeedsRepaint(boolean b) {
        this.needsRepaint = b;
    }

    /**
     * 地図データを設定します。
     * @param maps maps (String -> MapData)
     */
    public void setMapData(Map<String, MapData> maps) {
        this.maps = maps;
    }

    /**
     * オブジェクトが存在する最も大きい x 座標を取得します。
     * @return オブジェクトが存在する最も大きい x 座標
     */
    double getMaxX() {
        return this.maxX;
    }

    /**
     * オブジェクトが存在する最も大きい y 座標を取得します。
     * @return オブジェクトが存在する最も大きい y 座標
     */
    double getMaxY() {
        return this.maxY;
    }

    /**
     * オブジェクトが存在する最も小さい x 座標を取得します。
     * @return オブジェクトが存在する最も小さい x 座標
     */
    double getMinX() {
        return this.minX;
    }

    /** オブジェクトが存在する最も小さい y 座標を取得します。
     * @return オブジェクトが存在する最も小さい x 座標
     */
    double getMinY() {
        return this.minY;
    }

    /**
     * オブジェクトが存在する範囲を取得します。
     * @return オブジェクトが存在する範囲（仮想座標）
     */
    Rectangle2D getObjectArea() {
        return new Rectangle2D.Double(this.minX, this.minY, this.maxX - this.minX, this.maxY
                - this.minY);
    }

    /**
     * 表示されている範囲を取得します。
     * @return 表示されている範囲（仮想座標）
     */
    Rectangle2D getVisibleRectangle() {
        return new Rectangle2D.Double(this.offsetX / this.zoom, this.offsetY / this.zoom,
                getWidth() / this.zoom, getHeight() / this.zoom);
    }

    /**
     * 倍率を取得します。
     * @return 倍率
     */
    double getZoom() {
        return this.zoom;
    }

    /**
     * 指定したオブジェクトが表示エリア内にあるかどうかを取得します。
     * @param shape オブジェクト
     * @return 指定したオブジェクトが表示エリア内にあるかどうか
     */
    boolean isVisible(Shape shape) {
        return shape.intersects(getVisibleRectangle());
    }

    /**
     * 倍率が変更されたかどうかを取得します。
     * @return 倍率が変更されたかどうか
     */
    boolean isZoomChanged() {
        if (this.isZoomChanged) {
            this.isZoomChanged = false;
            return true;
        } else {
            return false;
        }
    }

    /**
     * 地図の中央が画面の中央になるように、地図をスクロールさせます。
     */
    public void moveToCenter() {
        calcMinMaxXY();
        this.offsetX = ((this.minX + this.maxX) / 2 * this.zoom) - (getWidth() / 2);
        this.offsetY = ((this.minY + this.maxY) / 2 * this.zoom) - (getHeight() / 2);
        setNeedsRepaint(true);
    }

    /**
     * 座標の最大値と最小値を計算します。
     */
    private void calcMinMaxXY() {
        this.minX = Double.POSITIVE_INFINITY;
        this.minY = Double.POSITIVE_INFINITY;
        this.maxX = Double.NEGATIVE_INFINITY;
        this.maxY = Double.NEGATIVE_INFINITY;
        if (this.maps != null) {
            for (MapData mapData : this.maps.values()) {
                Rectangle2D bounds = mapData.getBounds();
                if (bounds.getMinX() < this.minX) {
                    this.minX = bounds.getMinX();
                }
                if (bounds.getMinY() < this.minY) {
                    this.minY = bounds.getMinY();
                }
                if (this.maxX < bounds.getMaxX()) {
                    this.maxX = bounds.getMaxX();
                }
                if (this.maxY < bounds.getMaxY()) {
                    this.maxY = bounds.getMaxY();
                }
            }
        }
    }

    /**
     * パネルが再描画される時に呼び出されます。
     * @param g 描画対象
     */
    public void paint(Graphics g) {
        long startTime = System.currentTimeMillis();
        final long thresholdTime = 200;
        if (!this.isIdle || !this.needsRepaint) {
            if (this.image != null) {
                g.setColor(this.mapPreferences.getBackGroundColor());
                g.fillRect(0, 0, getWidth(), getHeight());
                g.drawImage(this.image, (int) this.mouseMotionWidth, (int) this.mouseMotionHeight,
                        this);
            }
            return;
        }
        setNeedsRepaint(false);
        this.image = createImage(getWidth(), getHeight());
        Graphics2D g2 = (Graphics2D) this.image.getGraphics();
        drawMap(thresholdTime, g2);
        g.drawImage(this.image, 0, 0, this);
        if ((System.currentTimeMillis() - startTime) > thresholdTime) {
            System.out.println("描画：" + (System.currentTimeMillis() - startTime + " ms"));
        }
    }

    /**
     * 地図を描画します。
     * @param thresholdTime 処理に要した時間を表示するしきい値
     * @param g2 グラフィクスコンテキスト
     */
    void drawMap(final long thresholdTime, final Graphics2D g2) {
        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        g2.setColor(this.mapPreferences.getBackGroundColor());
        g2.fillRect(0, 0, getWidth(), getHeight());
        //        AffineTransform transform = new AffineTransform();
        //        transform.translate(-this.offsetX, -this.offsetY);
        //        transform.scale(this.zoom, this.zoom);
        //        g2.setTransform(transform);
        g2.translate(-this.offsetX, -this.offsetY);
        g2.scale(this.zoom, this.zoom);
        double x = this.offsetX / this.zoom;
        double y = this.offsetY / this.zoom;
        double w = getWidth() / this.zoom;
        double h = getHeight() / this.zoom;
        try {
            if (this.maps != null) {
                g2.setStroke(new BasicStroke(1.0f));
                for (MapData mapData : this.maps.values()) {
                    if (mapData.getBounds().intersects(x, y, w, h)) {
                        // 海を描画する
                        if (mapData.hasGyousei()) {
                            g2.setColor(this.mapPreferences.getMizuPreferences().getFillColor());
                            g2.fill(mapData.getBounds());
                            g2.draw(mapData.getBounds());
                        }
                        // 丁目を描画する
                        if (mapData.hasTyome()) {
                            for (PolygonData polygon : mapData.getTyome().values()) {
                                if (polygon.getArea() != null) {
                                    if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_TYOME) {
                                        g2.setColor(this.mapPreferences.getTyomeFillColor(polygon
                                                .getTyomeColorIndex()));
                                        g2.fill(polygon.getArea());
                                    }
                                }
                            }
                        }
                    }
                }
                long startTime2 = System.currentTimeMillis();
                for (MapData mapData : this.maps.values()) {
                    if (mapData.getBounds().intersects(x, y, w, h)) {
                        // 場地を描画する
                        if (mapData.hasZyouti()) {
                            for (PolygonData polygon : mapData.getZyouti().values()) {
                                if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_RAILROAD) {
                                    g2.setColor(this.mapPreferences.getZyoutiPreferences()
                                            .getFillColor());
                                    g2.fill(polygon.getArea());
                                } else if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_PARK) {
                                    g2.setColor(this.mapPreferences.getParkPreferences()
                                            .getFillColor());
                                    g2.fill(polygon.getArea());
                                } else if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_SCHOOL) {
                                    g2.setColor(this.mapPreferences.getZyoutiPreferences()
                                            .getFillColor());
                                    g2.fill(polygon.getArea());
                                } else if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_TEMPLE) {
                                    g2.setColor(this.mapPreferences.getZyoutiPreferences()
                                            .getFillColor());
                                    g2.fill(polygon.getArea());
                                } else if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_GRAVEYARD) {
                                    g2.setColor(this.mapPreferences.getZyoutiPreferences()
                                            .getFillColor());
                                    g2.fill(polygon.getArea());
                                } else if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_OTHER) {
                                    g2.setColor(this.mapPreferences.getZyoutiPreferences()
                                            .getFillColor());
                                    g2.fill(polygon.getArea());
                                }
                            }
                        }
                        // 内水面を描画する
                        if (mapData.hasMizu()) {
                            for (PolygonData polygon : mapData.getMizu().values()) {
                                if (polygon.getArea() != null) {
                                    g2.setColor(this.mapPreferences.getMizuPreferences()
                                            .getFillColor());
                                    g2.fill(polygon.getArea());
                                    g2.draw(polygon.getArea());
                                }
                            }
                        }
                    }
                }
                for (MapData mapData : this.maps.values()) {
                    if (mapData.getBounds().intersects(x, y, w, h)) {
                        // 建物を描画する
                        if (mapData.hasTatemono()) {
                            for (PolygonData polygon : mapData.getTatemono().values()) {
                                if (polygon.getArea() != null) {
                                    g2.setColor(this.mapPreferences.getTatemonoPreferences()
                                            .getFillColor());
                                    g2.fill(polygon.getArea());
                                }
                            }
                        }
                    }
                }
                if ((System.currentTimeMillis() - startTime2) > thresholdTime) {
                    System.out.println("（場地、内水面、建物の塗りつぶし："
                            + (System.currentTimeMillis() - startTime2) + " ms）");
                }
                startTime2 = System.currentTimeMillis();
                g2
                        .setStroke(new BasicStroke(this.mapPreferences.getZyoutiPreferences()
                                .getWidth()));
                for (MapData mapData : this.maps.values()) {
                    if (mapData.getBounds().intersects(x, y, w, h)) {
                        // 場地界を描画する
                        if (mapData.hasZyouti()) {
                            for (ArcData arc : mapData.getOthers().values()) {
                                if (arc.getClassification() != ArcData.TYPE_RAILWAY) {
                                    if (arc.getTag() == ArcData.TAG_NORMAL) {
                                        g2.setColor(this.mapPreferences.getZyoutiPreferences()
                                                .getBorderColor());
                                        g2.draw(arc.getPath());
                                    }
                                }
                            }
                        }
                        // 内水面界を描画する
                        if (mapData.hasMizuArc()) {
                            for (ArcData arc : mapData.getMizuArc().values()) {
                                if (arc.getTag() == ArcData.TAG_NORMAL) {
                                    if (arc.getClassification() == ArcData.TYPE_MIZU_INSIDE) {
                                        g2.setColor(this.mapPreferences.getMizuPreferences()
                                                .getBorderColor());
                                        g2.draw(arc.getPath());
                                    }
                                }
                            }
                        }
                        // 建物界を描画する
                        if (mapData.hasTatemonoArc()) {
                            for (ArcData arc : mapData.getTatemonoArc().values()) {
                                if (arc.getTag() == ArcData.TAG_NORMAL) {
                                    g2.setColor(this.mapPreferences.getTatemonoPreferences()
                                            .getBorderColor());
                                    g2.draw(arc.getPath());
                                }
                            }
                        }
                    }
                }
                if ((System.currentTimeMillis() - startTime2) > thresholdTime) {
                    System.out.println("（場地界、内水面界、建物界の描画："
                            + (System.currentTimeMillis() - startTime2) + " ms）");
                }
                startTime2 = System.currentTimeMillis();
                for (MapData mapData : this.maps.values()) {
                    if (mapData.getBounds().intersects(x, y, w, h)) {
                        // 道路の輪郭を描画する
                        if (mapData.hasRoadArc() && mapData.hasTyome()) {
                            for (ArcData arc : mapData.getRoadArc().values()) {
                                if (arc.getRoadType() == ArcData.ROAD_HIGHWAY) {
                                    g2.setStroke(new BasicStroke(this.mapPreferences
                                            .getHighwayPreferences().getWidth() + 2));
                                    g2.setColor(this.mapPreferences.getHighwayPreferences()
                                            .getBorderColor());
                                    g2.draw(arc.getPath());
                                } else if (arc.getRoadType() == ArcData.ROAD_KOKUDO) {
                                    g2.setStroke(new BasicStroke(this.mapPreferences
                                            .getKokudoPreferences().getWidth() + 2));
                                    g2.setColor(this.mapPreferences.getKokudoPreferences()
                                            .getBorderColor());
                                    g2.draw(arc.getPath());
                                } else if (arc.getRoadType() == ArcData.ROAD_KENDO) {
                                    g2.setStroke(new BasicStroke(this.mapPreferences
                                            .getKendoPreferences().getWidth() + 2));
                                    g2.setColor(this.mapPreferences.getKendoPreferences()
                                            .getBorderColor());
                                    g2.draw(arc.getPath());
                                } else if (arc.getRoadType() == ArcData.ROAD_CHIHODO) {
                                    g2.setStroke(new BasicStroke(this.mapPreferences
                                            .getChihodoPreferences().getWidth() + 2));
                                    g2.setColor(this.mapPreferences.getChihodoPreferences()
                                            .getBorderColor());
                                    g2.draw(arc.getPath());
                                } else if (arc.getRoadType() == ArcData.ROAD_MAJOR) {
                                    g2.setStroke(new BasicStroke(this.mapPreferences
                                            .getMajorRoadPreferences().getWidth() + 2));
                                    g2.setColor(this.mapPreferences.getMajorRoadPreferences()
                                            .getBorderColor());
                                    g2.draw(arc.getPath());
                                } else {
                                    g2.setStroke(new BasicStroke(this.mapPreferences
                                            .getNormalRoadPreferences().getWidth() + 2));
                                    g2.setColor(this.mapPreferences.getNormalRoadPreferences()
                                            .getBorderColor());
                                    g2.draw(arc.getPath());
                                }
                            }
                        }
                    }
                }
                if ((System.currentTimeMillis() - startTime2) > thresholdTime) {
                    System.out.println("（道路の輪郭線の描画：" + (System.currentTimeMillis() - startTime2)
                            + " ms）");
                }
                startTime2 = System.currentTimeMillis();
                for (MapData mapData : this.maps.values()) {
                    if (mapData.getBounds().intersects(x, y, w, h)) {
                        // 道路の塗りつぶし部を描画する
                        if (mapData.hasRoadArc() && mapData.hasTyome()) {
                            // 一般の道路を描画する
                            for (ArcData arc : mapData.getRoadArc().values()) {
                                if (arc.getRoadType() == ArcData.ROAD_NORMAL) {
                                    g2.setColor(this.mapPreferences.getNormalRoadPreferences()
                                            .getFillColor());
                                    g2.setStroke(new BasicStroke(this.mapPreferences
                                            .getNormalRoadPreferences().getWidth()));
                                    g2.draw(arc.getPath());
                                } else if (arc.getRoadType() == ArcData.ROAD_MAJOR) {
                                    g2.setColor(this.mapPreferences.getMajorRoadPreferences()
                                            .getFillColor());
                                    g2.setStroke(new BasicStroke(this.mapPreferences
                                            .getMajorRoadPreferences().getWidth()));
                                    g2.draw(arc.getPath());
                                }
                            }
                            // 主要地方道、県道を描画する
                            for (ArcData arc : mapData.getRoadArc().values()) {
                                if (arc.getRoadType() == ArcData.ROAD_KENDO) {
                                    g2.setStroke(new BasicStroke(this.mapPreferences
                                            .getKendoPreferences().getWidth()));
                                    g2.setColor(this.mapPreferences.getKendoPreferences()
                                            .getFillColor());
                                    g2.draw(arc.getPath());
                                } else if (arc.getRoadType() == ArcData.ROAD_CHIHODO) {
                                    g2.setStroke(new BasicStroke(this.mapPreferences
                                            .getChihodoPreferences().getWidth()));
                                    g2.setColor(this.mapPreferences.getChihodoPreferences()
                                            .getFillColor());
                                    g2.draw(arc.getPath());
                                }
                            }
                            // 国道を描画する
                            g2.setStroke(new BasicStroke(this.mapPreferences.getKokudoPreferences()
                                    .getWidth()));
                            g2.setColor(this.mapPreferences.getKokudoPreferences().getFillColor());
                            for (ArcData arc : mapData.getRoadArc().values()) {
                                if (arc.getRoadType() == ArcData.ROAD_KOKUDO) {
                                    g2.draw(arc.getPath());
                                }
                            }
                            // 高速道路を描画する
                            g2.setStroke(new BasicStroke(this.mapPreferences
                                    .getHighwayPreferences().getWidth()));
                            g2.setColor(this.mapPreferences.getHighwayPreferences().getFillColor());
                            for (ArcData arc : mapData.getRoadArc().values()) {
                                if (arc.getRoadType() == ArcData.ROAD_HIGHWAY) {
                                    g2.draw(arc.getPath());
                                }
                            }
                        }
                    }
                }
                if ((System.currentTimeMillis() - startTime2) > thresholdTime) {
                    System.out.println("（道路の塗りつぶし部の描画：" + (System.currentTimeMillis() - startTime2)
                            + " ms）");
                }
                startTime2 = System.currentTimeMillis();
                for (MapData mapData : this.maps.values()) {
                    if (mapData.getBounds().intersects(x, y, w, h)) {
                        // 行政界を描画する
                        if (mapData.hasGyousei()) {
                            for (ArcData arc : mapData.getGyousei().values()) {
                                if (arc.getTag() == ArcData.TAG_NORMAL) {
                                    if ((arc.getClassification() == ArcData.TYPE_GYOUSEI_PREFECTURE)
                                            || (arc.getClassification() == ArcData.TYPE_GYOUSEI_CITY)) {
                                        g2.setColor(this.mapPreferences.getSi_tyoPreferences()
                                                .getBorderColor());
                                        g2.setStroke(new BasicStroke(this.mapPreferences
                                                .getSi_tyoPreferences().getWidth()));
                                    } else {
                                        g2.setColor(this.mapPreferences.getTyomePreferences()
                                                .getBorderColor());
                                        g2.setStroke(new BasicStroke(this.mapPreferences
                                                .getTyomePreferences().getWidth()));
                                    }
                                    g2.draw(arc.getPath());
                                }
                            }
                        }
                    }
                }
                if ((System.currentTimeMillis() - startTime2) > thresholdTime) {
                    System.out.println("（行政界の描画：" + (System.currentTimeMillis() - startTime2)
                            + " ms）");
                }
                float jrShinkansenRailwayWidth = this.mapPreferences.getJRShinkansenPreferences()
                        .getWidth();
                float otherRailwayWidth = this.mapPreferences.getRailwayPreferences().getWidth();
                for (MapData mapData : this.maps.values()) {
                    if (mapData.getBounds().intersects(x, y, w, h)) {
                        // 鉄道を描画する
                        if (mapData.hasOthers() && mapData.hasTyome()) {
                            g2.setStroke(new BasicStroke(this.mapPreferences.getJRPreferences()
                                    .getWidth() + 4));
                            for (ArcData arc : mapData.getOthers().values()) {
                                if (arc.getClassification() == ArcData.TYPE_RAILWAY) {
                                    if (arc.getRailwayType() == ArcData.RAILWAY_JR) {
                                        g2.setColor(this.mapPreferences.getJRPreferences()
                                                .getBorderColor());
                                        g2.draw(arc.getPath());
                                    } else if (arc.getRailwayType() == ArcData.RAILWAY_JR_SHINKANSEN) {
                                        g2.setColor(this.mapPreferences
                                                .getJRShinkansenPreferences().getBorderColor());
                                        g2.draw(arc.getPath());
                                    }
                                }
                            }
                            for (ArcData arc : mapData.getOthers().values()) {
                                if (arc.getClassification() == ArcData.TYPE_RAILWAY) {
                                    if (arc.getRailwayType() == ArcData.RAILWAY_JR) {
                                        g2.setStroke(new BasicStroke(this.mapPreferences
                                                .getJRPreferences().getWidth(),
                                                BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER, 10,
                                                new float[] { this.mapPreferences
                                                        .getJRPreferences().getWidth() * 5 }, 0));
                                        g2.setColor(this.mapPreferences.getJRPreferences()
                                                .getFillColor());
                                        g2.draw(arc.getPath());
                                    } else if (arc.getRailwayType() == ArcData.RAILWAY_JR_SHINKANSEN) {
                                        g2.setStroke(new BasicStroke(jrShinkansenRailwayWidth,
                                                BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER, 10,
                                                new float[] { this.mapPreferences
                                                        .getJRPreferences().getWidth() * 15 }, 0));
                                        g2.setColor(this.mapPreferences
                                                .getJRShinkansenPreferences().getFillColor());
                                        g2.draw(arc.getPath());
                                    } else {
                                        g2.setStroke(new BasicStroke(otherRailwayWidth));
                                        g2.setColor(this.mapPreferences.getRailwayPreferences()
                                                .getBorderColor());
                                        g2.draw(arc.getPath());
                                    }
                                }
                            }
                        }
                    }
                }
                g2.setColor(this.mapPreferences.getRailwayPreferences().getBorderColor());
                for (MapData mapData : this.maps.values()) {
                    if (mapData.getBounds().intersects(x, y, w, h)) {
                        // 丁目の情報がないときは長方形を描画する
                        if (!mapData.hasGyousei()) {
                            g2.setStroke(new BasicStroke(1f));
                            g2.setColor(this.mapPreferences.getMapBoundsColor());
                            g2.draw(mapData.getBounds());
                            g2.setStroke(new BasicStroke(1f / (float) this.zoom));
                            for (ArcData arc : mapData.getOthers().values()) {
                                if (arc.getClassification() == ArcData.TYPE_RAILWAY) {
                                    g2.draw(arc.getPath());
                                }
                            }
                            // for debug
                            //g2.setFont(new Font("default", Font.PLAIN, 300));
                            //g2.drawString(mapData.getMapName(), (float) mapData.getBounds().getX(), (float) mapData.getBounds().getMaxY());
                        }
                    }
                }
                startTime2 = System.currentTimeMillis();
                //                g2.setTransform(new AffineTransform()); // 座標変換をしない
                g2.scale(1 / this.zoom, 1 / this.zoom);
                g2.translate(this.offsetX, this.offsetY);
                g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
                        RenderingHints.VALUE_ANTIALIAS_OFF);
                g2.setStroke(new BasicStroke(1.0f));
                for (MapData mapData : this.maps.values()) {
                    if (isVisible(mapData.getBounds())) {
                        // 建物のラベルを描画する
                        if (mapData.hasTatemono()) {
                            g2.setColor(this.mapPreferences.getTatemonoPreferences()
                                    .getAttributeColor());
                            g2.setFont(this.mapPreferences.getTatemonoPreferences().getFont());
                            FontMetrics metrics = this.getFontMetrics(this.mapPreferences
                                    .getTatemonoPreferences().getFont());
                            int descent = metrics.getDescent();
                            double size = 4;
                            for (PolygonData polygon : mapData.getTatemono().values()) {
                                if (polygon.getAttribute() != null) {
                                    if (getVisibleRectangle().contains(polygon.getAttributeX(),
                                            polygon.getAttributeY())) {
                                        g2.fill(new Ellipse2D.Double((polygon.getX() * this.zoom)
                                                - this.offsetX - (size / 2),
                                                (polygon.getY() * this.zoom) - this.offsetY
                                                        - (size / 2), size, size));
                                        g2.drawString(polygon.getAttribute(), (float) ((polygon
                                                .getAttributeX() * this.zoom) - this.offsetX),
                                                (float) ((polygon.getAttributeY() * this.zoom)
                                                        - this.offsetY - descent));
                                    }
                                }
                            }
                        }
                        // 場地のラベルを描画する
                        if (mapData.hasZyouti()) {
                            g2.setColor(this.mapPreferences.getZyoutiPreferences()
                                    .getAttributeColor());
                            g2.setFont(this.mapPreferences.getZyoutiPreferences().getFont());
                            FontMetrics metrics = this.getFontMetrics(this.mapPreferences
                                    .getZyoutiPreferences().getFont());
                            int descent = metrics.getDescent();
                            double size = 4;
                            for (PolygonData polygon : mapData.getZyouti().values()) {
                                if (polygon.getAttribute() != null) {
                                    if (getVisibleRectangle().contains(polygon.getAttributeX(),
                                            polygon.getAttributeY())) {
                                        g2.drawString(polygon.getAttribute(), (float) ((polygon
                                                .getAttributeX() * this.zoom) - this.offsetX),
                                                (float) ((polygon.getAttributeY() * this.zoom)
                                                        - this.offsetY - descent));
                                    }
                                }
                            }
                        }
                        // 内水面のラベルを描画する
                        if (mapData.hasMizu()) {
                            g2.setColor(this.mapPreferences.getMizuPreferences()
                                    .getAttributeColor());
                            g2.setFont(this.mapPreferences.getMizuPreferences().getFont());
                            FontMetrics metrics = this.getFontMetrics(this.mapPreferences
                                    .getMizuPreferences().getFont());
                            int descent = metrics.getDescent();
                            double size = 4;
                            for (PolygonData polygon : mapData.getMizu().values()) {
                                if (polygon.getAttribute() != null) {
                                    if (getVisibleRectangle().contains(polygon.getAttributeX(),
                                            polygon.getAttributeY())) {
                                        g2.drawString(polygon.getAttribute(), (float) ((polygon
                                                .getAttributeX() * this.zoom) - this.offsetX),
                                                (float) ((polygon.getAttributeY() * this.zoom)
                                                        - this.offsetY - descent));
                                    }
                                }
                            }
                        }
                        // 丁目のラベルを描画する 
                        if (mapData.hasTyome()) {
                            g2.setColor(this.mapPreferences.getTyomePreferences()
                                    .getAttributeColor());
                            double size = 4;
                            for (PolygonData polygon : mapData.getTyome().values()) {
                                if (polygon.getTyomeFont() != null) {
                                    g2.setFont(polygon.getTyomeFont());
                                    FontMetrics metrics = this.getFontMetrics(polygon
                                            .getTyomeFont());
                                    int descent = metrics.getDescent();
                                    if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_TYOME) {
                                        if (polygon.getAttribute() != null) {
                                            if (getVisibleRectangle().contains(
                                                    polygon.getAttributeX(),
                                                    polygon.getAttributeY())) {
                                                g2
                                                        .drawString(
                                                                polygon.getAttribute(),
                                                                (float) ((polygon.getAttributeX() * this.zoom) - this.offsetX),
                                                                (float) ((polygon.getAttributeY() * this.zoom)
                                                                        - this.offsetY - descent));
                                            }
                                        }
                                    }
                                }
                            }
                        }
                        // 駅のラベルを描画する
                        if (mapData.hasEki()) {
                            double size = this.mapPreferences.getEkiPreferences().getWidth();
                            g2.setFont(this.mapPreferences.getEkiPreferences().getFont());
                            FontMetrics metrics = this.getFontMetrics(this.mapPreferences
                                    .getEkiPreferences().getFont());
                            int descent = metrics.getDescent();
                            for (PointData point : mapData.getEki().values()) {
                                if (point.getAttribute() != null) {
                                    g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
                                            RenderingHints.VALUE_ANTIALIAS_ON);
                                    Ellipse2D ellipse = new Ellipse2D.Double(
                                            (point.getX() * this.zoom) - this.offsetX - (size / 2),
                                            (point.getY() * this.zoom) - this.offsetY - (size / 2),
                                            size, size);
                                    g2.setColor(this.mapPreferences.getEkiPreferences()
                                            .getFillColor());
                                    g2.fill(ellipse);
                                    g2.setColor(this.mapPreferences.getEkiPreferences()
                                            .getBorderColor());
                                    g2.draw(ellipse);
                                    g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
                                            RenderingHints.VALUE_ANTIALIAS_OFF);
                                    g2.drawString(point.getAttribute(), (float) ((point
                                            .getAttributeX() * this.zoom) - this.offsetX),
                                            (float) ((point.getAttributeY() * this.zoom)
                                                    - this.offsetY - descent));
                                }
                            }
                        }
                        // 道路のラベルを描画する
                        if (mapData.hasRoadArc()) {
                            g2.setFont(this.mapPreferences.getNormalRoadPreferences().getFont());
                            FontMetrics metrics = this.getFontMetrics(this.mapPreferences
                                    .getNormalRoadPreferences().getFont());
                            int descent = metrics.getDescent();
                            g2.setColor(this.mapPreferences.getNormalRoadPreferences()
                                    .getAttributeColor());
                            for (ArcData arc : mapData.getRoadArc().values()) {
                                if (arc.getAttribute() != null) {
                                    g2
                                            .drawString(arc.getAttribute(), (float) ((arc
                                                    .getAttributeX() * this.zoom) - this.offsetX),
                                                    (float) ((arc.getAttributeY() * this.zoom)
                                                            - this.offsetY - descent));
                                }
                            }
                        }
                        // 鉄道のラベルを描画する
                        if (mapData.hasOthers()) {
                            g2.setFont(this.mapPreferences.getRailwayPreferences().getFont());
                            FontMetrics metrics = this.getFontMetrics(this.mapPreferences
                                    .getRailwayPreferences().getFont());
                            int descent = metrics.getDescent();
                            g2.setColor(this.mapPreferences.getRailwayPreferences()
                                    .getAttributeColor());
                            for (ArcData arc : mapData.getOthers().values()) {
                                if (arc.getAttribute() != null) {
                                    g2
                                            .drawString(arc.getAttribute(), (float) ((arc
                                                    .getAttributeX() * this.zoom) - this.offsetX),
                                                    (float) ((arc.getAttributeY() * this.zoom)
                                                            - this.offsetY - descent));
                                }
                            }
                        }
                    }
                }
                if ((System.currentTimeMillis() - startTime2) > thresholdTime) {
                    System.out.println("（属性の描画：" + (System.currentTimeMillis() - startTime2)
                            + " ms）");
                }
            }
        } catch (ConcurrentModificationException e) {
        } catch (Exception e) {
            System.err.println("Failed to draw map.");
            e.printStackTrace(System.err);
        }
    }

    /**
     * 実座標を取得します。
     * @param location 仮想座標
     * @return 実座標
     */
    Point2D toRealLocation(Point2D location) {
        return new Point2D.Double((location.getX() * this.zoom) - this.offsetX,
                (location.getY() * this.zoom) - this.offsetY);
    }

    /**
     * 仮想座標を取得します。
     * @param location 実座標
     * @return 仮想座標
     */
    Point2D toVirtualLocation(Point2D location) {
        return new Point2D.Double((this.offsetX + location.getX()) / this.zoom,
                (this.offsetY + location.getY()) / this.zoom);
    }

    /** 地図の縮尺を設定します。
     * @param zoom 縮尺
     */
    void zoom(double zoom) {
        this.zoom = zoom;
        this.isZoomChanged = true;
        setNeedsRepaint(true);
    }

    /**
     * 自動倍率設定します。
     */
    public void zoomAutomaticaly() {
        calcMinMaxXY();
        double zoomX = getWidth() / (this.maxX - this.minX);
        double zoomY = getHeight() / (this.maxY - this.minY);
        if (zoomY < zoomX) {
            this.zoom = zoomY;
        } else {
            this.zoom = zoomX;
        }
        this.isZoomChanged = true;
        setNeedsRepaint(true);
    }

    /**
     * パネルが操作されていない状態かどうかを設定します。
     * @param b パネルが操作されていない状態かどうか
     */
    void setIdle(boolean b) {
        this.isIdle = b;
    }

    /**
     * パネルが操作されていない状態かどうかを取得します。
     * @return パネルが操作されていない状態かどうか
     */
    boolean isIdle() {
        return this.isIdle;
    }

    /**
     * 現在表示されている地図をPSファイルに出力します。
     * @param file ファイル
     * @throws IOException ファイル入出力例外
     * @throws PrinterException 印刷例外
     */
    public void printPS(final File file) throws PrinterException, IOException {
        PSOut.print(file, new Printable() {
            public int print(Graphics graphics, PageFormat pageFormat, int pageIndex)
                    throws PrinterException {
                if (pageIndex == 0) {
                    final Graphics2D g = (Graphics2D) graphics;
                    g.translate(pageFormat.getImageableX(), pageFormat.getImageableY());
                    final double zoom = Math.min(pageFormat.getImageableWidth() / getWidth(),
                            pageFormat.getImageableHeight() / getHeight());
                    g.scale(zoom, zoom);
                    g.setClip(0, 0, getWidth(), getHeight());
                    drawMap(200, g);
                    return Printable.PAGE_EXISTS;
                } else {
                    return Printable.NO_SUCH_PAGE;
                }
            }
        });
    }

    /**
     * 現在表示されている地図をビットマップファイルに出力します。
     * @param file ファイル
     * @param format ファイル形式（png、jpg、bmp）
     * @throws IOException 入出力例外 
     */
    public void printBitmap(final File file, final String format) throws IOException {
        BufferedImage image = new BufferedImage(this.getWidth(), this.getHeight(),
                BufferedImage.TYPE_INT_RGB);
        drawMap(200, (Graphics2D) image.getGraphics());
        ImageIO.write(image, format, file);
    }

    /**
     * @return 地図を検索するためのデータ構造
     */
    public Search getSearch() {
        return this.search;
    }

    /**
     * @param search 地図を検索するためのデータ構造
     */
    public void setSearch(Search search) {
        this.search = search;
    }
}
