package map;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.print.PrinterException;
import java.io.File;
import java.io.IOException;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.WindowConstants;
import javax.swing.filechooser.FileFilter;

/**
 * 地図を表示するフレームです。
 * @author Kumano Tatsuo
 * Created on 2003/11/01, 16:00
 */
public class MainFrame extends JFrame {
    /**
     * 地図を表示するパネル
     */
    final MapPanel panel;

    /**
     * 新しくフレームを初期化します。
     * @param panel 地図を表示するパネル
     */
    public MainFrame(final MapPanel panel) {
        this.panel = panel;
        this.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
        this.setSize(800, 600);
        this.setTitle("Map");
        this.setLayout(new BorderLayout());
        this.add(panel, BorderLayout.CENTER);
        JMenuBar menuBar = new JMenuBar();
        this.add(menuBar, BorderLayout.NORTH);
        JMenu fileMenu = new JMenu("ファイル(F)");
        fileMenu.setMnemonic('F');
        menuBar.add(fileMenu);
        JMenuItem exportMenu = new JMenu("エクスポート(E)");
        exportMenu.setMnemonic('E');
        fileMenu.add(exportMenu);
        JMenuItem imageExportItem = new JMenuItem("ラスタ画像（PNG、JPEG、BMPファイル）(I)...");
        imageExportItem.setMnemonic('I');
        imageExportItem.addActionListener(new ImageExportListener());
        exportMenu.add(imageExportItem);
        JMenuItem psExportItem = new JMenuItem("PSファイル...");
        psExportItem.setMnemonic('P');
        psExportItem.addActionListener(new PSExportListener());
        exportMenu.add(psExportItem);
        fileMenu.addSeparator();
        JMenuItem exitItem = new JMenuItem("終了(X)");
        exitItem.setMnemonic('X');
        exitItem.addActionListener(new ExitListener());
        fileMenu.add(exitItem);
    }

    /**
     * PSファイルを出力する実装です。
     * @author Kumano Tatsuo
     * Created on 2005/03/20 16:42:34
     */
    private class PSExportListener implements ActionListener {
        public void actionPerformed(ActionEvent e) {
            try {
                JFileChooser chooser = new JFileChooser(new File("."));
                chooser.addChoosableFileFilter(new FileFilter() {
                    public boolean accept(File file) {
                        return file.getName().toLowerCase().endsWith(".ps");
                    }

                    public String getDescription() {
                        return "PSファイル（*.ps）";
                    }
                });
                chooser.showDialog(MainFrame.this, "エクスポート");
                final File selectedFile = chooser.getSelectedFile();
                if (selectedFile != null) {
                    final String fileName = selectedFile.getName();
                    if (fileName != null) {
                        if (fileName.toLowerCase().endsWith(".ps")) {
                            MainFrame.this.panel.printPS(selectedFile);
                        } else {
                            MainFrame.this.panel.printPS(new File(fileName + ".ps"));
                        }
                    }
                }
            } catch (PrinterException e1) {
                e1.printStackTrace();
            } catch (IOException e1) {
                e1.printStackTrace();
            }
        }
    }

    /**
     * ラスタ画像に出力する実装です。
     * @author Kumano Tatsuo
     * Created on 2005/02/28 15:39:42
     */
    private class ImageExportListener implements ActionListener {
        public void actionPerformed(ActionEvent e) {
            try {
                JFileChooser chooser = new JFileChooser(new File("."));
                chooser.addChoosableFileFilter(new FileFilter() {
                    public boolean accept(File file) {
                        return file.getName().toLowerCase().endsWith(".png")
                                || file.getName().toLowerCase().endsWith(".jpg")
                                || file.getName().toLowerCase().endsWith(".bmp");
                    }

                    public String getDescription() {
                        return "ラスタ画像ファイル（*.png、*.jpg、*.bmp）";
                    }
                });
                chooser.showDialog(MainFrame.this, "エクスポート");
                final File selectedFile = chooser.getSelectedFile();
                if (selectedFile != null) {
                    final String fileName = selectedFile.getName();
                    if (fileName != null) {
                        if (fileName.toLowerCase().endsWith(".bmp")) {
                            MainFrame.this.panel.printBitmap(selectedFile, "bmp");
                        } else if (fileName.toLowerCase().endsWith(".jpg")) {
                            MainFrame.this.panel.printBitmap(selectedFile, "jpg");
                        } else if (fileName.toLowerCase().endsWith(".png")) {
                            MainFrame.this.panel.printBitmap(selectedFile, "png");
                        }
                    }
                }
            } catch (IOException e1) {
                e1.printStackTrace();
            }
        }
    }

    /**
     * 終了の実装です。
     * @author Kumano Tatsuo
     * Created on 2005/02/28 17:49:02
     */
    private class ExitListener implements ActionListener {
        public void actionPerformed(ActionEvent e) {
            System.exit(0);
        }
    }
}
