import java.util.*;


/*
 * 作成日: 2004/01/09
 */

/**
 * 地図を丁目単位で塗り分けるクラスです。
 * @author Kumano Tatsuo
 */
class PaintTyome {
    private boolean isChanged;

    /** 地図を丁目単位で塗り分けます。
         * このメソッドを呼び出した直後に isChanged() を呼び出すと、
         * このメソッドによって地図の状態が変化したかどうかが取得できます。
         * @param maps 地図
         * @throws Exception 例外
         */
    void paintTyome(Map maps) throws Exception {
        isChanged = false;

        Map attributePolygonMap = new HashMap(); // String -> Collection<Polygon> の Map

        // 属性をキー、ポリゴンの集合を値とする Map を初期化する
        for (Iterator iter = maps.values().iterator(); iter.hasNext();) {
            MapData mapData = (MapData) iter.next();

            if (mapData.hasTyome()) {
                for (Iterator iter2 = mapData.getTyome().values().iterator();
                        iter2.hasNext();) {
                    PolygonData polygon = (PolygonData) iter2.next();
                    String attribute = polygon.getAttribute();

                    if (attribute != null) {
                        if (!attributePolygonMap.containsKey(attribute)) {
                            attributePolygonMap.put(attribute, new ArrayList());
                        }

                        ((Collection) attributePolygonMap.get(attribute)).add(polygon);
                    }
                }
            }
        }

        //System.out.println("attribute <-> polygon = " + attributePolygonMap);
        // Polygon をキー、Collection<Polygon> を値とする Map を作る
        Map adjacentGraph = new HashMap();

        for (Iterator iter = maps.values().iterator(); iter.hasNext();) {
            MapData mapData = (MapData) iter.next();

            if (mapData.hasTyome()) {
                Map adjacentPolygons = mapData.getAdjacentGraph();

                for (Iterator iter2 = adjacentPolygons.keySet().iterator();
                        iter2.hasNext();) {
                    String polygonName = (String) iter2.next();
                    adjacentGraph.put(mapData.getTyome().get(polygonName),
                        adjacentPolygons.get(polygonName));
                }
            }
        }

        //System.out.println("隣接グラフ：" + adjacentGraph);
        // 塗り分ける
        for (Iterator iter = maps.values().iterator(); iter.hasNext();) {
            MapData mapData = (MapData) iter.next();

            if (mapData.hasTyome()) {
                for (Iterator iter2 = mapData.getTyome().values().iterator();
                        iter2.hasNext();) {
                    PolygonData polygon = (PolygonData) iter2.next();

                    if (polygon.getPolygonName() != null) {
                        fixColorRecursively(polygon, adjacentGraph,
                            attributePolygonMap);
                    }
                }
            }
        }
    }

    /**
     * 直前の塗り分けで、地図の状態が変化したかどうかを取得します。
     * @return 地図の状態が変化したかどうか
     */
    boolean isChanged() {
        return isChanged;
    }

    /**
     * 指定したポリゴンの色を再帰的に決定します。
     * red, green, blue, yellow, magenda, cyan の順に色を決めます。
     * @param polygon ポリゴン
     * @param adjacentGraph 隣接グラフ
     * @param attributePolygonMap 属性とポリゴンの関連づけ
     */
    private void fixColorRecursively(PolygonData polygon, Map adjacentGraph,
        Map attributePolygonMap) {
        if (polygon.getTyomeColorIndex() != 0) {
            return;
        }

        boolean[] isUsed = new boolean[8];
        addUsedColors(polygon, adjacentGraph, isUsed);

        String attribute = polygon.getAttribute();

        if (attribute != null) {
            if (attributePolygonMap.containsKey(attribute)) {
                for (Iterator iter = ((Collection) attributePolygonMap.get(
                            attribute)).iterator(); iter.hasNext();) {
                    PolygonData polygon2 = (PolygonData) iter.next();

                    if (polygon != polygon2) {
                        addUsedColors(polygon2, adjacentGraph, isUsed);
                    }
                }
            }
        }

        boolean isPainted = false;

        for (int i = 1; i < 8; ++i) {
            if (!isUsed[i]) {
                polygon.setTyomeColorIndex(i);
                isPainted = true;

                break;
            }

            if (!isPainted) {
                polygon.setTyomeColorIndex(8);
            }
        }

        isChanged = true;

        if (attribute != null) {
            if (attributePolygonMap.containsKey(attribute)) {
                for (Iterator iter = ((Collection) attributePolygonMap.get(
                            attribute)).iterator(); iter.hasNext();) {
                    PolygonData polygon2 = (PolygonData) iter.next();

                    if (polygon != polygon2) {
                        polygon2.setTyomeColorIndex(polygon.getTyomeColorIndex());
                    }
                }
            }
        }

        if (adjacentGraph.containsKey(polygon)) {
            for (Iterator iter = ((Collection) adjacentGraph.get(polygon))
                    .iterator(); iter.hasNext();) {
                PolygonData polygon2 = (PolygonData) iter.next();
                fixColorRecursively(polygon2, adjacentGraph, attributePolygonMap);
            }
        }
    }

    private void addUsedColors(PolygonData polygon, Map adjacentGraph,
        boolean[] isUsed) {
        if (adjacentGraph.containsKey(polygon)) {
            for (Iterator iter = ((Collection) adjacentGraph.get(polygon))
                    .iterator(); iter.hasNext();) {
                PolygonData polygon21 = (PolygonData) iter.next();
                isUsed[polygon21.getTyomeColorIndex()] = true;
            }
        }
    }
}
