import java.awt.BasicStroke;
import java.awt.Cursor;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionAdapter;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.awt.geom.AffineTransform;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;

import javax.swing.JPanel;


/**
 * 地図をパネルに描画するクラスです。
 * @author Kumano Tatsuo
 */
class MapPanel extends JPanel {
    private Map maps;
    private boolean isZoomChanged; // 倍率が変更されたかどうか
    private double lastHeight; // 直前の高さ
    private double lastMouseX; // 直前のマウスカーソル座標
    private double lastMouseY; // 直前のマウスカーソル座標
    private double lastWidth; // 直前の幅
    private double maxX;
    private double maxY;
    private double minX;
    private double minY;
    private double offsetX; // オフセット(実座標)
    private double offsetY; // オフセット(実座標)
    private double zoom; // 表示倍率
    private boolean isIdle; // ユーザ操作がない状態かどうか
    private boolean isBusy; // ユーザ操作ができない状態かどうか
    private boolean needsRepaint; // 再描画が必要かどうか
    private double lastMousePressedX;
    private double lastMousePressedY;
    private double mouseMotionWidth;
    private double mouseMotionHeight;
    private Image image;
    private MapPreferences mapPreferences; // 地図の設定

    /**
     * コンストラクタです。
     * フレームのサイズを設定し、初期化します。
     */
    MapPanel() {
        mapPreferences = new DefaultMapPreferences();
        setBackground(mapPreferences.getBackGroundColor());
        offsetX = 0;
        offsetY = 0;
        zoom = 1;
        isZoomChanged = true;
        lastMouseX = offsetX;
        lastMouseY = offsetY;
        lastWidth = getWidth();
        lastHeight = getHeight();
        setIdle(true);
        setIsBusy(false);
        setNeedsRepaint(true);
        addMouseListener(new MouseAdapter() {
                public void mousePressed(MouseEvent e) {
                    if (!isBusy) {
                        lastMouseX = e.getX();
                        lastMouseY = e.getY();
                        lastMousePressedX = e.getX();
                        lastMousePressedY = e.getY();
                        setIdle(false);
                    }
                }

                public void mouseReleased(MouseEvent e) {
                    if (!isBusy) {
                        mouseMotionWidth = 0;
                        mouseMotionHeight = 0;
                        setNeedsRepaint(true);
                        setIdle(true);
                    }
                }
            });
        addMouseMotionListener(new MouseMotionAdapter() {
                public void mouseDragged(MouseEvent e) {
                    if (isBusy) {
                        lastMouseX = e.getX();
                        lastMouseY = e.getY();
                        lastMousePressedX = e.getX();
                        lastMousePressedY = e.getY();
                    } else {
                        offsetX -= (e.getX() - lastMouseX);
                        offsetY -= (e.getY() - lastMouseY);
                        lastMouseX = e.getX();
                        lastMouseY = e.getY();
                        mouseMotionWidth = e.getX() - lastMousePressedX;
                        mouseMotionHeight = e.getY() - lastMousePressedY;
                        repaint();
                    }
                }
            });
        addMouseWheelListener(new MouseWheelListener() {
                public void mouseWheelMoved(MouseWheelEvent e) {
                    double newZoom = zoom * (1
                        + ((double) e.getWheelRotation() / 50));
                    double newX = ((offsetX + e.getX()) / zoom * newZoom)
                        - e.getX();
                    double newY = ((offsetY + e.getY()) / zoom * newZoom)
                        - e.getY();
                    offsetX = newX;
                    offsetY = newY;
                    zoom = newZoom;
                    isZoomChanged = true;
                    setNeedsRepaint(true);
                }
            });
    }

    /**
     * ユーザ操作ができない状態かどうかを設定します。
     * @param ユーザ操作ができない状態かどうか
     */
    void setIsBusy(boolean b) {
        isBusy = b;

        if (b) {
            setCursor(new Cursor(Cursor.WAIT_CURSOR));
        } else {
            setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
        }
    }

    /**
     * ユーザ操作ができない状態かどうかを取得します。
     * @return ユーザ操作ができない状態かどうか
     */
    boolean isBusy() {
        return isBusy;
    }

    /**
     * 再描画が必要かどうかを設定します。
     * @param b 再描画が必要かどうか
     */
    void setNeedsRepaint(boolean b) {
        needsRepaint = b;
    }

    /**
     * 地図データを設定します。
     * @param maps maps (String -> MapData)
     */
    void setMapData(Map maps) {
        this.maps = maps;
    }

    /**
     * オブジェクトが存在する最も大きい x 座標を取得します。
     * @return オブジェクトが存在する最も大きい x 座標
     */
    double getMaxX() {
        return maxX;
    }

    /**
     * オブジェクトが存在する最も大きい y 座標を取得します。
     * @return オブジェクトが存在する最も大きい y 座標
     */
    double getMaxY() {
        return maxY;
    }

    /**
     * オブジェクトが存在する最も小さい x 座標を取得します。
     * @return オブジェクトが存在する最も小さい x 座標
     */
    double getMinX() {
        return minX;
    }

    /** オブジェクトが存在する最も小さい y 座標を取得します。
     * @return オブジェクトが存在する最も小さい x 座標
     */
    double getMinY() {
        return minY;
    }

    /**
     * オブジェクトが存在する範囲を取得します。
     * @return オブジェクトが存在する範囲（仮想座標）
     */
    Rectangle2D getObjectArea() {
        return new Rectangle2D.Double(minX, minY, maxX - minX, maxY - minY);
    }

    /**
     * 表示されている範囲を取得します。
     * @return 表示されている範囲（仮想座標）
     */
    Rectangle2D getVisibleRectangle() {
        return new Rectangle2D.Double(offsetX / zoom, offsetY / zoom,
            getWidth() / zoom, getHeight() / zoom);
    }

    /**
     * 倍率を取得します。
     * @return 倍率
     */
    double getZoom() {
        return zoom;
    }

    /**
     * 指定したオブジェクトが表示エリア内にあるかどうかを取得します。
     * @param shape オブジェクト
     * @return 指定したオブジェクトが表示エリア内にあるかどうか
     */
    boolean isVisible(Shape shape) {
        return shape.intersects(getVisibleRectangle());
    }

    /**
     * 倍率が変更されたかどうかを取得します。
     * @return 倍率が変更されたかどうか
     */
    boolean isZoomChanged() {
        if (isZoomChanged) {
            isZoomChanged = false;

            return true;
        } else {
            return false;
        }
    }

    /**
     * 地図の中央が画面の中央になるように、地図をスクロールさせます。
     */
    void moveToCenter() {
        calcMinMaxXY();
        offsetX = ((minX + maxX) / 2 * zoom) - (getWidth() / 2);
        offsetY = ((minY + maxY) / 2 * zoom) - (getHeight() / 2);
        setNeedsRepaint(true);
    }

    private void calcMinMaxXY() {
        minX = Double.POSITIVE_INFINITY;
        minY = Double.POSITIVE_INFINITY;
        maxX = Double.NEGATIVE_INFINITY;
        maxY = Double.NEGATIVE_INFINITY;

        if (maps != null) {
            for (Iterator iter = maps.entrySet().iterator(); iter.hasNext();) {
                MapData mapData = (MapData) ((Map.Entry) iter.next()).getValue();
                Rectangle2D bounds = mapData.getBounds();

                if (bounds.getMinX() < minX) {
                    minX = bounds.getMinX();
                }

                if (bounds.getMinY() < minY) {
                    minY = bounds.getMinY();
                }

                if (maxX < bounds.getMaxX()) {
                    maxX = bounds.getMaxX();
                }

                if (maxY < bounds.getMaxY()) {
                    maxY = bounds.getMaxY();
                }
            }
        }
    }

    /**
     * パネルが再描画される時に呼び出されます。
     * @param g 描画対象
     */
    public void paint(Graphics g) {
        long startTime = System.currentTimeMillis();

        if (!isIdle || !needsRepaint) {
            if (image != null) {
                g.setColor(mapPreferences.getBackGroundColor());
                g.fillRect(0, 0, getWidth(), getHeight());
                g.drawImage(image, (int) mouseMotionWidth,
                    (int) mouseMotionHeight, this);
            }

            return;
        }

        setNeedsRepaint(false);
        image = createImage(getWidth(), getHeight());

        Graphics2D g2 = (Graphics2D) image.getGraphics();
        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
            RenderingHints.VALUE_ANTIALIAS_ON);
        g2.setColor(mapPreferences.getBackGroundColor());
        g2.fillRect(0, 0, getWidth(), getHeight());

        AffineTransform transform = new AffineTransform();
        transform.translate(-offsetX, -offsetY);
        transform.scale(zoom, zoom);
        g2.setTransform(transform);

        double x = offsetX / zoom;
        double y = offsetY / zoom;
        double w = getWidth() / zoom;
        double h = getHeight() / zoom;
        Collection usedPoints = new HashSet(); // 使用済みのポリゴンの中心
        int loopCount = 4;
        long time = 200;

        try {
            if (maps != null) {
                g2.setStroke(new BasicStroke(1.0f));

                for (Iterator iter = maps.entrySet().iterator();
                        iter.hasNext();) {
                    MapData mapData = (MapData) ((Map.Entry) iter.next())
                        .getValue();

                    if (mapData.getBounds().intersects(x, y, w, h)) {
                        // 海を描画する
                        if (mapData.hasGyousei()) {
                            g2.setColor(mapPreferences.getMizuPreferences()
                                                      .getFillColor());
                            g2.fill(mapData.getBounds());
                            g2.draw(mapData.getBounds());
                        }

                        // 丁目を描画する
                        if (mapData.hasTyome()) {
                            for (Iterator iter2 = mapData.getTyome().values()
                                                         .iterator();
                                    iter2.hasNext();) {
                                PolygonData polygon = (PolygonData) iter2.next();

                                if (polygon.getArea() != null) {
                                    if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_TYOME) {
                                        g2.setColor(mapPreferences
                                            .getTyomeFillColor(
                                                polygon.getTyomeColorIndex()));
                                        g2.fill(polygon.getArea());
                                    }
                                }
                            }
                        }
                    }
                }

                long startTime2 = System.currentTimeMillis();

                for (Iterator iter = maps.entrySet().iterator();
                        iter.hasNext();) {
                    MapData mapData = (MapData) ((Map.Entry) iter.next())
                        .getValue();

                    if (mapData.getBounds().intersects(x, y, w, h)) {
                        // 場地を描画する
                        if (mapData.hasZyouti()) {
                            for (Iterator iter2 = mapData.getZyouti().entrySet()
                                                         .iterator();
                                    iter2.hasNext();) {
                                PolygonData polygon = ((PolygonData) ((Map.Entry) iter2
                                    .next()).getValue());

                                if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_RAILROAD) {
                                    g2.setColor(mapPreferences.getZyoutiPreferences()
                                                              .getFillColor());
                                    g2.fill(polygon.getArea());
                                } else if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_PARK) {
                                    g2.setColor(mapPreferences.getParkPreferences()
                                                              .getFillColor());
                                    g2.fill(polygon.getArea());
                                } else if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_SCHOOL) {
                                    g2.setColor(mapPreferences.getZyoutiPreferences()
                                                              .getFillColor());
                                    g2.fill(polygon.getArea());
                                } else if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_TEMPLE) {
                                    g2.setColor(mapPreferences.getZyoutiPreferences()
                                                              .getFillColor());
                                    g2.fill(polygon.getArea());
                                } else if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_GRAVEYARD) {
                                    g2.setColor(mapPreferences.getZyoutiPreferences()
                                                              .getFillColor());
                                    g2.fill(polygon.getArea());
                                } else if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_OTHER) {
                                    g2.setColor(mapPreferences.getZyoutiPreferences()
                                                              .getFillColor());
                                    g2.fill(polygon.getArea());
                                }
                            }
                        }

                        // 内水面を描画する
                        if (mapData.hasMizu()) {
                            for (Iterator iter2 = mapData.getMizu().entrySet()
                                                         .iterator();
                                    iter2.hasNext();) {
                                g2.setColor(mapPreferences.getMizuPreferences()
                                                          .getFillColor());

                                PolygonData polygon = (PolygonData) ((Map.Entry) iter2
                                    .next()).getValue();

                                if (polygon.getArea() != null) {
                                    g2.fill(polygon.getArea());
                                    g2.draw(polygon.getArea());
                                }
                            }
                        }
                    }
                }

                for (Iterator iter = maps.entrySet().iterator();
                        iter.hasNext();) {
                    MapData mapData = (MapData) ((Map.Entry) iter.next())
                        .getValue();

                    if (mapData.getBounds().intersects(x, y, w, h)) {
                        // 建物を描画する
                        if (mapData.hasTatemono()) {
                            for (Iterator iter2 = mapData.getTatemono()
                                                         .entrySet().iterator();
                                    iter2.hasNext();) {
                                g2.setColor(mapPreferences.getTatemonoPreferences()
                                                          .getFillColor());

                                PolygonData polygon = (PolygonData) ((Map.Entry) iter2
                                    .next()).getValue();

                                if (polygon.getArea() != null) {
                                    g2.fill(polygon.getArea());
                                }
                            }
                        }
                    }
                }

                if ((System.currentTimeMillis() - startTime2) > time) {
                    System.out.println("（場地、内水面、建物の塗りつぶし："
                        + (System.currentTimeMillis() - startTime2) + " ms）");
                }

                startTime2 = System.currentTimeMillis();
                g2.setStroke(new BasicStroke(mapPreferences.getZyoutiPreferences()
                                                           .getWidth()));

                for (Iterator iter = maps.entrySet().iterator();
                        iter.hasNext();) {
                    MapData mapData = (MapData) ((Map.Entry) iter.next())
                        .getValue();

                    if (mapData.getBounds().intersects(x, y, w, h)) {
                        // 場地界を描画する
                        if (mapData.hasZyouti()) {
                            for (Iterator iter2 = mapData.getOthers().entrySet()
                                                         .iterator();
                                    iter2.hasNext();) {
                                ArcData arc = ((ArcData) ((Map.Entry) iter2
                                    .next()).getValue());

                                if (arc.getClassification() != ArcData.TYPE_RAILWAY) {
                                    if (arc.getTag() == ArcData.TAG_NORMAL) {
                                        g2.setColor(mapPreferences.getZyoutiPreferences()
                                                                  .getBorderColor());
                                        g2.draw(arc.getPath());
                                    }
                                }
                            }
                        }

                        // 内水面界を描画する
                        if (mapData.hasMizuArc()) {
                            for (Iterator iter2 = mapData.getMizuArc().entrySet()
                                                         .iterator();
                                    iter2.hasNext();) {
                                ArcData arc = ((ArcData) ((Map.Entry) iter2
                                    .next()).getValue());

                                if (arc.getTag() == ArcData.TAG_NORMAL) {
                                    if (arc.getClassification() == ArcData.TYPE_MIZU_INSIDE) {
                                        g2.setColor(mapPreferences.getMizuPreferences()
                                                                  .getBorderColor());
                                        g2.draw(arc.getPath());
                                    }
                                }
                            }
                        }

                        // 建物界を描画する
                        if (mapData.hasTatemonoArc()) {
                            for (Iterator iter2 = mapData.getTatemonoArc()
                                                         .entrySet().iterator();
                                    iter2.hasNext();) {
                                ArcData arc = ((ArcData) ((Map.Entry) iter2
                                    .next()).getValue());

                                if (arc.getTag() == ArcData.TAG_NORMAL) {
                                    g2.setColor(mapPreferences.getTatemonoPreferences()
                                                              .getBorderColor());
                                    g2.draw(arc.getPath());
                                }
                            }
                        }
                    }
                }

                if ((System.currentTimeMillis() - startTime2) > time) {
                    System.out.println("（場地界、内水面界、建物界の描画："
                        + (System.currentTimeMillis() - startTime2) + " ms）");
                }

                startTime2 = System.currentTimeMillis();

                for (Iterator iter = maps.entrySet().iterator();
                        iter.hasNext();) {
                    MapData mapData = (MapData) ((Map.Entry) iter.next())
                        .getValue();

                    if (mapData.getBounds().intersects(x, y, w, h)) {
                        // 道路の輪郭を描画する
                        if (mapData.hasRoadArc()) {
                            for (Iterator iter2 = mapData.getRoadArc().values()
                                                         .iterator();
                                    iter2.hasNext();) {
                                ArcData arc = (ArcData) iter2.next();

                                if (arc.getRoadType() == ArcData.ROAD_HIGHWAY) {
                                    g2.setStroke(new BasicStroke(mapPreferences.getHighwayPreferences()
                                                                               .getWidth()
                                            + 2));
                                    g2.setColor(mapPreferences.getHighwayPreferences()
                                                              .getBorderColor());
                                    g2.draw(arc.getPath());
                                } else if (arc.getRoadType() == ArcData.ROAD_KOKUDO) {
                                    g2.setStroke(new BasicStroke(mapPreferences.getKokudoPreferences()
                                                                               .getWidth()
                                            + 2));
                                    g2.setColor(mapPreferences.getKokudoPreferences()
                                                              .getBorderColor());
                                    g2.draw(arc.getPath());
                                } else if (arc.getRoadType() == ArcData.ROAD_KENDO) {
                                    g2.setStroke(new BasicStroke(mapPreferences.getKendoPreferences()
                                                                               .getWidth()
                                            + 2));
                                    g2.setColor(mapPreferences.getKendoPreferences()
                                                              .getBorderColor());
                                    g2.draw(arc.getPath());
                                } else if (arc.getRoadType() == ArcData.ROAD_CHIHODO) {
                                    g2.setStroke(new BasicStroke(mapPreferences.getChihodoPreferences()
                                                                               .getWidth()
                                            + 2));
                                    g2.setColor(mapPreferences.getChihodoPreferences()
                                                              .getBorderColor());
                                    g2.draw(arc.getPath());
                                } else if (arc.getRoadType() == ArcData.ROAD_MAJOR) {
                                    g2.setStroke(new BasicStroke(mapPreferences.getMajorRoadPreferences()
                                                                               .getWidth()
                                            + 2));
                                    g2.setColor(mapPreferences.getMajorRoadPreferences()
                                                              .getBorderColor());
                                    g2.draw(arc.getPath());
                                } else {
                                    g2.setStroke(new BasicStroke(mapPreferences.getNormalRoadPreferences()
                                                                               .getWidth()
                                            + 2));
                                    g2.setColor(mapPreferences.getNormalRoadPreferences()
                                                              .getBorderColor());
                                    g2.draw(arc.getPath());
                                }
                            }
                        }
                    }
                }

                if ((System.currentTimeMillis() - startTime2) > time) {
                    System.out.println("（道路の輪郭線の描画："
                        + (System.currentTimeMillis() - startTime2) + " ms）");
                }

                startTime2 = System.currentTimeMillis();

                for (Iterator iter = maps.entrySet().iterator();
                        iter.hasNext();) {
                    MapData mapData = (MapData) ((Map.Entry) iter.next())
                        .getValue();

                    if (mapData.getBounds().intersects(x, y, w, h)) {
                        // 道路の塗りつぶし部を描画する
                        if (mapData.hasRoadArc()) {
                            // 一般の道路を描画する
                            for (Iterator iter2 = mapData.getRoadArc().values()
                                                         .iterator();
                                    iter2.hasNext();) {
                                ArcData arc = (ArcData) iter2.next();

                                if (arc.getRoadType() == ArcData.ROAD_NORMAL) {
                                    g2.setColor(mapPreferences.getNormalRoadPreferences()
                                                              .getFillColor());
                                    g2.setStroke(new BasicStroke(
                                            mapPreferences.getNormalRoadPreferences()
                                                          .getWidth()));
                                    g2.draw(arc.getPath());
                                } else if (arc.getRoadType() == ArcData.ROAD_MAJOR) {
                                    g2.setColor(mapPreferences.getMajorRoadPreferences()
                                                              .getFillColor());
                                    g2.setStroke(new BasicStroke(
                                            mapPreferences.getMajorRoadPreferences()
                                                          .getWidth()));
                                    g2.draw(arc.getPath());
                                }
                            }

                            // 主要地方道、県道を描画する
                            for (Iterator iter2 = mapData.getRoadArc().values()
                                                         .iterator();
                                    iter2.hasNext();) {
                                ArcData arc = (ArcData) iter2.next();

                                if (arc.getRoadType() == ArcData.ROAD_KENDO) {
                                    g2.setStroke(new BasicStroke(
                                            mapPreferences.getKendoPreferences()
                                                          .getWidth()));
                                    g2.setColor(mapPreferences.getKendoPreferences()
                                                              .getFillColor());
                                    g2.draw(arc.getPath());
                                } else if (arc.getRoadType() == ArcData.ROAD_CHIHODO) {
                                    g2.setStroke(new BasicStroke(
                                            mapPreferences.getChihodoPreferences()
                                                          .getWidth()));
                                    g2.setColor(mapPreferences.getChihodoPreferences()
                                                              .getFillColor());
                                    g2.draw(arc.getPath());
                                }
                            }

                            // 国道を描画する
                            g2.setStroke(new BasicStroke(
                                    mapPreferences.getKokudoPreferences()
                                                  .getWidth()));
                            g2.setColor(mapPreferences.getKokudoPreferences()
                                                      .getFillColor());

                            for (Iterator iter2 = mapData.getRoadArc().values()
                                                         .iterator();
                                    iter2.hasNext();) {
                                ArcData arc = (ArcData) iter2.next();

                                if (arc.getRoadType() == ArcData.ROAD_KOKUDO) {
                                    g2.draw(arc.getPath());
                                }
                            }

                            // 高速道路を描画する
                            g2.setStroke(new BasicStroke(
                                    mapPreferences.getHighwayPreferences()
                                                  .getWidth()));
                            g2.setColor(mapPreferences.getHighwayPreferences()
                                                      .getFillColor());

                            for (Iterator iter2 = mapData.getRoadArc().values()
                                                         .iterator();
                                    iter2.hasNext();) {
                                ArcData arc = (ArcData) iter2.next();

                                if (arc.getRoadType() == ArcData.ROAD_HIGHWAY) {
                                    g2.draw(arc.getPath());
                                }
                            }
                        }
                    }
                }

                if ((System.currentTimeMillis() - startTime2) > time) {
                    System.out.println("（道路の塗りつぶし部の描画："
                        + (System.currentTimeMillis() - startTime2) + " ms）");
                }

                startTime2 = System.currentTimeMillis();

                for (Iterator iter = maps.entrySet().iterator();
                        iter.hasNext();) {
                    MapData mapData = (MapData) ((Map.Entry) iter.next())
                        .getValue();

                    if (mapData.getBounds().intersects(x, y, w, h)) {
                        // 行政界を描画する
                        if (mapData.hasGyousei()) {
                            for (Iterator iter2 = mapData.getGyousei().keySet()
                                                         .iterator();
                                    iter2.hasNext();) {
                                ArcData arc = ((ArcData) mapData.getGyousei()
                                                                .get(iter2.next()));

                                if (arc.getTag() == ArcData.TAG_NORMAL) {
                                    if ((arc.getClassification() == ArcData.TYPE_GYOUSEI_PREFECTURE)
                                            || (arc.getClassification() == ArcData.TYPE_GYOUSEI_CITY)) {
                                        g2.setColor(mapPreferences.getSi_tyoPreferences()
                                                                  .getBorderColor());
                                        g2.setStroke(new BasicStroke(
                                                mapPreferences.getSi_tyoPreferences()
                                                              .getWidth()));
                                    } else {
                                        g2.setColor(mapPreferences.getTyomePreferences()
                                                                  .getBorderColor());
                                        g2.setStroke(new BasicStroke(
                                                mapPreferences.getTyomePreferences()
                                                              .getWidth()));
                                    }

                                    g2.draw(arc.getPath());
                                }
                            }
                        }
                    }
                }

                if ((System.currentTimeMillis() - startTime2) > time) {
                    System.out.println("（行政界の描画："
                        + (System.currentTimeMillis() - startTime2) + " ms）");
                }

                float jrShinkansenRailwayWidth = mapPreferences.getJRShinkansenPreferences()
                                                               .getWidth();
                float otherRailwayWidth = mapPreferences.getRailwayPreferences()
                                                        .getWidth();

                for (Iterator iter = maps.entrySet().iterator();
                        iter.hasNext();) {
                    MapData mapData = (MapData) ((Map.Entry) iter.next())
                        .getValue();

                    if (mapData.getBounds().intersects(x, y, w, h)) {
                        // 鉄道を描画する
                        if (mapData.hasOthers()) {
                            g2.setStroke(new BasicStroke(mapPreferences.getJRPreferences()
                                                                       .getWidth()
                                    + 4));

                            for (Iterator iter2 = mapData.getOthers().entrySet()
                                                         .iterator();
                                    iter2.hasNext();) {
                                ArcData arc = ((ArcData) ((Map.Entry) iter2
                                    .next()).getValue());

                                if (arc.getClassification() == ArcData.TYPE_RAILWAY) {
                                    if (arc.getRailwayType() == ArcData.RAILWAY_JR) {
                                        g2.setColor(mapPreferences.getJRPreferences()
                                                                  .getBorderColor());
                                        g2.draw(arc.getPath());
                                    } else if (arc.getRailwayType() == ArcData.RAILWAY_JR_SHINKANSEN) {
                                        g2.setColor(mapPreferences.getJRShinkansenPreferences()
                                                                  .getBorderColor());
                                        g2.draw(arc.getPath());
                                    }
                                }
                            }

                            for (Iterator iter2 = mapData.getOthers().entrySet()
                                                         .iterator();
                                    iter2.hasNext();) {
                                ArcData arc = ((ArcData) ((Map.Entry) iter2
                                    .next()).getValue());

                                if (arc.getClassification() == ArcData.TYPE_RAILWAY) {
                                    if (arc.getRailwayType() == ArcData.RAILWAY_JR) {
                                        g2.setStroke(new BasicStroke(
                                                mapPreferences.getJRPreferences()
                                                              .getWidth(),
                                                BasicStroke.CAP_BUTT,
                                                BasicStroke.JOIN_MITER, 10,
                                                new float[] {
                                                    mapPreferences.getJRPreferences()
                                                                  .getWidth() * 5
                                                }, 0));
                                        g2.setColor(mapPreferences.getJRPreferences()
                                                                  .getFillColor());
                                        g2.draw(arc.getPath());
                                    } else if (arc.getRailwayType() == ArcData.RAILWAY_JR_SHINKANSEN) {
                                        g2.setStroke(new BasicStroke(
                                                jrShinkansenRailwayWidth,
                                                BasicStroke.CAP_BUTT,
                                                BasicStroke.JOIN_MITER, 10,
                                                new float[] {
                                                    mapPreferences.getJRPreferences()
                                                                  .getWidth() * 15
                                                }, 0));
                                        g2.setColor(mapPreferences.getJRShinkansenPreferences()
                                                                  .getFillColor());
                                        g2.draw(arc.getPath());
                                    } else {
                                        g2.setStroke(new BasicStroke(
                                                otherRailwayWidth));
                                        g2.setColor(mapPreferences.getRailwayPreferences()
                                                                  .getBorderColor());
                                        g2.draw(arc.getPath());
                                    }
                                }
                            }
                        }
                    }
                }

                g2.setStroke(new BasicStroke(1.0f));
                g2.setColor(mapPreferences.getMapBoundsColor());

                for (Iterator iter = maps.entrySet().iterator();
                        iter.hasNext();) {
                    MapData mapData = (MapData) ((Map.Entry) iter.next())
                        .getValue();

                    if (mapData.getBounds().intersects(x, y, w, h)) {
                        // 丁目の情報がないときは長方形を描画する
                        if (!mapData.hasGyousei()) {
                            g2.draw(mapData.getBounds());

                            //g2.setFont(new Font("default", Font.PLAIN, 300));
                            //g2.drawString(mapData.getMapName(), (float) mapData.getBounds().getX(), (float) mapData.getBounds().getMaxY());
                        }
                    }
                }

                startTime2 = System.currentTimeMillis();
                g2.setTransform(new AffineTransform()); // 座標変換をしない
                g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
                    RenderingHints.VALUE_ANTIALIAS_OFF);

                for (Iterator iter = maps.entrySet().iterator();
                        iter.hasNext();) {
                    MapData mapData = (MapData) ((Map.Entry) iter.next())
                        .getValue();

                    if (isVisible(mapData.getBounds())) {
                        // 建物のラベルを描画する
                        if (mapData.hasTatemono()) {
                            g2.setColor(mapPreferences.getTatemonoPreferences()
                                                      .getAttributeColor());
                            g2.setFont(mapPreferences.getTatemonoPreferences()
                                                     .getFont());

                            FontMetrics metrics = this.getFontMetrics(mapPreferences.getTatemonoPreferences()
                                                                                    .getFont());
                            int descent = metrics.getDescent();
                            double size = 4;

                            for (Iterator iter2 = mapData.getTatemono().values()
                                                         .iterator();
                                    iter2.hasNext();) {
                                PolygonData polygon = (PolygonData) iter2.next();

                                if (polygon.getAttribute() != null) {
                                    if (getVisibleRectangle().contains(polygon
                                                .getAttributeX(),
                                                polygon.getAttributeY())) {
                                        g2.fill(new Ellipse2D.Double((polygon
                                                .getX() * zoom) - offsetX
                                                - (size / 2),
                                                (polygon.getY() * zoom)
                                                - offsetY - (size / 2), size,
                                                size));
                                        g2.drawString(polygon.getAttribute(),
                                            (float) ((polygon.getAttributeX() * zoom)
                                            - offsetX),
                                            (float) ((polygon.getAttributeY() * zoom)
                                            - offsetY - descent));
                                    }
                                }
                            }
                        }

                        // 場地のラベルを描画する
                        if (mapData.hasZyouti()) {
                            g2.setColor(mapPreferences.getZyoutiPreferences()
                                                      .getAttributeColor());
                            g2.setFont(mapPreferences.getZyoutiPreferences()
                                                     .getFont());

                            FontMetrics metrics = this.getFontMetrics(mapPreferences.getZyoutiPreferences()
                                                                                    .getFont());
                            int descent = metrics.getDescent();
                            double size = 4;

                            for (Iterator iter2 = mapData.getZyouti().values()
                                                         .iterator();
                                    iter2.hasNext();) {
                                PolygonData polygon = (PolygonData) iter2.next();

                                if (polygon.getAttribute() != null) {
                                    if (getVisibleRectangle().contains(polygon
                                                .getAttributeX(),
                                                polygon.getAttributeY())) {
                                        g2.drawString(polygon.getAttribute(),
                                            (float) ((polygon.getAttributeX() * zoom)
                                            - offsetX),
                                            (float) ((polygon.getAttributeY() * zoom)
                                            - offsetY - descent));
                                    }
                                }
                            }
                        }

                        // 内水面のラベルを描画する
                        if (mapData.hasMizu()) {
                            g2.setColor(mapPreferences.getMizuPreferences()
                                                      .getAttributeColor());
                            g2.setFont(mapPreferences.getMizuPreferences()
                                                     .getFont());

                            FontMetrics metrics = this.getFontMetrics(mapPreferences.getMizuPreferences()
                                                                                    .getFont());
                            int descent = metrics.getDescent();
                            double size = 4;

                            for (Iterator iter2 = mapData.getMizu().values()
                                                         .iterator();
                                    iter2.hasNext();) {
                                PolygonData polygon = (PolygonData) iter2.next();

                                if (polygon.getAttribute() != null) {
                                    if (getVisibleRectangle().contains(polygon
                                                .getAttributeX(),
                                                polygon.getAttributeY())) {
                                        g2.drawString(polygon.getAttribute(),
                                            (float) ((polygon.getAttributeX() * zoom)
                                            - offsetX),
                                            (float) ((polygon.getAttributeY() * zoom)
                                            - offsetY - descent));
                                    }
                                }
                            }
                        }

                        // 丁目のラベルを描画する 
                        if (mapData.hasTyome()) {
                            g2.setColor(mapPreferences.getTyomePreferences()
                                                      .getAttributeColor());

                            double size = 4;

                            for (Iterator iter2 = mapData.getTyome().keySet()
                                                         .iterator();
                                    iter2.hasNext();) {
                                PolygonData polygon = ((PolygonData) mapData.getTyome()
                                                                            .get(iter2
                                        .next()));

                                if (polygon.getTyomeFont() != null) {
                                    g2.setFont(polygon.getTyomeFont());

                                    FontMetrics metrics = this.getFontMetrics(polygon
                                            .getTyomeFont());
                                    int descent = metrics.getDescent();

                                    if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_TYOME) {
                                        if (polygon.getAttribute() != null) {
                                            if (getVisibleRectangle().contains(polygon
                                                        .getAttributeX(),
                                                        polygon.getAttributeY())) {
                                                g2.drawString(polygon
                                                    .getAttribute(),
                                                    (float) ((polygon
                                                    .getAttributeX() * zoom)
                                                    - offsetX),
                                                    (float) ((polygon
                                                    .getAttributeY() * zoom)
                                                    - offsetY - descent));
                                            }
                                        }
                                    }
                                }
                            }
                        }

                        // 駅のラベルを描画する
                        if (mapData.hasEki()) {
                            double size = mapPreferences.getEkiPreferences()
                                                        .getWidth();
                            g2.setFont(mapPreferences.getEkiPreferences()
                                                     .getFont());

                            FontMetrics metrics = this.getFontMetrics(mapPreferences.getEkiPreferences()
                                                                                    .getFont());
                            int descent = metrics.getDescent();

                            for (Iterator iter2 = mapData.getEki().values()
                                                         .iterator();
                                    iter2.hasNext();) {
                                PointData point = (PointData) iter2.next();

                                if (point.getAttribute() != null) {
                                    g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
                                        RenderingHints.VALUE_ANTIALIAS_ON);

                                    Ellipse2D ellipse = new Ellipse2D.Double((point
                                            .getX() * zoom) - offsetX
                                            - (size / 2),
                                            (point.getY() * zoom) - offsetY
                                            - (size / 2), size, size);
                                    g2.setColor(mapPreferences.getEkiPreferences()
                                                              .getFillColor());
                                    g2.fill(ellipse);
                                    g2.setColor(mapPreferences.getEkiPreferences()
                                                              .getBorderColor());
                                    g2.draw(ellipse);
                                    g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
                                        RenderingHints.VALUE_ANTIALIAS_OFF);
                                    g2.drawString(point.getAttribute(),
                                        (float) ((point.getAttributeX() * zoom)
                                        - offsetX),
                                        (float) ((point.getAttributeY() * zoom)
                                        - offsetY - descent));
                                }
                            }
                        }

                        // 道路のラベルを描画する
                        if (mapData.hasRoadArc()) {
                            g2.setFont(mapPreferences.getNormalRoadPreferences()
                                                     .getFont());

                            FontMetrics metrics = this.getFontMetrics(mapPreferences.getNormalRoadPreferences()
                                                                                    .getFont());
                            int descent = metrics.getDescent();
                            g2.setColor(mapPreferences.getNormalRoadPreferences()
                                                      .getAttributeColor());

                            for (Iterator iter2 = mapData.getRoadArc().values()
                                                         .iterator();
                                    iter2.hasNext();) {
                                ArcData arc = (ArcData) iter2.next();

                                if (arc.getAttribute() != null) {
                                    g2.drawString(arc.getAttribute(),
                                        (float) ((arc.getAttributeX() * zoom)
                                        - offsetX),
                                        (float) ((arc.getAttributeY() * zoom)
                                        - offsetY - descent));
                                }
                            }
                        }

                        // 鉄道のラベルを描画する
                        if (mapData.hasOthers()) {
                            g2.setFont(mapPreferences.getRailwayPreferences()
                                                     .getFont());

                            FontMetrics metrics = this.getFontMetrics(mapPreferences.getRailwayPreferences()
                                                                                    .getFont());
                            int descent = metrics.getDescent();
                            g2.setColor(mapPreferences.getRailwayPreferences()
                                                      .getAttributeColor());

                            for (Iterator iter2 = mapData.getOthers().values()
                                                         .iterator();
                                    iter2.hasNext();) {
                                ArcData arc = (ArcData) iter2.next();

                                if (arc.getAttribute() != null) {
                                    g2.drawString(arc.getAttribute(),
                                        (float) ((arc.getAttributeX() * zoom)
                                        - offsetX),
                                        (float) ((arc.getAttributeY() * zoom)
                                        - offsetY - descent));
                                }
                            }
                        }
                    }
                }

                if ((System.currentTimeMillis() - startTime2) > time) {
                    System.out.println("（属性の描画："
                        + (System.currentTimeMillis() - startTime2) + " ms）");
                }
            }
        } catch (Exception e) {
            System.err.println("Failed to draw map.");
            e.printStackTrace(System.err);
        }

        g.drawImage(image, 0, 0, this);

        if ((System.currentTimeMillis() - startTime) > time) {
            System.out.println("描画："
                + (System.currentTimeMillis() - startTime + " ms"));
        }
    }

    /**
     * 実座標を取得します。
     * @param location 仮想座標
     * @return 実座標
     */
    Point2D toRealLocation(Point2D location) {
        return new Point2D.Double((location.getX() * zoom) - offsetX,
            (location.getY() * zoom) - offsetY);
    }

    /**
     * 仮想座標を取得します。
     * @param location 実座標
     * @return 仮想座標
     */
    Point2D toVirtualLocation(Point2D location) {
        return new Point2D.Double((offsetX + location.getX()) / zoom,
            (offsetY + location.getY()) / zoom);
    }

    /** 地図の縮尺を設定します。
     * @param zoom 縮尺
     */
    void zoom(double zoom) {
        this.zoom = zoom;
        isZoomChanged = true;
        setNeedsRepaint(true);
    }

    /**
     * 自動倍率設定します。
     */
    void zoomAutomaticaly() {
        calcMinMaxXY();

        double zoomX = getWidth() / (maxX - minX);
        double zoomY = getHeight() / (maxY - minY);

        if (zoomY < zoomX) {
            zoom = zoomY;
        } else {
            zoom = zoomX;
        }

        isZoomChanged = true;
        setNeedsRepaint(true);
    }

    /**
     * パネルが操作されていない状態かどうかを設定します。
     * @param b パネルが操作されていない状態かどうか
     */
    void setIdle(boolean b) {
        isIdle = b;
    }

    /**
     * パネルが操作されていない状態かどうかを取得します。
     * @return パネルが操作されていない状態かどうか
     */
    boolean isIdle() {
        return isIdle;
    }
}
