﻿/*
	© 2013-2014 FrankHB.

	This file is part of the YSLib project, and may only be used,
	modified, and distributed under the terms of the YSLib project
	license, LICENSE.TXT.  By continuing to use, modify, or distribute
	this file you indicate that you have read the license and
	understand and accept it fully.
*/

/*!	\file Image.h
\ingroup Adaptor
\brief 平台中立的图像输入和输出。
\version r825
\author FrankHB <frankhb1989@gmail.com>
\since build 402
\par 创建时间:
	2013-05-05 12:34:03 +0800
\par 修改时间:
	2014-05-26 14:40 +0800
\par 文本编码:
	UTF-8
\par 模块名称:
	YSLib::Adaptor::Image
*/


#ifndef YSL_INC_Adaptor_Image_h_
#define YSL_INC_Adaptor_Image_h_ 1

#include "../Core/YModules.h"
#include YFM_YSLib_Core_YGDIBase
#include YFM_YSLib_Adaptor_YContainer

//包含 FreeImage 。
//#include <FreeImage.h>

//! \since build 402
struct FIBITMAP;
//! \since build 417
struct FIMEMORY;

namespace YSLib
{

namespace Drawing
{

//! \since build 418
class CompactPixmap;
//! \since build 470
class HBitmap;

//! \since build 402
using BitPerPixel = u8;


/*!
\brief 图像格式。
\note 枚举值和 ::FREE_IMAGE_FORMAT 兼容。
\see FreeImage 宏 FI_ENUM 。
\since build 457
*/
enum class ImageFormat : int
{
	Unknown = -1,
	BMP = 0,
	ICO = 1,
	JPEG = 2,
	//! \since build 470
	PNG = 13,
	GIF = 25
};


/*!
\brief 图像解码器标识。
\note 数值对应 FreeImage 实现。
\since build 457
*/
enum class ImageDecoderFlags : int
{
	Default = 0,
	GIF_Load256 = 1,
	GIF_Playback = 2,
	ICO_MakeAlpha = 1,
	JPEG_Fast = 0x0001,
	JPEG_Accurate = 0x0002,
	JPEG_CMYK = 0x0004,
	JPEG_EXIFRotate = 0x0008,
	JPEG_GreyScale = 0x0010,
	JPEG_QualitySuperb = 0x80,
	JPEG_QualityGood = 0x0100,
	JPEG_QualityNormal = 0x0200,
	JPEG_QualityAverage = 0x0400,
	JPEG_QualityBad = 0x0800,
	JPEG_Progressive = 0x2000,
	JPEG_Subsampling_411 = 0x1000,
	JPEG_Subsampling_420 = 0x4000,
	JPEG_Subsampling_422 = 0x8000,
	JPEG_Subsampling_444 = 0x10000,
	JPEG_Optimize = 0x20000,
	JPEG_Baseline = 0x40000
};

/*!
\relates ImageDecoderFlags
\since build 457
*/
DefBitmaskEnum(ImageDecoderFlags)


/*!
\brief 采样过滤算法。
\note 和 ::FREE_IMAGE_FILTER 兼容。
\see FreeImage 宏 FI_ENUM 。
\since build 430
*/
enum class SamplingFilter
{
	Box = 0,
	Bicubic = 1,
	Bilinear = 2,
	BSpline = 3,
	CatmullRom = 4,
	Lanczos3 = 5
};


/*!
\brief 图像资源分配失败异常：表示存储等资源不足导致无法创建图像。
\since build 402
*/
class YF_API BadImageAlloc : public std::bad_alloc
{};


/*!
\brief 未被支持的图像格式异常：表示请求的操作涉及的图像格式不受库的支持。
\since build 471
*/
class YF_API UnsupportedImageFormat
	: public LoggedEvent, public ystdex::unsupported
{
public:
	UnsupportedImageFormat(const std::string& str)
		: LoggedEvent(str), unsupported()
	{}
};


//! \since build 417
//@{
/*!
\brief 未知图像格式异常：表示请求的操作涉及的图像格式因为不明确而不受库的支持。
*/
class YF_API UnknownImageFormat : public UnsupportedImageFormat
{
public:
	UnknownImageFormat(const std::string& str)
		: UnsupportedImageFormat(str)
	{}
};


/*!
\brief 图像内存对象；保存图像内容的数据缓冲对象。
\note 仅当新建缓冲区时可访问缓冲区数据。
*/
class YF_API ImageMemory final
{
public:
	using NativeHandle = ::FIMEMORY*;
	//! \since build 470
	using Buffer = vector<octet>;

private:
	Buffer buffer;
	NativeHandle handle;
	ImageFormat format;

public:
	/*!
	\exception LoggedEvent 打开内存缓冲区失败。
	\note 需要校验图像格式正确。
	\since build 470
	*/
	//@{
	/*!
	\brief 构造：从现有图像打开。
	\post <tt>GetBuffer().empty()</tt> 。
	\throw UnknownImageFormat 未知图像格式。
	\throw LoggedEvent 图像为空。
	\throw LoggedEvent 图像保存到缓冲区失败。
	*/
	explicit
	ImageMemory(const HBitmap&, ImageFormat = ImageFormat::BMP,
		ImageDecoderFlags = ImageDecoderFlags::Default);
	/*!
	\post <tt>!GetBuffer().empty()</tt> 。
	\exception LoggedEvent 缓冲区大小等于 0 。
	\exception LoggedEvent 打开内存缓冲区失败。
	*/
	//@{
	//! \brief 构造：打开指定的内存缓冲区。
	ImageMemory(Buffer);
	//! \brief 构造：以指定格式打开指定的内存缓冲区。
	ImageMemory(Buffer, ImageFormat);
	//! \brief 构造：新建并打开内存缓冲区。
	template<typename _fCallable>
	ImageMemory(_fCallable f)
		: ImageMemory(f())
	{}
	//! \brief 构造：以指定格式新建并打开内存缓冲区。
	template<typename _fCallable>
	ImageMemory(_fCallable f, ImageFormat fmt)
		: ImageMemory(f(), fmt)
	{}
	//@}
	DefDelMoveCtor(ImageMemory)
	//@}
	//! \since build 461
	~ImageMemory();

	//! \since build 470
	DefGetter(const ynothrow, const Buffer&, Buffer, buffer)
	DefGetter(const ynothrow, ImageFormat, Format, format)
	DefGetter(const ynothrow, NativeHandle, NativeHandle, handle)
};
//@}


/*!
\brief 位图句柄：指向位图数据。
\note 内部数据的平凡容器：深复制且可转移。
\since build 418
*/
class YF_API HBitmap final
{
public:
	using DataPtr = ::FIBITMAP*;

private:
	DataPtr bitmap;

public:
	/*
	\brief 构造：使用现有数据指针。
	\note 取得所有权。
	\since build 430
	*/
	HBitmap(DataPtr ptr = {}) ynothrow
		: bitmap(ptr)
	{}
	//! \throw BadImageAlloc 分配空间失败。
	HBitmap(const Size&, BitPerPixel = 0);
	/*!
	\brief 构造：从矩形像素图缓冲区按指定大小和扫描线跨距增量复制并转换图像数据。
	\pre 断言：输入指针非空。
	\throw LoggedEvent 转换失败。
	\note 扫描线跨距的单位为字节，
		等于图像的宽乘以每像素字节数与输入的扫描线跨距增量之和。
	\since build 471
	*/
	explicit
	HBitmap(BitmapPtr, const Size&, size_t = 0);
	/*!
	\brief 构造：从标准矩形像素图缓冲区复制并转换图像数据。
	\exception LoggedEvent 转换失败。
	\since build 471
	*/
	HBitmap(const CompactPixmap&);
	/*!
	\throw LoggedEvent 读取失败。
	\since build 457
	*/
	//@{
	/*
	\brief 构造：使用指定 UTF-8 文件名和解码器标识。
	\throw UnknownImageFormat 未知图像格式。
	*/
	HBitmap(const char*, ImageDecoderFlags = ImageDecoderFlags::Default);
	//! \brief 构造：使用指定 UTF-8 文件名、指定格式和解码器标识。
	HBitmap(const char*, ImageFormat,
		ImageDecoderFlags = ImageDecoderFlags::Default);
	/*!
	\brief 构造：使用指定 UCS-2 文件名和解码器标识。
	\throw UnknownImageFormat 未知图像格式。
	*/
	HBitmap(const char16_t*, ImageDecoderFlags = ImageDecoderFlags::Default);
	//! \brief 构造：使用指定 UCS-2 文件名、指定格式和解码器标识。
	HBitmap(const char16_t*, ImageFormat,
		ImageDecoderFlags = ImageDecoderFlags::Default);
	/*!
	\brief 构造：使用指定字符串文件名和解码器标识。
	\throw UnknownImageFormat 未知图像格式。
	\since build 483
	*/
	template<class _tString,
		yimpl(typename = ystdex::enable_for_string_class_t<_tString>)>
	HBitmap(const _tString& filename,
		ImageDecoderFlags = ImageDecoderFlags::Default)
		: HBitmap(&filename[0])
	{}
	/*!
	\brief 构造：使用指定字符串文件名和解码器标识。
	\since build 483
	*/
	template<class _tString,
		yimpl(typename = ystdex::enable_for_string_class_t<_tString>)>
	HBitmap(const _tString& filename, ImageFormat fmt,
		ImageDecoderFlags = ImageDecoderFlags::Default)
		: HBitmap(&filename[0], fmt)
	{}
	//@}
	/*!
	\throw LoggedEvent 读取失败。
	\since build 457
	*/
	HBitmap(const ImageMemory&, ImageDecoderFlags = ImageDecoderFlags::Default);
	/*!
	\brief 构造指定图像转换为指定色深的基于 RGB 像素格式的位图副本。
	\throw UnsupportedImageFormat 指定的色深对指定图形不被支持。
	\throw LoggedEvent 转换失败（包括色深被支持但具体格式不被实现支持的情形）。
	\note 对于 16 位位图使用 RGB555 。对于 32 位位图使用 RGBA8888 。
	\since build 471
	*/
	HBitmap(const HBitmap&, BitPerPixel);
	//! \since build 417
	//@{
	/*!
	\brief 构造指定图像缩放至指定大小的副本。
	\throw LoggedEvent 缩放失败。
	\since build 430
	*/
	HBitmap(const HBitmap&, const Size&, SamplingFilter);
	//! \throw BadImageAlloc 分配空间失败。
	HBitmap(const HBitmap&);
	HBitmap(HBitmap&&) ynothrow;
	//! \since build 461
	~HBitmap();
	//@}

	//! \since build 430
	//@{
	//! \brief 统一赋值：使用值参数和交换函数进行复制或转移赋值。
	HBitmap&
	operator=(HBitmap pixmap) ynothrow
	{
		pixmap.swap(*this);
		return *this;
	}

	PDefHOp(bool, !, ) const ynothrow
		ImplRet(!bitmap)

	/*!
	\brief 取扫描线数据。
	\pre 断言： <tt>bitmap</tt> 。
	\pre 断言： 参数值小于高。
	\return 扫描线数据的起始指针。
	\note 扫描线宽为跨距。
	\since build 471
	*/
	byte*
	operator[](size_t) const ynothrow;

	explicit DefCvt(const ynothrow, bool, bitmap)
	//@}

	/*!
	\brief 转换为标准矩形像素图缓冲区。
	\since build 471
	*/
	operator CompactPixmap() const;

	BitPerPixel
	GetBPP() const ynothrow;
	//! \since build 417
	DefGetter(const ynothrow, DataPtr, DataPtr, bitmap)
	SDst
	GetHeight() const ynothrow;
	//! \since build 417
	DefGetter(const ynothrow, Size, Size, {GetWidth(), GetHeight()});
	//! \since build 417
	SDst
	GetPitch() const ynothrow;
	/*!
	\brief 取像素数据。
	\return 若数据指针为空则为空指针，否则为像素数据起始非空指针。
	\note 像素数据由连续的扫面线数据构成，数量等于高度值。
	\since build 471
	*/
	byte*
	GetPixels() const ynothrow;
	/*!
	\brief 取扫描线数据。
	\pre 间接断言：参数值小于高。
	\return 若数据指针为空则为空指针，否则为扫描线数据起始非空指针。
	\note 使用 <tt>operator[]</tt> 实现。
	\sa operator[]
	\since build 471
	*/
	PDefH(byte*, GetScanLine, size_t idx) const ynothrow
		ImplRet(bitmap ? (*this)[idx] : nullptr)
	SDst
	GetWidth() const ynothrow;

	/*!
	\brief 缩放为指定大小。
	\since build 430
	*/
	void
	Rescale(const Size&, SamplingFilter = SamplingFilter::Box);

	/*!
	\return 是否保存成功。
	\since build 471 。
	*/
	//@{
	//!\ brief 保存：使用指定 UTF-8 文件名、格式和解码器标识。
	bool
	SaveTo(const char*, ImageFormat = ImageFormat::BMP,
		ImageDecoderFlags = ImageDecoderFlags::Default) const ynothrow;
	//!\ brief 保存：使用指定 UTF-16 文件名、格式和解码器标识。
	bool
	SaveTo(const char16_t*, ImageFormat = ImageFormat::BMP,
		ImageDecoderFlags = ImageDecoderFlags::Default) const ynothrow;
	/*!
	\brief 保存：使用指定字符串文件名、格式和解码器标识。
	\since build 483
	*/
	template<class _tString,
		yimpl(typename = ystdex::enable_for_string_class_t<_tString>)>
	bool
	SaveTo(const _tString& filename, ImageFormat fmt = ImageFormat::BMP,
		ImageDecoderFlags flags = ImageDecoderFlags::Default) const
	{
		return SaveTo(&filename[0], fmt, flags);
	}
	//@}

	/*
	\brief 交换。
	\since build 430
	*/
	PDefH(void, swap, HBitmap& pixmap) ynothrow
		ImplExpr(std::swap(bitmap, pixmap.bitmap))
};

/*!
\relates HBitmap
\since build 430
*/
inline DefSwap(ynothrow, HBitmap)


//! \since build 456
//@{
/*!
\brief 多页面位图数据。
\note 非公开实现。
*/
class MultiBitmapData;

/*!
\brief 多页面位图句柄：指向多页面位图数据。
\note 共享复制且可转移。
\todo 增加使用 ImageMemory 的构造函数。
*/
class YF_API HMultiBitmap final
{
public:
	using DataPtr = shared_ptr<MultiBitmapData>;
	//! \since build 460
	//@{
	class YF_API iterator : public std::iterator<std::input_iterator_tag,
		HBitmap, ptrdiff_t, const HBitmap*, HBitmap>
	{
	private:
		const HMultiBitmap* p_bitmaps;
		size_t index;

	public:
		iterator()
			: p_bitmaps()
		{}
		iterator(const HMultiBitmap& bmps, size_t idx = 0)
			: p_bitmaps(&bmps), index(idx)
		{}

		iterator&
		operator++() ynothrowv;

		reference
		operator*() const;

		YF_API friend bool
		operator==(const iterator&, const iterator&) ynothrow;

		DefGetter(const ynothrow, const HMultiBitmap*, HMultiBitmapPtr,
			p_bitmaps)
		DefGetter(const ynothrow, size_t, Index, index)
	};
	using const_iterator = iterator;
	//@}

private:
	DataPtr pages;

public:
	/*!
	\throw LoggedEvent 读取失败。
	\note 非多页面读取结果为空。
	\since build 457
	*/
	//@{
	/*!
	\brief 构造：使用指定 UTF-8 文件名和解码器标识。
	\throw UnknownImageFormat 未知图像格式。
	*/
	HMultiBitmap(const char*, ImageDecoderFlags = ImageDecoderFlags::Default);
	//! \brief 构造：使用指定 UTF-8 文件名、指定格式和解码器标识。
	HMultiBitmap(const char*, ImageFormat,
		ImageDecoderFlags = ImageDecoderFlags::Default);
	/*!
	\brief 构造：使用指定 UCS-2 文件名和解码器标识。
	\throw UnknownImageFormat 未知图像格式。
	*/
	HMultiBitmap(const char16_t*,
		ImageDecoderFlags = ImageDecoderFlags::Default);
	//! \brief 构造：使用指定 UCS-2 文件名、指定格式和解码器标识。
	HMultiBitmap(const char16_t*, ImageFormat,
		ImageDecoderFlags = ImageDecoderFlags::Default);
	//! \since build 483
	//@{
	/*!
	\brief 构造：使用指定字符串文件名和解码器标识。
	\throw UnknownImageFormat 未知图像格式。
	*/
	template<class _tString,
		yimpl(typename = ystdex::enable_for_string_class_t<_tString>)>
	HMultiBitmap(const _tString& filename,
		ImageDecoderFlags flags = ImageDecoderFlags::Default)
		: HMultiBitmap(&filename[0], flags)
	{}
	//! \brief 构造：使用指定字符串文件名和解码器标识。
	template<class _tString,
		yimpl(typename = ystdex::enable_for_string_class_t<_tString>)>
	HMultiBitmap(const _tString& filename, ImageFormat fmt,
		ImageDecoderFlags flags = ImageDecoderFlags::Default)
		: HMultiBitmap(&filename[0], fmt, flags)
	{}
	//@}
	//@}
	DefDeCopyCtor(HMultiBitmap)
	DefDeMoveCtor(HMultiBitmap)

	DefDeCopyAssignment(HMultiBitmap)
	DefDeMoveAssignment(HMultiBitmap)

	PDefHOp(bool, !, ) const ynothrow
		ImplRet(!pages)

	explicit DefCvt(const ynothrow, bool, bool(pages))

	size_t
	GetPageCount() const ynothrow;

	HBitmap
	Lock(size_t = 0) const;

	//! \brief 交换。
	PDefH(void, swap, HMultiBitmap& multi_pixmap) ynothrow
		ImplExpr(std::swap(pages, multi_pixmap.pages))

	//! \since build 461
	//@{
	PDefH(iterator, begin, ) const ynothrow
		ImplRet(GetPageCount() != 0 ? HMultiBitmap::iterator(*this)
			: HMultiBitmap::iterator())

	PDefH(iterator, end, ) const ynothrow
		ImplRet(HMultiBitmap::iterator())
	//@}
};

inline HMultiBitmap::iterator&
HMultiBitmap::iterator::operator++() ynothrowv
{
	YAssertNonnull(p_bitmaps);
	if(++index == p_bitmaps->GetPageCount())
		p_bitmaps = {};
	return *this;
}

inline HMultiBitmap::iterator::reference
HMultiBitmap::iterator::operator*() const
{
	YAssertNonnull(p_bitmaps);
	return p_bitmaps->Lock(index);
}

inline bool
operator!=(const HMultiBitmap::iterator& x, const HMultiBitmap::iterator& y)
	ynothrow
{
	return !(x == y);
}

//! \relates HMultiBitmap::iterator
inline bool
is_undereferenceable(const HMultiBitmap::iterator& i) ynothrow
{
	return !i.GetHMultiBitmapPtr();
}

//! \relates HMultiBitmap
inline DefSwap(ynothrow, HMultiBitmap)
//@}


//! \since build 417
class YF_API ImageCodec final
{
public:
	ImageCodec();
	//! \since build 461
	~ImageCodec();

	/*!
	\brief 检测图像格式。
	\note 对于文件，若根据内容检测失败则根据扩展名判断（不保证正确性）。
	\since build 457
	*/
	//@{
	//! \note 使用图像内存的本机句柄和大小。
	static ImageFormat
	DetectFormat(ImageMemory::NativeHandle, size_t);
	//! \note 使用指定 UTF-8 文件名。
	static ImageFormat
	DetectFormat(const char*);
	//! \note 使用指定 UCS-2 文件名。
	static ImageFormat
	DetectFormat(const char16_t*);
	//@}

	//! \since build 470
	static CompactPixmap
	Load(ImageMemory::Buffer);

	/*!
	\brief 读取指定路径的多页面图片文件为用于直接呈现的帧序列。
	\note 非多页面读取结果为空。
	\note 对多页面图片自动选择解码器标识（当前仅支持 GIF 格式）；其它为默认参数。
	\since build 458
	*/
	//@{
	//! \note 使用指定 UTF-8 文件名。
	static HMultiBitmap
	LoadForPlaying(const char*);
	//! \note 使用指定 UCS-2 文件名。
	static HMultiBitmap
	LoadForPlaying(const char16_t*);
	/*!
	\brief 构造：使用指定字符串文件名。
	\since build 483
	*/
	template<class _tString,
		yimpl(typename = ystdex::enable_for_string_class_t<_tString>)>
	static HMultiBitmap
	LoadForPlaying(const _tString& filename)
	{
		return LoadForPlaying(&filename[0]);
	}
	//@}

	/*!
	\brief 读取指定路径的图片文件为用于直接呈现的帧序列。
	\note 使用 LoadForPlaying 按多页面读取，若结果为空按单页面读取。
	\sa LoadForPlaying
	\since build 461
	*/
	template<class _tSeqCon, typename _type>
	static _tSeqCon
	LoadSequence(const _type& path)
	{
		const auto multi_bitmap(LoadForPlaying(path));
		_tSeqCon con{multi_bitmap.begin(), multi_bitmap.end()};

		if(con.empty())
			con.emplace_back(path);
		return con;
	}
};

} // namespace Drawing;

} // namespace YSLib;

#endif

