# -*- coding: utf-8 -*-
# vi:ts=4:et
#
# $Date: 2004/04/25 17:03:24 $
# $Revision: 1.11 $
# =====================================================

"""library for permutation and combination"""

import sys
import operator
import itertools


__all__ = [
           'factorial',
           'permutation',
           'combination',
           'catalan',
           'catalan_generator',
           'k_composition',
           'weak_k_composition',
           ]

def factorial(x, y=0):
    """factorial([m=1), n) -> Return the factorial of n.

    If m is given, return the multiply of sequence from m to n.
    m x (m+1) x ... x n
    """
    if y:
        start, stop = x, y
    else:
        start, stop = 2, x

    # factorial(2,5) -> 2*3*4*5
    #factorial(3,4) -> 3*4

    # 4! -> 4*3*2*1 = 24
    # 3! -> 3*2*1 = 6

    assert isinstance(stop, (int, long)) and stop >= 0, "argument must be a non-negative integer."

    assert isinstance(start, (int, long)) and start >= 0, "argument must be a non-negative integer."


    if stop < 2:
        return 1

    assert start <= stop, \
            "first argument <= second argument is needed." +\
            "%s > %s is given."%(start, stop)

    return _mul_of_sequence(range(start, stop + 1), 1)

def _mul_of_sequence(seq, initial=None):
    """Return the multipy of sequence.
    """
    # [sample]

    # if seq is [2,4,6]
    # _mul_of_sequence(seq) = 2 * 4 * 6 = 48

    if initial:
        return reduce(operator.mul, seq, initial)
    else:
        return reduce(operator.mul, seq)

def combination(first, second):
    """combination(n, m) -> Return the combination of (n,m).

    n can be a sequence or an integer.
    m must be an integer.

    If n is an integer, return the combination of (n, m)

    If n is a sequence,
    out of the sequence, take m elements.
    """
    assert isinstance(second, (int, long)) and second >= 0, "second argument must be a non-negative integer."

    if isinstance(first, (int, long)):
        if second == 0:
            return 1
        else:
            return _combination_of_integer(first, second)

    # Now we know that the first argument is not an integer.
    # the first needs to be a sequence.

    try:
        len(first)
    except TypeError, e:
        raise TypeError, "bad operand type was found in the argument : %s"%first
    else:
        return _combination_of_sequence(first, second)

def _combination_of_sequence(seq, num):
    """out of the sequence, take num elements
    """
    if len(seq) < num:
        num = len(seq)
    #assert len(seq) >= num,  seq

    return list(com_gen(seq, num))

def _combination_of_integer(big, small):
    """combination of two integers.
    """
    # (4,2)  -> 4*3 / 2*1 = 6
    # (n, m) -> n!/(m! * (n-m)!)
    # = n*(n-1) * ... * (n-m+1)/m!
    assert big >= small >=0,\
            "invalid argument (%s, %s)\n"%(big, small) + \
            " "*4 +\
            "first argument must be greater than or equal to second argument."

    return factorial(big - small +1, big)/factorial(small)

def permutation(*seq):
    """permutation(sequence) -> Return the permutation of sequence
    """
    # XXX
    # by using arbitrary argument lists,
    # you can use expressions both
    # permutation(('a','b','c'))    # -> single argument of tuple
    # and
    # permutation('a', 'b', 'c')    # -> three arguments of string

    if len(seq) == 1:
        seq = seq[0]

    # list method, 'pop', is used in perm_gen
    if not isinstance(seq, list):
        try:
            seq = list(seq)
        except TypeError, e:
            raise TypeError, "can't convert to list: %r"%(repr(seq))

    if not seq:
        #  seq = [], range(0), etc.

        # XXX
        # permutation([]) == [] or [[]]?

        # empty set
        return []
    elif len(seq) == 1:
        # seq = [1], ['a'], etc.
        return [seq]
    else:
        return list(perm_gen(seq))


def permhalf(seq):
    # permutation has a symmetric structure.
    # symmetric group <--> permutation

    # XXX
    # helper function for perm_gen
    pop, insert, append = seq.pop, seq.insert, seq.append
    llen = len(seq)
    if llen <= 2:
        yield seq
    else:
        aRange = range(llen)
        v = pop()
        for p in permhalf(seq):
            for j in aRange:
                insert(j, v)
                yield seq
                del seq[j]
        append(v)

def half_perm_gen(seq):
    """generate half of all permutations from a given sequence.
    """
    ph = permhalf(seq)

    for h in ph:
        # XXX
        # don't forget to add [:] at yield statement.
        yield h[:]

def perm_gen(seq):
    """generate all permutations from a given sequence.
    """
    # largely owes the idea from comp.lang.python community.

    ph = permhalf(seq)

    for h in ph:
        p = h[:]

        # XXX
        # don't forget to add [:] at yield statement.
        yield p[:]
        p.reverse()
        yield p[:]

# NOTE
# next function is included in the distribution.
# see the following script.
# /src/Lib/test/test_generators.py
# the original name was 'gcomb'
def com_gen(seq, k):
    """Generate all combinations of k elements from the seq.
    """
    if k > len(seq):
        return
    if k == 0:
        yield []
    else:
        first, rest = seq[0], seq[1:]
        # A combination does or doesn't contain first.
        # If it does, the remainder is a k-1 comb of rest.
        for c in com_gen(rest, k-1):
            c.insert(0, first)
            yield c
        # If it doesn't contain first, it's a k comb of rest.
        for c in com_gen(rest, k):
            yield c

def catalan(num):
    """catalan(n) -> Return the Catalan number
    """
    # just for fun

    # C_n = (1 / (n+1)) * (2n, n)
    #
    #     = (1 / (n+1)) * (2n)! / (n!)^2
    #
    #     = (2n)! / ((n+1)! * n!)
    #
    #     = (2n * (2n-1) * ... * (n+1) / (n+1) !
    #
    #     = (2n * (2n-1) * ... * (n+2) / n !

    assert isinstance(num, (int, long)) and num >= 0, "must be a non-negative integer."
    if num in (0, 1):
        return 1

    return factorial(num+2, 2 * num) / factorial(num)

def catalan_generator():
    """catalan_generator() -> \
Return the Catalan number as a generator.
    """
    counter = itertools.count(0)
    while True:
        yield catalan(counter.next())

def k_composition(n, k):
    """k_composition(n, k) -> Return the number of k-composition of n.
    """
    # Stanley EC1 p.14
    assert k-1 >= 0, "second argument must be non-negative"
    return combination(n-1, k-1)

def weak_k_composition(n, k):
    """weak_k_composition(n, k) -> Return the number of weak k-composition of n.
    """
    # Stanley EC1 p.15

    assert k-1 >= 0, "second argument must be non-negative"
    return combination(n + k -1, k-1)

