# vim fileencoding=utf-8
# vi:ts=4:et
# $Id: test_young.py,v 1.4 2004/07/23 05:28:36 etale Exp $
# =====================================================

"""
unittest for young.py
"""
import sets
import itertools

import unittest
from test import test_support

from young import *

class TestYoung(unittest.TestCase):
    def test_total(self):
        for i in range(1, 6):
            self.assertEqual(young(i).get_square(), factorial(i))

    def test_repr_str(self):
        y = young(3)

        `y`
        print y

        repr(y)
        str(y)

        y.__repr__()
        y.__str__()

        "%r"%y
        "%s"%y

    def test_up(self):
        expectations = (
            [(1,1,1,1), ((2, 1, 1, 1), (1, 1, 1, 1, 1))],
            [(2,1,1), ((3, 1, 1), (2, 2, 1), (2, 1, 1, 1))],
            [(2,2), ((3, 2), (2, 2, 1))],
            [(3,1), ((4, 1), (3, 2), (3, 1, 1))],
            [(4,), ((5,), (4, 1))],
            )

        for part, expectation in expectations:
            y = young(part)
            y.up()
            self.assertEqual(sets.Set(y.get_partition()), sets.Set(expectation))

    def test_down(self):
        expectations = (
            [(1,1,1,1), ((1,1,1),)],
            [(2,1,1), ((1,1,1), (2,1))],
            [(2,2), ((2,1),)],
            [(3,1), ((2,1), (3,))],
            [(4,), ((3,),)],
            )

        for part, expectation in expectations:
            p = Partition(part)
            y = young(part)
            y.down()
            self.assertEqual(sets.Set(y.get_partition()), sets.Set(expectation))


    def test_number_of_up(self):
        expectations = ((2,2), (4,6), (10,24), (26,120), (76,720))

        y = young(2)
        for expectation in expectations:
            self.assertEqual((y.size(), y.get_square()), expectation)
            y.up()

    def test_number_of_down(self):
        expectations = ((76, 720), (26, 120), (10, 24), (4, 6), (2, 2))

        y = young(7)
        for expectation in expectations:
            y.down()
            self.assertEqual((y.size(), y.get_square()), expectation)

    def test_generate(self):
        expectation = (
            [[1], [2], [3], [4]],
            [[1, 2], [3], [4]],
            [[1, 3], [2], [4]],
            [[1, 4], [2], [3]],
            [[1, 2], [3, 4]],
            [[1, 3], [2, 4]],
            [[1, 2, 3], [4]],
            [[1, 2, 4], [3]],
            [[1, 3, 4], [2]],
            [[1, 2, 3, 4]],
            )

        y = young(4)
        self.assertEqual(len(y), len(expectation))
        for tableaux in y:
            self.assert_(tableaux in expectation)

    def test_partition(self):
        for i in range(1, 5):
            set_of_part = sets.Set(partition(i))
            set_of_young = sets.Set(young(i).get_partition())
            self.assertEqual(set_of_part, set_of_young)


class TestBump(unittest.TestCase):
    def test_bump1(self):
        # Fulton p.12
        insert = Young([[1,3], [2]])

        y = Young([[1,2,2,3], [2,3,5,5],[4,4,6],[5,6]])

        expectations = [
            [[1,2,2,2], [2,3,3,5],[4,4,5],[5,6,6]],
            [[1,1,2,2], [2,2,3,5],[3,4,5],[4,6,6],[5]],
            [[1,1,2,2,3], [2,2,3,5],[3,4,5],[4,6,6],[5]],
        ]

        for box, expectation in itertools.izip((2,1,3), expectations):
            y.bump(box)
            self.assertEqual(y, expectation)

    def test_bump2(self):
        # Fulton p.37
        y = Young() 
        word = [5,4,8,2,3,4,1,7,5,3,1]
        for w in word:
            y.bump(w)

        self.assertEqual(y, Young([[1,1,3,5],[2,3],[4,4],[5,7],[8]]))

    def test_bump3(self):
        # Fulton p.40
        y = Young()
        word = (1,2,2,1,2,1,1,1,2)
        for w in word:
            y.bump(w)

        self.assertEqual(y, Young([[1,1,1,1,1,2],[2,2,2]]))

    def test_bump_multi(self):
        # Fulton p.12
        young_to_insert = Young([[1,3], [2]])
        young_to_be_inserted = Young([[1,2,2,3], [2,3,5,5],[4,4,6],[5,6]])

        # young_to_be_inserted <- young_to_insert
        young_to_be_inserted*young_to_insert

        self.assertEqual(young_to_be_inserted, Young([[1,1,2,2,3], [2,2,3,5],[3,4,5],[4,6,6],[5]]))

    def test_bump_hotta1(self):
        # Hotta p.120
        y = Young()
        word = (3,2,6,1,4,5)
        for w in word:
            y.bump(w)

        self.assertEqual(y, Young([[1,4,5],[2,6],[3]]))

    def test_bump_hotta2(self):
        # Hotta p.120
        y = Young()
        word = (5,3,4,7,1,2,6)
        for w in word:
            y.bump(w)

        self.assertEqual(y, Young([[1,2,6],[3,4,7],[5]]))

class TestSkew(unittest.TestCase):
    # test of skew tableaux

    def test_repr_str(self):
        t = Young([[1,1,2],[2],[3]])
        y = Young([[1,1],[2]])
        skew = t / y

        `skew`
        print skew

        repr(skew)
        str(skew)

        skew.__repr__()
        skew.__str__()

        "%r"%skew
        "%s"%skew

    def test_skew1(self): 
        t = Young([[1,1,2],[2],[3]])
        y = Young([[1,1],[2]])
        skew = t / y
        self.assertEqual(skew,
                [[None, None, 2],[None], [3]])

    def test_skew2(self): 
        t = Young([[1,2,2,3],[2,3,5,5],[4,4,6],[5,6]])
        y = Young([[1,3],[2]])
        skew = t / y
        self.assertEqual(skew,
                [[None, None, 2, 3],[None, 3,5,5], [4,4,6],[5,6]])

    def test_jeudetaquin1(self):
        # Fulton p.15
        # Excercise 1

        # rectification of skew tableau
        y1 = Young([[1,1,1,1,1,3],[1,1,1,1,2],[1,2,2,3],[2,3,5,5],[4,4,6],[5,6]])
        y2 = Young([[1,1,1,1],[1,1,1,1]])
        skew = y1 / y2

        # product of y3 and y4
        y3 = Young([[1,2,2,3],[2,3,5,5],[4,4,6],[5,6]])
        y4 = Young([[1,3],[2]])
        y3 * y4

        self.assertEqual(skew.rectify(), y3)

    def test_jeudetaquin2(self):

        y3 = Young([[1,2,2,3],[2,3,5,5],[4,4,6],[5,6]])
        y4 = Young([[1,3],[2]])

        skew = y3 / y4
        self.assertEqual(skew.rectify(), Young([[2,3,5,5],[3,4,6],[4,6],[5]]))

    def test_method_equality1(self):
        y1 = Young([[1,1,2],[2],[3]])
        y2 = Young([[1,1],[2]])

        s1 = y1 / y2
        s2 = y1 / y2
        s3 = y1 / y2
        self.assert_(s1.rectify()==s2.taquin()==s3.jeu_de_taquin())

    def test_method_equality2(self):
        y1 = Young([[1,2,2,3],[2,3,5,5],[4,4,6],[5,6]])
        y2 = Young([[1,3],[2]])

        s1 = y1 / y2
        s2 = y1 / y2
        s3 = y1 / y2
        self.assert_(s1.rectify()==s2.taquin()==s3.jeu_de_taquin())

    def test_jeudetaquin3(self):
        # Fulton p.13
        y1 = Young([[1,1,1,1,2],[1,1,1,1,6],[1,2,4,4],[2,3,6],[5,5]])
        y2 = Young([[1,1,1,1],[1,1,1,1], [1]])
        skew = y1 / y2
        self.assertEqual(skew.rectify(), Young([[2,2,2,4,6],[3,4,6],[5,5]]))
    def test_jeudetaquin4(self):
        # This example is taken from Fulton p.20

        # For a skew tableau
        # * * x
        # y z
        #
        # if x < y <= z, then
        #
        # x z
        # y
        y1 = Young([[1,1,1],[2,3]])

        y2 = Young([[1,1]])
        skew = y1 / y2
        self.assertEqual(skew.rectify(), Young([[1,3],[2]]))

    def test_jeudetaquin5(self):
        # This example is taken from Fulton p.20

        # For a skew tableau
        # * * y
        # x z
        #
        # if x <= y < z, then
        #
        # x y
        # z
        #
        # In this case, x == y
        y1 = Young([[1,1,1],[1,3]])
        y2 = Young([[1,1]])
        skew = y1 / y2
        self.assertEqual(skew.rectify(), Young([[1,1],[3]]))

    def test_jeudetaquin6(self):
        # This example is taken from Fulton p.20

        # For a skew tableau
        # * * y
        # x z
        #
        # if x <= y < z, then
        #
        # x y
        # z
        #
        # In this case, x < y
        y1 = Young([[1,1,2],[1,3]])
        y2 = Young([[1,1]])
        skew = y1 / y2
        self.assertEqual(skew.rectify(), Young([[1,2],[3]]))

class TestWord(unittest.TestCase):
    def test_repr_str(self):
        w = Word((2,4,1,3,5))

        `w`
        print w

        repr(w)
        str(w)

        w.__repr__()
        w.__str__()

        "%r"%w
        "%s"%w

    def test_product1(self):
        # Fulton p.12
        insert = Word((2,1,2))

        word = Word((5,6,4,4,6,2,3,5,5,1,2,2,3))

        expectations = [
            (5, 6, 6, 4, 4, 5, 2, 3, 3, 5, 1, 2, 2, 2),
            (5, 4, 6, 6, 3, 4, 5, 2, 2, 3, 5, 1, 1, 2, 2),
            (5, 4, 6, 6, 3, 4, 5, 2, 2, 3, 5, 1, 1, 2, 2, 2),
        ]

        for w, expectation in itertools.izip(insert, expectations):
            word * w
            self.assertEqual(word, Word(expectation))

    def test_product2(self):
        # Fulton p.37

        word = Word((5,6,4,4,6,2,3,5,5,1,2,2,3))
        another_word = Word((2,1,2))

        word * another_word
        self.assertEqual(word, Word((5, 4, 6, 6, 3, 4, 5, 2, 2, 3, 5, 1, 1, 2, 2, 2)))

    def test_product3(self):
        # Fulton p.40

        expectations = [
            (1,) ,
            (1, 2) ,
            (1, 2, 2) ,
            (2, 1, 1, 2) ,
            (2, 1, 1, 2, 2) ,
            (2, 2, 1, 1, 1, 2) ,
            (2, 2, 2, 1, 1, 1, 1) ,
            (2, 2, 2, 1, 1, 1, 1, 1) ,
            (2, 2, 2, 1, 1, 1, 1, 1, 2) ,
            ]
        word = Word()

        for w, expectation in itertools.izip((1,2,2,1,2,1,1,1,2), expectations):
            word * w
            self.assertEqual(word, Word(expectation))

    def test_bump_multi(self):
        # Fulton p.12
        young_to_insert = Young([[1,3], [2]])
        young_to_be_inserted = Young([[1,2,2,3], [2,3,5,5],[4,4,6],[5,6]])

        word = Word((5, 6, 4, 4, 6, 2, 3, 5, 5, 1, 2, 2, 3))
        another_word = Word((2, 1, 3))

        word * another_word
        self.assertEqual(word, Word((5, 4, 6, 6, 3, 4, 5, 2, 2, 3, 5, 1, 1, 2, 2, 3)))

    def test_product_hotta1(self):
        # Hotta p.120

        word = Word()
        for w in (3,2,6,1,4,5):
            word * w

        self.assertEqual(word, Word((3, 2, 6, 1, 4, 5)))

    def test_product_hotta2(self):
        # Hotta p.120

        word = Word()
        for w in (5,3,4,7,1,2,6):
            word * w

        self.assertEqual(word, Word((5, 3, 4, 7, 1, 2, 6)))

    def test_L(self):
        # Fulton p. 31
        w = Word((3,4,4,2,3,3,1,1,2,2,2,3))
        self.assertEqual(w.L(1), 6)
        self.assertEqual(w.L(2), 9)
        self.assertEqual(w.L(3), 12)

class TestYoungWord(unittest.TestCase):
    def test_y2w(self):
        for y in young(5):
            w = y.toword()
            self.assertEqual(w.toyoung(), y)

def test_main(verbose=None):
    testclasses = (
            TestYoung,
            TestBump,
            TestSkew,
            TestYoungWord,
            TestWord,
            )

    test_support.run_unittest(*testclasses)

if __name__ == '__main__':
    test_main()
