# vim fileencoding=utf-8
# vi:ts=4:et
# $Id: hook.py,v 1.2 2004/05/15 08:59:01 etale Exp $
# =====================================================
"""library for Hook"""

import operator
import itertools

# young
import partition
import combination
import mathformat
import util

__all__ = [
    "Hook",
]

class Hook(list):
    """This class represents a hook.
    """
    # Fulton p. 53

    # depending on whether the tableaux is standard or not,
    # the hook length formula is different

    def __init__(self, shape):
        # XXX
        # Here, we need to guarantee that shape is an instance of Partition class.
        if not isinstance(shape, partition.Partition):
            shape = partition.Partition(shape)
        self.shape = shape

        list.__init__(self, self.label_hook_length())

    def label_hook_length(self):
        """labelling hook lengths"""
        # This example is taken from Fulton P.53)
        #
        # If the shape is (6,5,5,3), then the hook length is
        # ==>
        # 9 8 7 5 4 1
        # 7 6 5 3 2
        # 6 5 4 2 1
        # 3 2 1

        partition1 = self.shape
        partition2 = self.shape.conjugate()

        for i in xrange(len(partition1)):
            yield [partition1[i] + partition2[j] -i -j -1 for j in xrange(partition1[i]) ]

    def __repr__(self):
        return mathformat.pprint_tableaux(self)

    __str__ = __repr__

    def get_hook_tableau(self):
        return list(self)

    def get_young(self):
        """return young tableaux with the same content as this hook."""
        return young.young(self.shape)

    def get_partition(self):
        return self.shape

    def hook_length_formula(self):
        """Return the number of standard tableaux with a given shape.
        """
        # Fulton P.53
        # Hook Length Formula

        return combination.factorial(sum(self.shape)) / reduce(operator.mul, util.flatten_tableau(self))

    def hook_length_formula_ex9(self):
        """Return the number of standard tableaux with a given shape.
        Based on Exercise #9.
        """
        # Fulton p.54
        # Exercise 9
        
        # yet another hook length formula for standard tableaux.
        
        # How many rows are there?
        k = len(self.shape)

        # l_series = (l_1, l_2, ..., l_k)
        l_series = [x + k -i -1 for i, x in enumerate(self.shape)]

        return combination.factorial(sum(self.shape)) * \
               reduce(operator.mul, [l_series[i]-l_series[j] for i in xrange(k) for j in xrange(i+1, k)]) / \
               reduce(operator.mul, itertools.imap(combination.factorial, l_series))

    # The argument m is used to calculate the hook length formula for nonstandard tableaux.
    # This variable must be a non-negative integer.
    def hook_length_formula_nonstd(self, m):
        """Return the number of nonstandard tableaux with a given shape whose entries are taken from [m].
        """
        # Fulton P.55 (9)

        # hook length formula for nonstandard tableaux by Stanley.
        # middle formula.

        return reduce(operator.mul, [m+j-i for i in xrange(len(self)) for j in xrange(len(self[i]))] ) / \
               reduce(operator.mul, util.flatten_tableau(self))

    def hook_length_formula_nonstd2(self, m):
        """Return the number of nonstandard tableaux with a given shape whose entries are taken from [m].
        """
        # Fulton P.55(9)

        # yet another hook length formula for nonstandard tableaux by Stanley.
        # rightmost formula.

        return self.hook_length_formula() * \
               reduce(operator.mul, [m+j-i for i in xrange(len(self)) for j in xrange(len(self[i]))] ) / \
               combination.factorial(sum(self.shape))

    # alias
    # calculate the hook length formula with this method name

    def number(self, m=None):
        if m is not None:
            assert m >= 0, "m must be a non-negative integer"
            return self.hook_length_formula_nonstd(m)

        return self.hook_length_formula()

