/* $Id: davcp.c,v 1.11 2003/01/09 12:43:53 rtakano Exp $
 * davcp: WebDAV copy
 * Copyright (c) 2002 TAKANO Ryousei <rtakano@sourceforge.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <sys/stat.h>
#include <sys/param.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <dirent.h>
#include <getopt.h>
#include <assert.h>
#include <errno.h>

#include <ne_session.h>
#include <ne_uri.h>
#include <ne_basic.h>
#include <ne_auth.h>
#include <ne_utils.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "davcp.h"


enum {
  METHOD_GET = 0,
  METHOD_PUT = 1,
};

static int recursive;
static int verbose = 1;
static char logname[256];

static void usage(void)
{
  printf("\
Usage: davcp [OPTION] SOURCE DEST\n\
OPTION:\n\
\t-l\tlogin name\n\
\t-r\tcopy directories recursively\n\
\t-v\tdisplay what is being done\n\
\t-h\tprint this usage message\n\
");
}

/* callback function of authentication. */
static int supply_creds_server(void* userdata, const char* realm, int attempt,
			       char* username, char* password)
{
  char* user;
  char* passwd;
  if (attempt > 1) {
    return -1;
  }

  user = (logname[0] != '\0') ? logname : getenv("LOGNAME");
  strcpy(username, user);
  passwd = getpass("Password: "); /* TODO: getpass is a obsolete function. */
  strcpy(password, passwd);
  return 0;
}

static int open_connection(ne_uri* uri, ne_session** session)
{
  char* proxy_host;
  int proxy_port = 0;

  /* Create session. */
  *session = ne_session_create(uri->scheme, uri->host, uri->port);

  /* Get the proxy details */
  proxy_host = getenv("http_proxy");
  if (proxy_host != NULL) {
    char* tmp;
    if (strncmp(proxy_host, "http://", 7) == 0) {
      proxy_host = &proxy_host[7];
    }
    for (tmp = proxy_host; *tmp && *tmp != ':'; tmp++);
    if (*tmp) {
      *tmp = '\0';
      proxy_port = atoi(tmp + 1);
    }
  }

  if (proxy_port == 0) {
    proxy_port = 8080;
  }

  if (proxy_host) {
    ne_session_proxy(*session, proxy_host, proxy_port);
  }

  ne_set_useragent(*session, PACKAGE "/" VERSION);
  ne_set_server_auth(*session, supply_creds_server, NULL);

  return 0;
}

static error_exit(ne_session* session)
{
  fprintf(stderr, "davcp: Failed: %s\n", ne_get_error(session));
  exit(1);
}

static get_file(ne_session* session, const char* local, const char* remote)
{
  int fd;
  int ret;
  struct resource* reslist = NULL;
  char* col;

  if (verbose) {
    printf("get  : %s -> %s\n", remote, local);
  }

  /* Is 'remote' a collection? */
  col = ne_concat(remote, "/", NULL);
  ret = fetch_resource_list(session, col, NE_DEPTH_ZERO, 1, &reslist);
  if (ret == NE_OK) {
    struct resource* current = reslist;
    if (current->type == resr_collection) {
      free(col);
      if (recursive) {
	return get_collection(session, local, remote);
      } else {
	fprintf(stderr, "davcp: not recursive copy mode.\n");
	return -1;
      }
    }
    free(col);
  }

  /* create 'local' file. */
  fd = open(local, O_WRONLY|O_CREAT|O_TRUNC, 0644);
  if (fd == -1) {
    perror("open");
    return -1;
  }

  ret = ne_get(session, remote, fd);
  if (ret != NE_OK) {
    error_exit(session);
  }
  close(fd);
  return 0;
}

int get_collection(ne_session* session, const char* name, const char* base)
{
  struct resource* reslist = NULL;
  struct resource* current;
  struct resource* next;
  char* col;
  int ret;

  ret = mkdir(name, 0755);
  if (ret != 0 && errno != EEXIST) {
    perror("mkdir");
    return -1;
  }

  col = ne_concat(base, "/", NULL);
  ret = fetch_resource_list(session, col, NE_DEPTH_ONE, 0, &reslist);
  if (ret != NE_OK) {
    error_exit(session);
  }
  for (current = reslist; current != NULL; current = next) {
    next = current->next;
    switch (current->type) {
    case resr_normal:
      {
	char* local;
	char* p;

	p = strrchr(current->uri, '/');
	local = ne_concat(name, p, NULL);
	ret = get_file(session, local, current->uri);
	free(local);
	if (ret == -1) goto out;
	break;
      }
    case resr_collection:
      {
	char* local;
	char* p;

	if (ne_path_has_trailing_slash(current->uri)) {
	  current->uri[strlen(current->uri) - 1] = '\0';
	}
	p = strrchr(current->uri, '/');
	local = ne_concat(name, p, NULL);
	ret = get_collection(session, local, current->uri);
	free(local);
	if (ret == -1) goto out;
	break;
      }
    default:
      break;
    }
  }

 out:
  free(col);
  return ret;
}

static put_file(ne_session* session, const char* local, const char* remote)
{
  int fd;
  int ret;

  if (verbose) {
    printf("put  : %s -> %s\n", local, remote);
  }
  fd = open(local, O_RDONLY);
  if (fd == -1) {
    perror("open");
    return -1;
  }

  ret = ne_put(session, remote, fd);
  if (ret != NE_OK) {
    error_exit(session);
  }
  close(fd);
  return 0;
}

int put_collection(ne_session* session, const char* name, const char* base)
{
  DIR* dirp;
  struct dirent* dp;
  char path[MAXPATHLEN];
  char* col;
  time_t time;
  int ret;

  dirp = opendir(name);
  if (dirp == NULL) {
    perror("opendir");
    return 1;
  }

  /* TODO: a very ad-hoc mkcol code. */
  col = ne_concat(base, "/", name, NULL);

  ret = ne_getmodtime(session, col, &time);
  if (ret == NE_ERROR) {
    if (strncmp(ne_get_error(session), "404", 3) == 0) {
      if (verbose) {
	printf("mkcol: %s\n", col);
      }
      ret = ne_mkcol(session, col);
      if (ret != NE_OK) {
	error_exit(session);
      }
    }
  }
  free(col);

  while ((dp = readdir(dirp)) != NULL) {
    if (dp->d_ino == 0) {
      continue;
    }
    if (strcmp(dp->d_name, ".") == 0 || strcmp(dp->d_name, "..") == 0) {
      continue;
    }

    snprintf(path, sizeof(path), "%s/%s", name, dp->d_name);
    if (dp->d_type == DT_DIR) {
      ret = put_collection(session, path, base);
      if (ret == -1) goto out;
    } else {
      char* remote;
      remote = ne_concat(base, "/", path, NULL);
      ret = put_file(session, path, remote);
      free(remote);
      if (ret == -1) goto out;
    }
  }

 out:
  closedir(dirp);
  return ret;
}

int dispatch_method(ne_session* session, const char* local, const char* remote,
		    int direction)
{
  struct stat statbuf;
  int ret;

  ret = stat(local, &statbuf);
  if (direction == METHOD_GET) {
    if (ret == -1) {
      return get_file(session, local, remote);
    }

    switch (statbuf.st_mode & S_IFMT) {
    case S_IFREG:
      ret = get_file(session, local, remote);
      break;
    case S_IFDIR:
      {
	char* p;
	char* fname;

	p = strrchr(remote, '/');
	if (p != NULL) {
	  fname = ne_concat(local, p, NULL);
	  ret = get_file(session, fname, remote);
	  free(fname);
	}
	break;
      }
    default:
      break;
    }
  } else if (direction == METHOD_PUT) {
    if (ret == -1) {
      perror("stat");
      return ret;
    }

    switch (statbuf.st_mode & S_IFMT) {
    case S_IFREG:
      {
	char* p;
	char* fname;

	p = strrchr(local, '/');
	if (p != NULL) {
	  fname = ne_concat(remote, p, NULL);
	} else {
	  fname = ne_concat(remote, "/", local, NULL);
	}
	ret = put_file(session, local, fname);
	free(fname);
	break;
      }
    case S_IFDIR:
      {
	if (recursive) {
	  ret = put_collection(session, local, remote);
	} else {
	  fprintf(stderr, "davcp: not recursive copy mode.\n");
	  ret = -1;
	}
	break;
      }
    default:
      ret = -1;
      break;
    }
  }
  return ret;
}

int main(int argc, char** argv)
{
  ne_session* session;
  ne_uri src_uri = {0};
  ne_uri dst_uri = {0};
  ne_uri *uri;
  char* local_path;
  int direction;
  int ret;
  int ch;

  while ((ch = getopt(argc, argv, "l:nrvh")) != EOF) {
    switch(ch) {
    case 'l':
      strncpy(logname, optarg, sizeof(logname));
      break;
    case 'r':
      recursive = 1;
      break;
    case 'v':
      verbose = 1;
      break;
    case 'h':
      usage();
      return 1;
    }
  }

  argc -= optind;
  argv += optind;
  if (argc < 2) {
    usage();
    return 1;
  }

  ne_uri_parse(argv[0], &src_uri);
  ne_uri_parse(argv[1], &dst_uri);
  if (src_uri.scheme != NULL && dst_uri.scheme == NULL) {
    direction = METHOD_GET;
    uri = &src_uri;
    local_path = argv[1];
  } else if (src_uri.scheme == NULL && dst_uri.scheme != NULL) {
    direction = METHOD_PUT;
    local_path = argv[0];
    uri = &dst_uri;
  } else {
    fprintf(stderr, "invalid copy direction: %s -> %s\n", argv[0], argv[1]);
    ne_uri_free(&src_uri);
    ne_uri_free(&dst_uri);
    return 1;
  }

  /* Set defaults. */
  if (uri->port == 0) {
    uri->port = ne_uri_defaultport(uri->scheme);
  }
  if (uri->path == NULL) {
    uri->path = "/";
  }

  /* Initialize socket libraries */
  if (ne_sock_init()) {
    fprintf(stderr, "davcp: Failed to initialize socket libraries.\n");
    ne_uri_free(&src_uri);
    ne_uri_free(&dst_uri);
    return 1;
  }

  /* Initialize connection */
  open_connection(uri, &session);

  /* Dispatch a request */
  if (ne_path_has_trailing_slash(local_path)) {
    local_path[strlen(local_path) - 1] = '\0';
  }
  if (ne_path_has_trailing_slash(uri->path)) {
    uri->path[strlen(uri->path) - 1] = '\0';
  }
  ret = dispatch_method(session, local_path, uri->path, direction);

  ne_session_destroy(session);
  ne_uri_free(&src_uri);
  ne_uri_free(&dst_uri);

  return ret;
}
