// $Id$

//=============================================================================
/**
 *  @file    fileio.cpp
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2006 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "bee/fileio.hpp"
#include "bee/trace.hpp"
namespace fs = boost::filesystem;


namespace bee
{

class File
{
	friend class fileio;
public:
    File() {}

    ~File()
    {
        if (fio_.is_open())
            fio_.close();
    }
	fs::fstream fio_;
};

//-----------------------------------------------------------------------------
// Constructor/Destructor
//-----------------------------------------------------------------------------
fileio::fileio() : path_("")
{
}

fileio::fileio(const BCHAR * filename) : path_(filename)
{
}

fileio::fileio(const std::string& filename) : path_(filename)
{
}

fileio::fileio(const std::string& dir, const std::string& filename,
           const std::string& ext) : path_(dir)
{
	if (dir.at(dir.size() - 1) != DIR_SEPARATOR_CHAR)
	{
		path_ /= DIR_SEPARATOR_STRING;
	}
	path_ /= filename;
	if (filename.at(filename.size() - 1) != '.')
	{
		path_ /= _TX(".");
	}
    path_ /= ext;
}

fileio::~fileio()
{
}

//-----------------------------------------------------------------------------
// Open the file
//-----------------------------------------------------------------------------
int fileio::open(const std::string& filename)
{
    TRACE_FUNCTION("fileio::open");

    path_ = filename;
    if (file_->fio_.is_open())
    {
        file_->fio_.close();
    }
    file_->fio_.open(path_);
    if (file_->fio_.fail())
    {
        TRACE_ERROR_FMT("Error connect text file (%1%)\n", path_.string());
        return -1;
    }

    return 0;
}

int fileio::close()
{
    if (file_->fio_.is_open())
        file_->fio_.close();
	return 0;
}

void fileio::remove()
{
	std::remove(path_.string().c_str());
}


//-----------------------------------------------------------------------------
// Write data
//-----------------------------------------------------------------------------
int fileio::write(const std::string& textstr)
{
    TRACE_FUNCTION("fileio::write");

    file_->fio_.write(textstr.c_str(), textstr.size());
    if (file_->fio_.bad())
    {
        TRACE_ERROR("%s\n", strerror(errno));
        return -1;
    }
    return 0;
}

//-----------------------------------------------------------------------------
int fileio::write(const BYTE * buf, size_t size)
{
    TRACE_FUNCTION("fileio::write");
    file_->fio_.write((char *)buf, size);
    if (file_->fio_.bad())
    {
        TRACE_ERROR("%s\n", strerror(errno));
        return -1;
    }
    return 0;
}

//-----------------------------------------------------------------------------
// Read
//-----------------------------------------------------------------------------
int fileio::read(BYTE * buf, size_t size)
{
    TRACE_FUNCTION("fileio::read");
    std::streamsize len = file_->fio_.readsome((char *)buf, size);
    if (file_->fio_.fail())
    {
        TRACE_ERROR("%s\n", strerror(errno));
        return -1;
    }
    return len;
}

//-----------------------------------------------------------------------------
// Get size
//-----------------------------------------------------------------------------
size_t fileio::size()
{
    TRACE_FUNCTION("fileio::size");
    file_->fio_.seekp(0, std::ios_base::end);
    if (file_->fio_.fail())
    {
        TRACE_ERROR("%s\n", strerror(errno));
        return 0;
    }
    std::fstream::pos_type len = file_->fio_.tellp();
    return (size_t)len;
}

}  // *namespace* - bee

