// $Id$

//=============================================================================
/**
 *  @file    datetime.cpp
 *
 *  @author  Fukasawa Mitsuo
 *
 *    Copyright (C) 2001-2006 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "bee/datetime.hpp"
using namespace std;
using namespace bee;
using namespace boost::gregorian;
using namespace boost::posix_time;

#include "time.h"

namespace bee
{

//-----------------------------------------------------------------------------
// Manipulate string
//-----------------------------------------------------------------------------
void datetime_base::toString(std::string& rs)
{
    rs = to_simple_string(tv_);
}

std::string datetime_base::toString()
{
    return to_simple_string(tv_);
}

std::string datetime_base::toName()
{
    //time_duration td = microseconds(500);
    //tv_ += td;

    std::string tmp = to_iso_string(tv_);
    tmp.erase(8, 1); // delete 'T'
    //tmp.erase(18); // delete micro second
    return tmp;
}

//-----------------------------------------------------------------------------
// Convert string of Name(YYYYMMDDhhmmsscc) format to boost::pisix_time
//-----------------------------------------------------------------------------
void datetime_base::set(const std::string& rhs)
{
    if (rhs.size() == 16)
    { // SEMI TIMEFORMAT
        std::string timestr = rhs.substr(0, 8) + _TX("T");
        timestr += rhs.substr(8, 6);
        //timestr += ".";
        //timestr += rhs.substr(14);  // set milliseconds
        //timestr += "0000";
        tv_ = from_iso_string(timestr);
    }
    else if (rhs.size() == 12)
    {   // SEMI TIMEFORMAT
        // ptime t(second_clock::local_time());
        // std::string yearstr = t.to_iso_string();
        // std::string timestr = yearstr.substr(0, 2);
        std::string timestr = _TX("20");
        timestr += rhs.substr(0, 6);
        timestr += _TX("T");
        timestr += rhs.substr(6);
        tv_ = from_iso_string(timestr);
    }
    else
    {
        tv_ = time_from_string(rhs);
    }
}

//-----------------------------------------------------------------------------
// Create boost::posix_time (static)
//-----------------------------------------------------------------------------
ptime datetime_base::create_time(struct tm * current)
{
    date d(current->tm_year + 1900, current->tm_mon + 1,
           current->tm_mday);
    time_duration td(current->tm_hour, current->tm_min,
                     current->tm_sec);
    return ptime(d, td);
}

//-----------------------------------------------------------------------------
// Convert boost::posix_time to time struct (static)
//-----------------------------------------------------------------------------
void datetime_base::to_localtime(ptime pt, struct tm& retm, long& millisec)
{
    date d(pt.date());
    date::ymd_type ymd = d.year_month_day();
    time_duration  td = pt.time_of_day();
    retm.tm_year = ymd.year - 1900;
    retm.tm_mon = ymd.month - 1;
    retm.tm_mday = ymd.day;
    retm.tm_hour = td.hours();
    retm.tm_min = td.minutes();
    retm.tm_sec = td.seconds();
    millisec = td.fractional_seconds();
}


//-----------------------------------------------------------------------------
// Calculate lapse seconds in the day. (static)
//-----------------------------------------------------------------------------
unsigned int datetime_base::lapse(time_t curtime)
{
    struct tm * newtime;
    struct tm begintime;
    time_t basetime;

    newtime = localtime(&curtime);            // Convert to local time.

    begintime.tm_mday  = newtime->tm_mday;    // Convert to begin of the day.
    begintime.tm_mon   = newtime->tm_mon;
    begintime.tm_year  = newtime->tm_year;
    begintime.tm_wday  = newtime->tm_wday;
    begintime.tm_yday  = newtime->tm_yday;
    begintime.tm_isdst = newtime->tm_isdst;
    begintime.tm_hour  = 0;
    begintime.tm_sec   = 0;
    begintime.tm_min   = 0;

    basetime = mktime(&begintime);

    double diffsec = difftime(curtime, basetime);
    unsigned int lapse = (unsigned int)diffsec;

    return lapse;
}

//-----------------------------------------------------------------------------
// Make log file name by date. (static)
//-----------------------------------------------------------------------------
int datetime_base::makeName(std::string& tvname)
{
	struct timeval curtime;
	::gettimeofday(&curtime, NULL);
    return datetime_base::makeName(curtime.tv_sec, tvname);	
}

//-----------------------------------------------------------------------------
int datetime_base::makeName(time_t curtime, std::string& tvname)
{
    struct tm * newtime;
    BCHAR  buf[16];

    newtime = localtime(&curtime);            // Convert to local time.
#if 0
    // Use logging per date
    sprintf(buf, _TX("%04d%02d%02d"), newtime->tm_year + 1900,
                                      newtime->tm_mon + 1,
                                      newtime->tm_mday);
#else
    sprintf(buf, _TX("%d%02d%02d%02d%02d%02d"), newtime->tm_year + 1900,
                                                newtime->tm_mon + 1,
                                                newtime->tm_mday,
                                                newtime->tm_hour,
                                                newtime->tm_min,
                                                newtime->tm_sec);
#endif
    tvname = buf;
    return 0;
}

}  // *namespace* - bee
