// $Id$

//=============================================================================
/**
 *  @file  nameable.hpp
 *
 *  @author Fukasawa Mitsuo
 *
 *    Copyright (C) 2003-2005 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#ifndef BEE_NAMEABLE_H
#define BEE_NAMEABLE_H

#include "bee/beeconf.hpp"
#include "bee/trace.hpp"


namespace bee
{

typedef std::map<const std::string, const std::string> name_map;

struct string_less
{
	bool operator()(const std::string& lhs, const std::string& rhs) const
	{
		return (lhs < rhs);
	}
};

//
// Interface class
//
class BEE_Export nameable
{
public:
    typedef std::map<std::string, nameable *> dictionary;

    nameable(const std::string& name) : uniqueName_(name) {}
    nameable(const nameable& rhs) : uniqueName_(rhs.uniqueName_) {}
    ~nameable() {}

    nameable& operator=(const nameable& rhs)
    {
        if (this != &rhs)
        {
            uniqueName_ = rhs.uniqueName_;
        }
        return *this;
    }

    void swap(nameable& rhs)
    {
        std::swap(uniqueName_, rhs.uniqueName_);
     }

    void  name_s(const std::string& name) { uniqueName_ = name; }
    const std::string& name_s() const { return uniqueName_; }
    const std::string& key()const     { return uniqueName_; }
    const BCHAR * name_c() const      { return uniqueName_.c_str(); }

    template<class T>
    static T * downcast(const nameable * nmptr)
    {
        T * result = NULL;
        try
        {
            result = static_cast<T *>(const_cast<nameable *>(nmptr));
        }
        catch (const std::bad_cast& ex)
        {
            ERROR_LOG((_TX("Failure cast(%s): %s\n"),
                const_cast<nameable *>(nmptr)->name_c(), ex.what()));
            throw ex;
        }
        return result;
    }

    friend bool operator==(const nameable& lhs, const nameable& rhs);
    friend bool operator<(const nameable& lhs, const nameable& rhs);

protected:
    nameable() : uniqueName_(_TX("")) {}
    void  uniqueName(const std::string& nm) { uniqueName_ = nm; }

protected:
    std::string uniqueName_;
};

//
// Operational functions
//
inline bool operator<(const nameable& lhs, const nameable& rhs)
{
    return (lhs.uniqueName_ < rhs.uniqueName_);
}
inline bool operator==(const nameable& lhs, const nameable& rhs)
{
    return (lhs.uniqueName_ == rhs.uniqueName_);
}

//
// Nameable map
//
class nameable_map
{
public:
    template<class T>
    T * find(const std::string& name) const
    {
        nameable::dictionary::const_iterator iter = map_.find(name);
        if (iter == map_.end())
        {
            return NULL;
        }

        T * result = nameable::downcast<T>(iter->second);
        return result;
    }
    
    //template<>
    nameable * find(const std::string& name) const
    {
        nameable::dictionary::const_iterator iter = map_.find(name);
        if (iter == map_.end())
        {
            return NULL;
        }
        return iter->second;
    }

    int add(nameable * nameptr)
    {
        std::pair<nameable::dictionary::iterator, bool> ent_state;
        ent_state = map_.insert(std::make_pair(nameptr->name_s(), nameptr));
        if (ent_state.second == false)
        {   // member exist already
            return -1;
        }
        return 0;
    }

    template<class T>
    T * remove(const std::string& name)
    {
        nameable::dictionary::iterator iter = map_.find(name);
        if (iter == map_.end())
        {
            return NULL;
        }
        nameable * base = iter->second;
        map_.erase(iter);
        T * result = nameable::downcast<T>(base);
        return result;
    }

    nameable * remove(const std::string& name)
    {
        nameable::dictionary::iterator iter = map_.find(name);
        if (iter == map_.end())
        {
            return NULL;
        }
        nameable * result = iter->second;
        map_.erase(iter);
        return result;
    }

    void clear() { map_.clear(); }

    nameable::dictionary map_;
};

} // *namespace* - gem

#endif
