<?php
/*
Plugin Name: Feedback Champuru
Version: 0.4.0
Plugin URI: 
Description: This plugin makes WordPress Comment boisterous adding feedbacks of Twitter, Social Bookmarks and so on.
Author: wokamoto
Author URI: http://dogmap.jp/
Text Domain: feedback-champuru
Domain Path: /languages/

License:
 Released under the GPL license
  http://www.gnu.org/copyleft/gpl.html

  Copyright 2010 wokamoto (email : wokamoto1973@gmail.com)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

//**********************************************************************************
// Require wp-load.php or wp-config.php
//**********************************************************************************
if( !function_exists('get_option') ) {
	$path = (defined('ABSPATH') ? ABSPATH : dirname(dirname(dirname(dirname(__FILE__)))) . '/');
	require_once(file_exists($path.'wp-load.php') ? $path.'wp-load.php' : $path.'wp-config.php');
}

//**********************************************************************************
// Template tag
//**********************************************************************************
function feedback_type($commenttxt = false, $trackbacktxt = false, $pingbacktxt = false, $tweettext = false, $hatenatext = false, $delisioustext = false, $livedoortext = false, $buzzurltext = false, $echo = true) {
	global $feedback_champru;

	if (!isset($feedback_champru))
		$feedback_champru = new FeedbackChampuru();

	add_filter('get_comment_type', array(&$feedback_champru, 'get_comment_type'));
	$type = get_comment_type();
	remove_filter('get_comment_type', array(&$feedback_champru, 'get_comment_type'));

	switch( $type ) {
		case 'trackback' :
			$text = ( false === $trackbacktxt ? __( 'Trackback' ) : $trackbacktxt );
			break;
		case 'pingback' :
			$text = ( false === $pingbacktxt ? __( 'Pingback' ) : $pingbacktxt );
			break;
		case 'tweet' :
			$text = ( false === $tweettext ? __( 'Tweet', 'feedback-champuru' ) : $tweettext );
			break;
		case 'hatena' :
			$text = ( false === $hatenatext ? __( 'Hatena Bookmark', 'feedback-champuru' ) : $hatenatext );
			break;
		case 'delicious' :
			$text = ( false === $delisioustext ? __( 'Delicious', 'feedback-champuru' ) : $delisioustext );
			break;
		case 'livedoor' :
			$text = ( false === $livedoortext ? __( 'livedoor Clip', 'feedback-champuru' ) : $delisioustext );
			break;
		case 'buzzurl' :
			$text = ( false === $buzzurltext ? __( 'buzzurl', 'feedback-champuru' ) : $delisioustext );
			break;
		default :
			$text = ( false === $commenttxt ? _x( 'Comment', 'noun' ) : $commenttxt );
	}

	if ($echo)
		echo $text;
	else
		return $text;
}

//**********************************************************************************
// Feedback Champuru
//**********************************************************************************
class FeedbackChampuru {
	var $plugin_name   = 'feedback-champuru';
	var $plugin_ver    = '0.4.0';

	var $plugin_dir;
	var $plugin_file;
	var $plugin_url;
	var $textdomain_name;

	var $icon_cache = true;
	var $cache_path;
	var $cache_url;

	var $topsy_api     = 'http://otter.topsy.com/trackbacks.json?url=%s&tracktype=tweet&perpage=50';
	var $hatena_api    = 'http://b.hatena.ne.jp/entry/jsonlite/?url=%s';
	var $delicious_api = 'http://feeds.delicious.com/v2/json/url/%s';
	var $livedoor_api  = 'http://api.clip.livedoor.com/json/comments?link=%s';
	var $buzzurl_api   = 'http://api.buzzurl.jp/api/posts/get/v1/json/?url=%s';

	var $meta_key_pre  = '_feedback_';

	var $cache_expired = 30;			// Request cache expired (minutes)
	var $comment_type  = '';
	var $comment_order = 'ASC';

	var $feedbacks     = array('tweet', 'hatena', 'delicious');

	var $comment_id    = 0;

	/**********************************************************
	* Constructor
	***********************************************************/
	function FeedbackChampuru(){
		$this->__construct();
	}
	function __construct() {
		$this->_set_plugin_dir(__FILE__);
		$this->_load_textdomain();

		// feedback types
		$this->feedbacks = apply_filters('feedback-champuru/feedback_types', $this->feedbacks);

		// icon cache
		if ( apply_filters('feedback-champuru/icon_cache', $this->icon_cache) && function_exists('imagepng') ) {
			$icon_cache_dir   = apply_filters('feedback-champuru/icon_cach_dir', 'cache/' . $this->plugin_name);
			$this->cache_path = $this->_wp_content_dir( $icon_cache_dir );
			if( !file_exists( dirname($this->cache_path) ) )
				@mkdir( dirname($this->cache_path), 0777 );
			if( !file_exists($this->cache_path) )
				@mkdir( $this->cache_path, 0777 );
			$this->icon_cache = file_exists($this->cache_path);
			$this->cache_url  = $this->_wp_content_url( $icon_cache_dir );
		} else {
			$this->icon_cache = false;
		}

		// add filters
		add_filter('comments_array', array(&$this, 'comments_champuru'), 10, 2);
		add_filter('get_comments_number', array(&$this, 'get_comments_number'), 10, 2);
		add_filter('comment_class', array(&$this, 'comment_class'), 10, 4);
		add_filter('get_avatar', array(&$this, 'get_avatar'), 10, 5);
		add_filter('comment_reply_link', array(&$this, 'comment_reply_link'), 10, 4);
		add_filter('edit_comment_link', array(&$this, 'edit_comment_link'), 10, 2);

		add_filter('feedback-champuru/cache_expired', array(&$this, 'get_cache_expired'), 10, 2);
		add_filter('feedback-champuru/comments_array', array(&$this, 'set_gmt_offset'), 10, 2);

		add_filter('post_link', array(&$this, 'percent_encode_capital_letter'));
		add_filter('page_link', array(&$this, 'percent_encode_capital_letter'));
		add_filter('tag_link', array(&$this, 'percent_encode_capital_letter'));
		add_filter('category_link', array(&$this, 'percent_encode_capital_letter'));
		add_filter('preview_post_link', array(&$this, 'percent_encode_capital_letter'));
		add_filter('get_the_guid', array(&$this, 'percent_encode_capital_letter'));
	}

	// set plugin dir
	function _set_plugin_dir( $file = '' ) {
		$file_path = ( !empty($file) ? $file : __FILE__);
		$filename = explode("/", $file_path);
		if(count($filename) <= 1) $filename = explode("\\", $file_path);
		$this->plugin_dir  = $filename[count($filename) - 2];
		$this->plugin_file = $filename[count($filename) - 1];
		$this->plugin_url  = $this->_wp_plugin_url($this->plugin_dir);
		unset($filename);
	}

	// load textdomain
	function _load_textdomain( $sub_dir = 'languages', $textdomain_name = '' ) {
		$this->textdomain_name = (empty($textdomain_name) ? $this->plugin_dir : $textdomain_name);
		$plugins_dir = trailingslashit(defined('PLUGINDIR') ? PLUGINDIR : 'wp-content/plugins');
		$abs_plugin_dir = $this->_wp_plugin_dir($this->plugin_dir);
		$sub_dir = ( !empty($sub_dir)
			? preg_replace('/^\//', '', $sub_dir)
			: (file_exists($abs_plugin_dir.'languages') ? 'languages' : (file_exists($abs_plugin_dir.'language') ? 'language' : (file_exists($abs_plugin_dir.'lang') ? 'lang' : '')))
			);
		$textdomain_dir = trailingslashit(trailingslashit($this->plugin_dir) . $sub_dir);

		if ( $this->_check_wp_version("2.6") && defined('WP_PLUGIN_DIR') )
			load_plugin_textdomain($this->textdomain_name, false, $textdomain_dir);
		else
			load_plugin_textdomain($this->textdomain_name, $plugins_dir . $textdomain_dir);
	}

	// check wp version
	function _check_wp_version($version, $operator = ">=") {
		global $wp_version;
		return version_compare($wp_version, $version, $operator);
	}

	// WP_CONTENT_DIR
	function _wp_content_dir($path = '') {
		return trailingslashit( trailingslashit( defined('WP_CONTENT_DIR')
			? WP_CONTENT_DIR
			: trailingslashit(ABSPATH) . 'wp-content'
			) . preg_replace('/^\//', '', $path) );
	}

	// WP_CONTENT_URL
	function _wp_content_url($path = '') {
		return trailingslashit( trailingslashit( defined('WP_CONTENT_URL')
			? WP_CONTENT_URL
			: trailingslashit(get_option('siteurl')) . 'wp-content'
			) . preg_replace('/^\//', '', $path) );
	}

	// WP_PLUGIN_DIR
	function _wp_plugin_dir($path = '') {
		return trailingslashit($this->_wp_content_dir( 'plugins/' . preg_replace('/^\//', '', $path) ));
	}

	// WP_PLUGIN_URL
	function _wp_plugin_url($path = '') {
		return trailingslashit($this->_wp_content_url( 'plugins/' . preg_replace('/^\//', '', $path) ));
	}

	// Get post_meta
	function _get_post_meta($post_id, $key) {
		return maybe_unserialize(get_post_meta($post_id, $key, true));
	}

	// Add or Update post_meta
	function _update_post_meta($post_id, $key, $val) {
		return (
			add_post_meta($post_id, $key, $val, true) or
			update_post_meta($post_id, $key, $val)
			);
	}

	// json decode
	function json_decode( $string ) {
		if ( function_exists('json_decode') ) {
			return json_decode( $string );
		} else {
			// For PHP < 5.2.0
			if ( !class_exists('Services_JSON') ) {
				require_once( 'includes/class-json.php' );
			}
			$json = new Services_JSON();
			return $json->decode( $string );
		}
	}

	// remote_get
	function remote_get( $url, $args = array() ){
		$ret = wp_remote_get( $url, $args );
		if ( is_array($ret) && isset($ret["body"]) && !empty($ret["body"]) )
			return $ret["body"];
		else
			return false;
	}

	//**************************************************************************************
	// plugin activation
	//**************************************************************************************
	function activation(){
	}

	//**************************************************************************************
	// plugin deactivation
	//**************************************************************************************
	function deactivation(){
		global $wpdb;
		$wpdb->query($wpdb->prepare(
			"DELETE FROM $wpdb->postmeta WHERE meta_key like %s" ,
			$this->meta_key_pre . '%'
			)
		);
	}

	/**********************************************************
	* Champuru!
	***********************************************************/
	function comments_champuru($comments, $post_id){
		$permalink = get_permalink($post_id);
		foreach ($this->feedbacks as $type) {
			$comments = array_merge($comments ,	$this->_get_feedback($type, $post_id, $permalink));
		}
		$this->comment_order = (strtoupper(get_settings('comment_order')) !== 'DESC' ? 'ASC' : 'DESC');
		usort($comments, array($this, '_comments_cmp'));

		return $comments;
	}

	function _comments_cmp($a, $b){
	    if ($a->comment_date == $b->comment_date)
	        return 0;
	    if (strtotime($a->comment_date) < strtotime($b->comment_date)) 
			return ( $this->comment_order === 'ASC' ? -1 : 1);
		else
			return ( $this->comment_order === 'ASC' ? 1 : -1);
	}

	function comment_build($type, $post_id, $author = '', $author_url = '', $datetime = 0, $content = '', $photo_url = ''){
		$gmt_offset = 3600 * get_option('gmt_offset');
		$comment = (object) array(
			"comment_ID"           => $type . '-' . $this->comment_id++ ,
			"comment_post_ID"      => (string) $post_id ,
			"comment_author"       => $author ,
			"comment_author_email" => $photo_url ,
			"comment_author_url"   => $author_url ,
			"comment_author_IP"    => '' ,
			"comment_date"         => date('Y-m-d H:i:s', $datetime + ($type !== 'hatena' ? $gmt_offset : 0 )) ,
			"comment_date_gmt"     => gmdate('Y-m-d H:i:s', $datetime) ,
			"comment_content"      => $content ,
			"comment_karma"        => '0' ,
			"comment_approved"     => '1' ,
			"comment_agent"        => '0' ,
			"comment_type"         => apply_filters('feedback-champuru/comment_type', $this->comment_type, $type) ,
			"comment_parent"       => '0' ,
			"user_id"              => '0' ,
			);
		if ($this->akismet($post_id, $comment))
			$comment->comment_approved = 'spam';
		return $comment;
	}

	function _get_type_from_ID($comment_id = ''){
		$comment_id = (empty($comment_id)) || is_null($comment_id)
			? get_comment_ID()
			: (!is_null($comment_id) ? $comment_id : '');
		$comment_type = '';
		if (preg_match('/^([^\-]*)\-[\d]+$/', $comment_id, $matches))
			$comment_type = strtolower($matches[1]);
		unset($matches);
		return $comment_type;
	}

	/**********************************************************
	* Get Feedbacks
	***********************************************************/
	function _get_feedback($type, $post_id, $permalink){
		$comments_array = array();
		$permalink = $this->percent_encode_capital_letter($permalink);

		switch ($type){
		case 'tweet':
			$comments_array = $this->_get_topsy($type, $post_id, $permalink);
			break;
		case 'hatena':
			$comments_array = $this->_get_hatena($type, $post_id, $permalink);
			break;
		case 'delicious':
			$comments_array = $this->_get_delicious($type, $post_id, $permalink);
			break;
		case 'livedoor':
			$comments_array = $this->_get_livedoor($type, $post_id, $permalink);
			break;
		case 'buzzurl':
			$comments_array = $this->_get_buzzurl($type, $post_id, $permalink);
			break;
		default:
			$comments_array = apply_filters('feedback-champuru/get_feedback', $comments_array, $type, $post_id, $permalink);
			break;
		}

		return apply_filters('feedback-champuru/comments_array', $comments_array, $type);
	}

	/**********************************************************
	* Get Twitter Feedbacks (Topsy)
	***********************************************************/
	function _get_topsy($type, $post_id, $permalink){
		$meta_key = $this->meta_key_pre . $type;
		$cache = $this->_get_post_meta($post_id, $meta_key);
		if ( isset($cache["expired"]) && $cache["expired"] > time() )
			return $cache["comments"];
		if ( !is_singular() || is_robots() )
			return ( isset($cache["comments"]) ? $cache["comments"] : array() );

		$comments = array();
		$response = $this->remote_get(sprintf($this->topsy_api, urlencode($permalink)));
		if ($response !== false){
			$json = $this->json_decode($response);
			$list = (isset($json->response) ? $json->response->list : array());
			foreach ((array) $list as $item){
//				$author     = esc_attr($item->author->name);
				$author     = preg_replace('/^http:\/\/topsy\.com\/twitter\/([^\?]*)\?.*$/i', '$1', esc_attr($item->author->topsy_author_url));
//				$author_url = esc_attr((isset($item->author->url) && !empty($item->author->url)) ? $item->author->url : 'http://twitter.com/' . $author);
//				$author_url = esc_attr('http://topsy.com/' . str_replace('http://', '', $permalink));
				$author_url = esc_attr($item->permalink_url);
				$datetime   = (int) $item->date;
				$content    = esc_attr($item->content);
				$photo_url  = esc_attr($item->author->photo_url);

//				if ( !empty($content) )
//					$content .= "\n\n<i>" .
//						sprintf(
//							__('This comment was originally posted on %s.', 'feedback-champuru' ) ,
//							'<a href="' . esc_attr($item->permalink_url) . '">Twitter</a>'
//							) . '</i>';

				$content    = apply_filters('feedback-champuru/content', $content, $type, $author, $author_url, $datetime, $photo_url, $item);

				if ( $content )
					$comments[] = $this->comment_build($type, $post_id, $author, $author_url, $datetime, $content, $photo_url);
			}
		}
		$comments = (count($comments) > 0 ? $comments : (isset($cache["comments"]) ? $cache["comments"] : $comments));

		$cache = array(
			"expired" => time() + apply_filters('feedback-champuru/cache_expired', $this->cache_expired * 60, $post_id) ,
			"comments" => $comments ,
			);
		$this->_update_post_meta($post_id, $meta_key, $cache );

		return $comments;
	}

	/**********************************************************
	* Get Hatena Bookmark Feedbacks
	***********************************************************/
	function _get_hatena($type, $post_id, $permalink){
		$meta_key = $this->meta_key_pre . $type;
		$cache = $this->_get_post_meta($post_id, $meta_key);
		if ( isset($cache["expired"]) && $cache["expired"] > time() )
			return $cache["comments"];
		if ( !is_singular() || is_robots() )
			return ( isset($cache["comments"]) ? $cache["comments"] : array() );

		$comments = array();
		$response = $this->remote_get(sprintf($this->hatena_api, urlencode($permalink)));
		if ($response !== false){
			$json = $this->json_decode($response);
			$list = (isset($json->bookmarks) ? $json->bookmarks : array());
			foreach ((array) $list as $item){
				$author     = esc_attr($item->user);
//				$author_url = 'http://b.hatena.ne.jp/' . $author;
				$author_url = esc_attr('http://b.hatena.ne.jp/entry/' . str_replace('http://', '', $permalink));
				$datetime   = strtotime(str_replace('/', '-', $item->timestamp));
				$content    = esc_attr($item->comment);
				$photo_url  = 'http://www.hatena.ne.jp/users/' . substr($author, 0, 2) . '/' . $author . '/profile.gif';

				$content    = apply_filters('feedback-champuru/content', $content, $type, $author, $author_url, $datetime, $photo_url, $item);

				if ( $content )
					$comments[] = $this->comment_build($type, $post_id, $author, $author_url, $datetime, $content, $photo_url);
			}
		}
		$comments = (count($comments) > 0 ? $comments : (isset($cache["comments"]) ? $cache["comments"] : $comments));

		$cache = array(
			"expired" => time() + apply_filters('feedback-champuru/cache_expired', $this->cache_expired * 60, $post_id) ,
			"comments" => $comments ,
			);
		$this->_update_post_meta($post_id, $meta_key, $cache );

		return $comments;
	}

	/**********************************************************
	* Get Delicious Feedbacks
	***********************************************************/
	function _get_delicious($type, $post_id, $permalink){
		$meta_key = $this->meta_key_pre . $type;
		$cache = $this->_get_post_meta($post_id, $meta_key);
		if ( isset($cache["expired"]) && $cache["expired"] > time() )
			return $cache["comments"];
		if ( !is_singular() || is_robots() )
			return ( isset($cache["comments"]) ? $cache["comments"] : array() );

		$comments = array();
		$response = $this->remote_get(sprintf($this->delicious_api, md5($permalink)));
		if ($response !== false){
			$json = $this->json_decode($response);
			foreach ((array) $json as $item){
				$author     = esc_attr($item->a);
//				$author_url = 'http://delicious.com/'.$author;
				$author_url = esc_attr('http://delicious.com/url/' . md5($permalink));
				$datetime   = strtotime($item->dt);
				$content    = esc_attr($item->n);
				$photo_url  = '';

				$content    = apply_filters('feedback-champuru/content', $content, $type, $author, $author_url, $datetime, $photo_url, $item);

				if ( $content )
					$comments[] = $this->comment_build($type, $post_id, $author, $author_url, $datetime, $content, $photo_url);
			}
		}
		$comments = (count($comments) > 0 ? $comments : (isset($cache["comments"]) ? $cache["comments"] : $comments));

		$cache = array(
			"expired" => time() + apply_filters('feedback-champuru/cache_expired', $this->cache_expired * 60, $post_id) ,
			"comments" => $comments ,
			);
		$this->_update_post_meta($post_id, $meta_key, $cache );

		return $comments;
	}

	/**********************************************************
	* Get livedoor clip Feedbacks
	***********************************************************/
	function _get_livedoor($type, $post_id, $permalink){
		$meta_key = $this->meta_key_pre . $type;
		$cache = $this->_get_post_meta($post_id, $meta_key);
		if ( isset($cache["expired"]) && $cache["expired"] > time() )
			return $cache["comments"];
		if ( !is_singular() || is_robots() )
			return ( isset($cache["comments"]) ? $cache["comments"] : array() );

		$comments = array();
		$response = $this->remote_get(sprintf($this->livedoor_api, urlencode($permalink)));
		if ($response !== false){
			$json = $this->json_decode($response);
			$list = (isset($json->Comments) ? $json->Comments : array());
			foreach ((array) $list as $item){
				$author     = esc_attr($item->livedoor_id);
//				$author_url = 'http://clip.livedoor.com/clips/' . $author;
				$author_url = esc_attr('http://clip.livedoor.com/page/' . $permalink);
				$datetime   = $item->created_on;
				$content    = esc_attr($item->notes);
				$photo_url  = 'http://image.profile.livedoor.jp/icon/' . $author . '_60.gif';

				$content    = apply_filters('feedback-champuru/content', $content, $type, $author, $author_url, $datetime, $photo_url, $item);

				if ( $content )
					$comments[] = $this->comment_build($type, $post_id, $author, $author_url, $datetime, $content, $photo_url);
			}
		}
		$comments = (count($comments) > 0 ? $comments : (isset($cache["comments"]) ? $cache["comments"] : $comments));

		$cache = array(
			"expired" => time() + apply_filters('feedback-champuru/cache_expired', $this->cache_expired * 60, $post_id) ,
			"comments" => $comments ,
			);
		$this->_update_post_meta($post_id, $meta_key, $cache );

		return $comments;
	}

	/**********************************************************
	* Get buzzurl Feedbacks
	***********************************************************/
	function _get_buzzurl($type, $post_id, $permalink){
		$meta_key = $this->meta_key_pre . $type;
		$cache = $this->_get_post_meta($post_id, $meta_key);
		if ( isset($cache["expired"]) && $cache["expired"] > time() )
			return $cache["comments"];
		if ( !is_singular() || is_robots() )
			return ( isset($cache["comments"]) ? $cache["comments"] : array() );

		$comments = array();
		$response = $this->remote_get(sprintf($this->buzzurl_api, urlencode($permalink)));
		if ($response !== false){
			$json = $this->json_decode($response);
			$list = (isset($json->posts) ? $json->posts : array());
			foreach ((array) $list as $item){
				$author     = esc_attr($item->user_name);
//				$author_url = 'http://buzzurl.jp/user/' . $author;
				$author_url = esc_attr('http://buzzurl.jp/entry/' . $permalink);
				$datetime   = strtotime($item->date);
				$content    = esc_attr($item->comment);
				$photo_url  = 'http://buzzurl.jp.eimg.jp/user/' . $author . '/photo.png';

				$content    = apply_filters('feedback-champuru/content', $content, $type, $author, $author_url, $datetime, $photo_url, $item);

				if ( $content )
					$comments[] = $this->comment_build($type, $post_id, $author, $author_url, $datetime, $content, $photo_url);
			}
		}
		$comments = (count($comments) > 0 ? $comments : (isset($cache["comments"]) ? $cache["comments"] : $comments));

		$cache = array(
			"expired" => time() + apply_filters('feedback-champuru/cache_expired', $this->cache_expired * 60, $post_id) ,
			"comments" => $comments ,
			);
		$this->_update_post_meta($post_id, $meta_key, $cache );

		return $comments;
	}

	/**********************************************************
	* Akismet
	***********************************************************/
	function akismet($post_id, $comment) {
		global $akismet_api_host, $akismet_api_port;

		if ( !function_exists('akismet_http_post') || !(get_option('wordpress_api_key') || $wpcom_api_key) )
			return false;

		$c = array();
		$author = $author_email = $author_url = $content = '';

		$c['blog'] = get_option( 'home' );
		$c['comment_type'] = $comment->comment_type;
		if ( $permalink = get_permalink() )
			$c['permalink'] = $permalink;
		if ( '' != $comment->comment_author )
			$c['comment_author'] = $comment->comment_author;
		if ( '' != $comment->comment_author_email )
			$c['comment_author_email'] = $comment->comment_author_email;
		if ( '' != $comment->comment_author_url )
			$c['comment_author_url'] = $comment->comment_author_url;
		if ( '' != $comment->comment_content )
			$c['comment_content'] = $comment->comment_content;

		$query_string = '';
		foreach ( $c as $key => $data ) {
			$query_string .= $key . '=' . urlencode( stripslashes( $data ) ) . '&';
		}
		$result_key = md5($query_string);

		$meta_key = $this->meta_key_pre . 'akismet_result';
		$akismet_result = (array) $this->_get_post_meta($post_id, $meta_key);
		if ( !isset($akismet_result[$result_key]) ) {
			$response = akismet_http_post( $query_string, $akismet_api_host, '/1.1/comment-check', $akismet_api_port );
			$akismet_result[$result_key] = ('true' == $response[1] ? true : false);
			$this->_update_post_meta($post_id, $meta_key, $akismet_result );
		}
		return $akismet_result[$result_key];
	}

	/**********************************************************
	* get comments number
	***********************************************************/
	function get_comments_number($count, $post_id = ''){
		$comments = $this->comments_champuru(array(), $post_id);
		return $count + count($comments);
	}

	/**********************************************************
	* comment class
	***********************************************************/
	function comment_class($classes, $class = '', $comment_id = '', $post_id = ''){
		$comment_type = $this->_get_type_from_ID($comment_id);
		if (!empty($comment_type))
			$classes[] = esc_attr($comment_type);
		return $classes;
	}

	/**********************************************************
	* comment type
	***********************************************************/
	function get_comment_type($comment_type){
		global $comment;
		$comment_type = $this->_get_type_from_ID(isset($comment) ? $comment->comment_ID : '');
		if (!empty($comment_type))
			$comment_type = esc_attr($comment_type);
		return $comment_type;
	}

	/**********************************************************
	* get avatar
	***********************************************************/
	function get_avatar($avatar, $id_or_email = '', $size = '96', $default = '', $alt = false){
		global $pagenow, $comment;

		if($pagenow == 'options-discussion.php')
			return $avatar;
		if (!isset($comment))
			return $avatar;

		$type_pattern = '/^('.implode('|',$this->feedbacks).')\-/i';
		if (preg_match('/^http:\/\//i', $comment->comment_author_email)) {
			$img_url = $comment->comment_author_email;
		} elseif (preg_match($type_pattern, $comment->comment_ID, $matches)) {
			$img_url = site_url(str_replace(ABSPATH, '', dirname(__FILE__))) . '/images/';
			$img_url .= $matches[1] . ( $size <= 24 ? '_16.png' : '_32.png' );
		} else {
			$img_url = '';
		}

		if (!empty($img_url)) {
			if ( $this->icon_cache ) {
				$cache_file_name = $this->_cache_file_name($img_url, $size);
				$cache_file = $this->cache_path . $cache_file_name;
				$cache_expired = $this->cache_expired * 60;
//				if ( file_exists($cache_file) && time() < (filemtime($cache_file) + $cache_expired) )
				if ( file_exists($cache_file) )
					$img_url = $this->cache_url . $cache_file_name;
				else
					$img_url = $this->plugin_url
						. basename(__FILE__)
						. '?url=' . base64_encode($img_url)
						. '&amp;size=' . $size;
			}

			$avatar = preg_replace(
				  '/^(.*<img [^>]* src=[\'"])http:\/\/[^\'"]+([\'"][^>]*\/>.*)$/i'
				, "$1{$img_url}$2"
				, $avatar);
		}

		return $avatar;
	}

	// _cache_file_name
	function _cache_file_name($img_url, $img_size) {
		return md5($img_url . $img_size) . '.png';
	}

	// Function getImage
	function get_image($img_url, $img_size){
		if(parse_url($img_url) === false)
			die();
		if(is_numeric($img_size) === false || $img_size > 96)
			die();

		$cache_file = $this->cache_path . $this->_cache_file_name($img_url, $img_size);
		$cache_expired = $this->cache_expired * 60;

		$image = false;
		if( !file_exists($cache_file) )
			$image = $this->_get_resize_image($img_url, $img_size, $cache_file);
//		elseif ( time() >= (filemtime($cache_file) + $cache_expired) )
//			$image = $this->_get_resize_image($img_url, $img_size, $cache_file);

		if ($image === false && file_exists($cache_file))
			$image = imagecreatefrompng($cache_file);

		if ($image !== false) {
			header('Content-Type: image/png');
			header('Expires: '.gmdate('D, d M Y H:i:s', time() + $cache_expired).' GMT');
			imagepng($image);
			imagedestroy($image);
		} else {
			header("HTTP/1.0 404 Not Found");
		}
	}

	// Function _get_resize_image
	function _get_resize_image($img_url, $img_size = 96, $cache_file = '') {
		$imgbin = $this->remote_get($img_url);
		if ($imgbin === false)
			return false;

		$img_resized = @imagecreatetruecolor($img_size, $img_size);
		$bgc = @imagecolorallocate($img_resized, 255, 255, 255);
		@imagefilledrectangle($img_resized, 0, 0, $img_size, $img_size, $bgc);

		$img = @imagecreatefromstring($imgbin);
		if($img === false)
			return ( !file_exists($cache_file) ? $img_resized : false );

		$img_width  = imagesx($img);
		$img_height = imagesx($img);
		@imagecopyresampled(
			$img_resized,
			$img,
			0, 0, 0, 0,
			$img_size, $img_size,
			$img_width, $img_height);

		@imagepng($img_resized, $cache_file);

		return $img_resized;
	}

	/**********************************************************
	* comment reply link
	***********************************************************/
	function comment_reply_link($link, $args = '', $comment = '', $post = ''){
		return (preg_match('/^[\d]+$/', isset($comment->comment_ID) ? $comment->comment_ID : get_comment_ID()) ? $link : '');
	}

	/**********************************************************
	* edit comment link
	***********************************************************/
	function edit_comment_link($link, $comment_id = ''){
		return (preg_match('/^[\d]+$/', !empty($comment_id) ? $comment_id : get_comment_ID()) ? $link : '');
	}

	/**********************************************************
	* percent encode capital letter
	***********************************************************/
	function percent_encode_capital_letter($uri) {
		return preg_replace_callback('/(%[0-9a-f]{2}?)+/', create_function('$matches','return strtoupper($matches[0]);'), $uri);
	}

	/**********************************************************
	* Get cache expired
	***********************************************************/
	function get_cache_expired($expired, $id = 0){
		$post = &get_post($id);
		$post_date_diff = (time() - strtotime($post->post_date_gmt . ' GMT')) / 60;
		return (int) ( $post_date_diff > 100 && log($post_date_diff) > 1
			? $expired * log10($post_date_diff)
			: $expired );
	}

	/**********************************************************
	* Set gmt offset
	***********************************************************/
	function set_gmt_offset( $comments_array, $type = '' ){
		$gmt_offset = 3600 * get_option('gmt_offset');
		foreach ( (array)$comments_array as $key => $comment ) {
			switch( $type ) {
				case 'hatena' :
					$comments_array[$key]->comment_date_gmt = gmdate('Y-m-d H:i:s', strtotime($comment->comment_date_gmt) - 32400);
				case 'tweet' :
				case 'delicious' :
					$comments_array[$key]->comment_date = date('Y-m-d H:i:s', strtotime($comment->comment_date_gmt) + $gmt_offset);
					break;
				case 'livedoor' :
				case 'buzzurl' :
					break;
				default :
					break;
			}
		}
		return $comments_array;
	}
}
global $feedback_champru;

$feedback_champru = new FeedbackChampuru();

if (function_exists('register_activation_hook'))
	register_activation_hook(__FILE__, array(&$feedback_champru, 'activation'));
if (function_exists('register_deactivation_hook'))
	register_deactivation_hook(__FILE__, array(&$feedback_champru, 'deactivation'));

if ( $feedback_champru->icon_cache && strpos($_SERVER['PHP_SELF'], basename(__FILE__)) !== false && isset($_GET['url']) ) {
	// Get Image from Cache
	$img_url  = base64_decode($_GET['url']);
	$img_size = (int) (isset($_GET['size']) ? stripslashes($_GET['size']) : 48);

	$feedback_champru->get_image($img_url, $img_size);
}
?>
