<?php 
/* 
Plugin Name: Ktai Location
Plugin URI: http://wppluginsj.sourceforge.jp/ktai_location/
Description: Read latitude-longitude data from a map URL which is provided with location services for mobile phone, or from an EXIF infomation of a posted picture. Or, convert place names and/or addresses to latitude-longitude data (geocoding). And save the location data as "Lat_Long" custom fields.
Author: IKEDA yuriko
Version: 0.97
Author URI: http://www.yuriko.net/cat/wordpress
*/

/*  Copyright (c) 2007 yuriko

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; version 2 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 * Special Thanks:
	- "TSJ Blog II": author of wp-eznavi.
		[http://blog2.atsj.net/]
*/

//define('GOOGLEMAPS_API_KEY', 'XXXXXXXXXX');

define('GEO_META_FIELD_NAME', 'Lat_Long');
define('CSS_CLASS_FOR_LOCATION', 'locationurl');
define('EZNAVI_WGS84',  0);
define('EZNAVI_TOKYO',  1);
define('EZNAVI_ITRF',   2);
define('EZNAVI_DMS',    0);
define('EZNAVI_DEGREE', 1);
define('NAVITIME_WGS84',  0);
define('NAVITIME_TOKYO',  1);
define('NAVITIME_ITRF',   2);
define('NAVITIME_DMS',    0);
define('NAVITIME_DEGREE', 1);

define('EQUATORIAL_RADIUS_TOKYO', 6377397.155);
define('EQUATORIAL_RADIUS_WGS84', 6378137.000);
define('REV_OBLATENESS_TOKYO', 299.152813);
define('REV_OBLATENESS_WGS84', 298.257223563);

add_action('wp_head',      array('Ktai_Location', 'output_style'), 20);
add_action('publish_post', array('Ktai_Location', 'read_location'), 20);
//add_action('edit_post',    array('Ktai_Location', 'read_location'), 20);

/* ==================================================
 *   Ktai_Location class
   ================================================== */

class Ktai_Location {
	var $content;
	var $locations;

/* ==================================================
 * Add custom style sheet into XHTML header.
 * @param	none
 * @return	none
*/
//static
function output_style() {
?>
<style type="text/css" media="all">
.<?php echo CSS_CLASS_FOR_LOCATION; ?> {
	display:none;
}
</style>
<?php
}

/* ==================================================
 * Read location info from post content.
 * @param	Int $postID
 * @return	Int $postID
*/
//static
function read_location($postID) {
	if (! is_numeric($postID)) {
		return $postID;
	}

	$post = get_post($postID);
	if ($post->ID != $postID) {
			return $postID;
	}
	$loc = new Ktai_Location($post->post_content);
	if (count($loc->locations)) {
		$loc->uniq();
		$updated = $loc->update_meta($post->ID);
		$touched = FALSE;
		foreach ($loc->locations as $l) {
			if ($updated && isset($l->url)) {
				$touched = $loc->format_url($l->url);
			} elseif (isset($l->geotag)) {
				$touched = $loc->delete_geotag($l->geotag, $l->place);
			}
		}
		if ($touched) {
			$post->post_content = $loc->content;
			global $wpdb;
			$content_sql = $wpdb->escape($post->post_content);
			$id_sql = intval($post->ID);
			$wpdb->query("UPDATE {$wpdb->posts} SET post_content = '$content_sql' WHERE ID = $id");
			$posts = array($post);
			update_post_cache($posts);
		}
	}
	return  $postID;
}

/* ==================================================
 * Create an array of location info objects along location services.
 * @param	String $content
 * @return	none
*/
//static
function Ktai_Location($content) {
	$this->content = $content;
	$this->locations = array_merge(
		$this->read_gps_url(), 
		$this->read_gps_exif(), 
		$this->geocoding()
	);
	return;
}


/* ==================================================
 * Search for GPS URLs.
 * @param	none;
 * @return	Array $locations
*/
//private
function read_gps_url() {
	$locations = array();
	if (preg_match_all('#^(\S+\s*)?http://(\S+)#m', $this->content, $m)) {
		foreach ($m[0] as $n) {
			switch (TRUE) {
			case $loc = Ktai_Location_EZ_Navi::factory($n):
				break;
			case $loc = Ktai_Location_DoCoMoGPS::factory($n):
				break;
			case $loc = Ktai_Location_Navitime::factory($n):
				break;
			case $loc = Ktai_Location_itsumoNavi::factory($n):
				break;
			case $loc = Ktai_Location_iMapFan::factory($n):
				break;
			case $loc = Ktai_Location_iMapion::factory($n):
				break;
			case $loc = Ktai_Location_iZenrin::factory($n):
				break;
			}
			if ($loc) {
				$locations[] = $loc;
			}
		}
	}
	return $locations;
}

/* ==================================================
 * Search for GPS field in EXIF.
 * @param	none;
 * @return	Array $locations
*/
//private
function read_gps_exif() {
	if (! function_exists('exif_read_data')) {
		return array();
	}
	$locations = Ktai_Location_EXIF::factory($this->content);
	return $locations ?  $locations : array();
}

/* ==================================================
 * Search for GPS field in EXIF.
 * @param	none;
 * @return	Array $locations
*/
//private
function geocoding() {
	$locations = Ktai_Location_Geocoding::factory($this->content);
	return $locations ?  $locations : array();
}

/* ==================================================
 * Remove the same locations.
 * @param	none
 * @return	Int $num_removed;
*/
//private
function uniq() {
	if (count($this->locations) < 1) {
		return FALSE;
	}
	$loc_index = array();
	$num_removed = 0;
	foreach ($this->locations as $i => $l) {
		if (! isset($l->lat) || ! isset($l->lon)) {
			continue;
		}
		$lat = round($l->lat, 10);
		$lon = round($l->lon, 10);
		if (isset($loc_index["$lat,$lon"])) {
			unset($this->locations[$i]);
			$num_removed++;
		} else {
			$loc_index["$lat,$lon"] = $i;
		}
	}
	return $num_removed;
}

/* ==================================================
 * Insert/Update the location info of the postmeta table of the DB.
 * @param	Int     $postID
 * @return	Boolean $do_update
*/
//private
function update_meta($postID) {
	$meta_values = get_post_meta($postID, GEO_META_FIELD_NAME);
	if ($meta_values) {
		foreach ($meta_values as $v) {
			$latlng = explode(',', $v);
			if (floatval($latlng[0]) && floatval($latlng[1])) {
				return FALSE; // keep existing Lat_Long field
			}
		}
		delete_post_meta($postID, GEO_META_FIELD_NAME);
	}
	foreach ($this->locations as $l) {
		if (isset($l->lat) && isset($l->lon)) {
			add_post_meta($postID, GEO_META_FIELD_NAME, $l->lat . ',' . $l->lon);
		}
	}
	return TRUE;
}

/* ==================================================
 * Format the location url from the content.
 * @param	String  $url
 * @return	Boolean $touched;
*/
//private
function format_url($url) {
	$new_url = str_replace('http://', 'HTTP://', $url);
	$this->content = str_replace($url, 
		'<div class="' . CSS_CLASS_FOR_LOCATION . '">' . $new_url . '</div>', 
		$this->content);
	return TRUE;
}

/* ==================================================
 * Delete geo tags from the content.
 * @param	String  $geotag
 * @return	Boolean $touched;
*/
//private
function delete_geotag($geotag, $place) {
	$this->content = str_replace($geotag, $place, $this->content);
	return TRUE;
}

// ===== End of class ====================
}

/* ==================================================
 *   Ktai_Location_Info class
   ================================================== */

class Ktai_Location_Info {
	var $lat;
	var $lon;
	var $alt;
	var $url;
	var $geometory;
	var $unit;
	var $accuracy;
	var $fixed_mode;
	var $geotag;
	var $place;

/* ==================================================
 * Parse an HTTP GET parameters in the request URL.
 * @param	String $query
 * @return	Array $params
*/
//protected
function get_params($query) {
	$query = str_replace('&amp;', '&', $query);
	parse_str(urldecode($query), $params);
	return $params;
}

/* ==================================================
 * Convert a degree-minites-second format into degree format.
 * @param	String $dms
 * @param	Int    $limit
 * @return	Float  $value
*/
//protected
function dms2deg($dms, $limit) {
	$value = NULL;
	if (preg_match("#([-+]?)(\d+)\.(\d+)\.([.\d]+)#", $dms, $m)) {
		$value = ($m[1] == '-' ? -1 : 1) * 
			(intval($m[2]) + intval($m[3]) / 60 + floatval($m[4]) / 3600);
	} elseif (is_numeric($str)) {
		$value = floatval($str);
	}
	if ($value < -1 * $limit || $value > $limit) {
		$value = NULL;
	}
	return $value;
}

/* ==================================================
 * Convert a location info from Tokyo geometry to WGS84 geometry.
 * @param	none
 * @return	none
*/
//protected
function tokyo2wgs84() {
	/*
	 * Thanks to "Mac.GPS.Perl"
	 * <http://homepage3.nifty.com/Nowral/02_DATUM/02_DATUM.html>
	 *
	 * ----- Test script -----
	$loc = new Ktai_Location_Info;
	$loc->lat = 35  + 20/60 + 39.984328/3600;
	$loc->lon = 138 + 35/60 +  8.086122/3600;
	$loc->alt = 697.681000;
	$good = array(35 + 20 / 60 + 51.685555 / 3600,  138 + 34 / 60 + 56.838916 / 3600, 737.895217);
	$loc->tokyo2wgs84();
	print_r($loc);
	if ($loc->lat == $good[0] && $loc->lon == $good[1] && $loc->alt == $good[2]) {
		echo "Good Anser!!\n";
	} else {
		echo "Bad Answer\nExpected: ";
		print_r($good);
	}
	 */
	if (! $this->alt) {
		$this->lat = $this->lat - 0.000106950 * $this->lat + 0.000017464 * $this->lon + 0.0046017;
		$this->lon = $this->lon - 0.000046038 * $this->lat - 0.000083043 * $this->lon + 0.0100400;
	} else {
		list($x, $y, $z) = $this->lla2xyz($this->lat, $this->lon, $this->alt, EQUATORIAL_RADIUS_TOKYO, REV_OBLATENESS_TOKYO);
		list($this->lat, $this->lon, $this->alt) = $this->xyz2lla($x - 148, $y + 507, $z + 681, EQUATORIAL_RADIUS_WGS84, REV_OBLATENESS_WGS84);
	}
	return;
}

//private
function lla2xyz($lat, $lon, $alt, $a, $ro) {
	$e2 = 2 / $ro - 1/($ro * $ro);
	$lat_sin = sin(deg2rad($lat));
	$lat_cos = cos(deg2rad($lat));
	$radius = $a / sqrt(1 - $e2 * $lat_sin * $lat_sin);
	$x = ($radius + $alt) * $lat_cos * cos(deg2rad($lon));
	$y = ($radius + $alt) * $lat_cos * sin(deg2rad($lon));
	$z = ($radius * (1 - $e2) + $alt) * $lat_sin;
	return array($x, $y, $z);
}

//private
function xyz2lla($x, $y, $z, $a, $ro) {
	$e2 = 2 / $ro - 1/($ro * $ro);
	$bda = sqrt(1 - $e2);
	$p = sqrt($x * $x + $y * $y);
	$t = atan2($z, $p * $bda);
	$t_sin = sin($t);
	$t_cos = cos($t);
	$lat = atan2($z + $e2 * $a / $bda * $t_sin * $t_sin * $t_sin, 
	             $p - $e2 * $a * $t_cos * $t_cos * $t_cos);
	$lon = atan2($y, $x);
	$lat_sin = sin($lat);
	$alt = $p /cos($lat) - $a / sqrt(1 - $e2 * $lat_sin * $lat_sin);
	return array(rad2deg($lat), rad2deg($lon), $alt);
}

/* ==================================================
 * Validate the location value.
 * @param	none
 * @return	none
*/
//protected
function check_values() {
	if (is_numeric($this->lat)) {
		$this->lat = floatval($this->lat);
	} else {
		$this->lat = NULL;
	}
	if (is_numeric($this->lon)) {
		$this->lon = floatval($this->lon);
	} else {
		$this->lon = NULL;
	}
	return ($this->lat && $this->lon);
}

// ===== End of class ====================
}

/* ==================================================
 *   Ktai_Location_EZ_Navi class
 * example: http://walk.eznavi.jp/map/?datum=AAA&unit=BBB&lat=XXX&lon=YYY
   ================================================== */

class Ktai_Location_EZ_Navi extends Ktai_Location_Info {

//static
function factory($url) {
	if (! preg_match('#^(\[GPS.*?\])?\s*http://walk\.eznavi\.jp/map/\?(\S+)#m', $url, $m)) {
		return NULL;
	}
	$loc = new Ktai_Location_EZ_Navi;
	$loc->url = $m[0];
	$params = parent::get_params($m[2]);
	$loc->geometory  = @$params['datum'];
	$loc->unit       = @$params['unit'];
	$loc->lat        = trim(@$params['lat']);
	$loc->lon        = trim(@$params['lon']);
	$loc->alt        = trim(@$params['alt']);
	$loc->fixed_mode = @$params['fm'];
	if ($loc->unit != EZNAVI_DEGREE) {
		$loc->lat = parent::dms2deg($loc->lat, 90);
		$loc->lon = parent::dms2deg($loc->lon, 180);
	}
	if (isset($loc->geometory) && $loc->geometory == EZNAVI_TOKYO) {
		$loc->tokyo2wgs84();
	}
	if (! $loc->check_values()) {
		return NULL;
	}
	return $loc;
}

// ===== End of class ====================
}

/* ==================================================
 *   Ktai_Location_DoCoMoGPS class
 * Document: http://www.nttdocomo.co.jp/service/imode/make/content/html/about/location_info.html
 * example: http://docomo.ne.jp/cp/map.cgi?lat=%2B34.40.47.178&lon=%2B135.10.40.074&geo=wgs84&x-acc=3
   ================================================== */

class Ktai_Location_DoCoMoGPS extends Ktai_Location_Info {

//static
function factory($url) {
	if (! preg_match('#^http://docomo\.ne\.jp/cp/map\.cgi\?(\S+)#m', $url, $m)) {
		return NULL;
	}
	$loc = new Ktai_Location_DoCoMoGPS;
	$loc->url = $m[0];
	$params = parent::get_params($m[1]);
	$loc->lat       = trim(@$params['lat']);
	$loc->lon       = trim(@$params['lon']);
	$loc->alt       = trim(@$params['alt']);
	$loc->unit      = @$params['unit'];
	$loc->geometory = @$params['geo'];
	$loc->accuracy  = @$params['x-acc'];
	$loc->lat       = parent::dms2deg($loc->lat, 90);
	$loc->lon       = parent::dms2deg($loc->lon, 180);
	if ($loc->geometory == 'tokyo') {
		$loc->tokyo2wgs84();
	}
	if (! $loc->check_values()) {
		return NULL;
	}
	return $loc;
}

// ===== End of class ====================
}

/* ==================================================
 *   Ktai_Location_Navitime class
 * example: http://map.navitime.jp/?datum=0&unit=1&lat=+34.65856&lon=+135.50640&fm=1
 * example: http://map.navitime.jp/?lat=%2835.51.57.17&lon=%28139.45.05.009&geo=wgs84&x-acc=3
   ================================================== */

class Ktai_Location_Navitime extends Ktai_Location_Info {

//static
function factory($url) {
	if (! preg_match('#^(\S+:\s*)?http://map\.navitime\.jp/?\?(\S+)#m', $url, $m)) {
		return NULL;
	}
	$loc = new Ktai_Location_Navitime;
	$loc->url = $m[0];
	$params = parent::get_params($m[2]);
	if (isset($params['pos'])) {
		preg_match('/^([NS])([.0-9]+)([EW])([.0-9]+)$/', $params['pos'], $n);
		$loc->lat = ($n[1] == 'S' ? -1: 1) * parent::dms2deg($n[2], 90);
		$loc->lon = ($n[3] == 'W' ? -1: 1) * parent::dms2deg($n[4], 180);
		$loc->accuracy  = isset($params['x-acr'])? $params['x-acr'] : @$params['x-acc'];
	} else {
		$loc->lat        = trim(@$params['lat']);
		$loc->lon        = trim(@$params['lon']);
		$loc->alt        = trim(@$params['alt']);
		$loc->unit       = @$params['unit'];
		$loc->fixed_mode = @$params['fm'];
		if ($loc->unit != NAVITIME_DEGREE) {
			$loc->lat = parent::dms2deg($loc->lat, 90);
			$loc->lon = parent::dms2deg($loc->lon, 180);
		}
	}
	$loc->geometory = isset($params['geo']) ? $params['geo'] : @$params['datum'];
	if (isset($loc->geometory) && ($loc->geometory == 'tokyo' || $loc->geometory == EZNAVI_TOKYO)) {
		$loc->tokyo2wgs84();
	}
	if (! $loc->check_values()) {
		return NULL;
	}
	return $loc;
}

// ===== End of class ====================
}

/* ==================================================
 *   Ktai_Location_itsumoNavi class
 * example: http://mobile.its-mo.com/MapToLink/p2?pos=N35.39.19.183E139.44.55.335&geo=tokyo&x-acr=3
   ================================================== */

class Ktai_Location_itsumoNavi extends Ktai_Location_Info {

//static
function factory($url) {
	if (! preg_match('#^(\[GPS.*?\])?\s*http://mobile\.its-mo\.com/MapToLink/p2\?(\S+)#', $url, $m)) {
		return NULL;
	}
	$loc = new Ktai_Location_itsumoNavi;
	$loc->url = $m[0];
	$params = parent::get_params($m[2]);
	$loc->geometory = @$params['geo'];
	$loc->accuracy  = @$params['x-acr'];
	preg_match('/^([NS])([.0-9]+)([EW])([.0-9]+)$/', @$params['pos'], $n);
	$loc->lat = ($n[1] == 'S' ? -1: 1) * parent::dms2deg($n[2], 90);
	$loc->lon = ($n[3] == 'W' ? -1: 1) * parent::dms2deg($n[4], 180);
	if ($loc->geometory == 'tokyo') {
		$loc->tokyo2wgs84();
	}
	if (! $loc->check_values()) {
		return NULL;
	}
	return $loc;
}

// ===== End of class ====================
}

/* ==================================================
 *   Ktai_Location_iMapFan class
 * example: http://i.mapfan.com/m.cgi?uid=NULLGWDOCOMO&F=AP&M=E139.34.4.3N35.36.50.6&SC=SY3JY8GE&AR=07900
   ================================================== */

class Ktai_Location_iMapFan extends Ktai_Location_Info {

//static
function factory($url) {
	if (! preg_match('#^http://i\.mapfan\.com/m\.cgi\?uid=NULLGWDOCOMO&(\S+)#m', $url, $m)) {
		return NULL;
	}
	$loc = new Ktai_Location_iMapFan;
	$loc->url = $m[0];
	$params = parent::get_params($m[1]);
	preg_match('/^([EW])([.0-9]+)([NS])([.0-9]+)$/', @$params['M'], $n);
	$loc->lon = ($n[1] == 'W' ? -1: 1) * parent::dms2deg($n[2], 180);
	$loc->lat = ($n[3] == 'S' ? -1: 1) * parent::dms2deg($n[4], 90);
	$loc->tokyo2wgs84();
	if (! $loc->check_values()) {
		return NULL;
	}
	return $loc;
}

// ===== End of class ====================
}

/* ==================================================
 *   Ktai_Location_iMapion class
 * example: http://i.mapion.co.jp/c/f?uc=1&nl=35/36/02.038&el=139/30/38.558&grp=mall&scl=625000&R=1&uid=NULLGWDOCOMO
   ================================================== */

class Ktai_Location_iMapion extends Ktai_Location_Info {

//static
function factory($url) {
	if (! preg_match('#^http://i\.mapion\.co\.jp/c/f\?(\S+)#m', $url, $m)) {
		return NULL;
	}
	$loc = new Ktai_Location_iMapion;
	$loc->url = $m[0];
	$params = parent::get_params($m[1]);
	$lat = str_replace('/', '.', @$params['nl']);
	$lon = str_replace('/', '.', @$params['el']);
	$loc->lat = parent::dms2deg($lat, 90);
	$loc->lon = parent::dms2deg($lon, 180);
	$loc->tokyo2wgs84();
	if (! $loc->check_values()) {
		return NULL;
	}
	return $loc;
}

// ===== End of class ====================
}

/* ==================================================
 *   Ktai_Location_iZenrin class
 * example: http://i.i.zenrin.co.jp/MapToLink/p1?scd=00300&rl=%2fzi%2fmenu%2far1%3farea%3d07900&x=6&n=35.616297&e=139.565364&uid=NULLGWDOCOMO
   ================================================== */

class Ktai_Location_iZenrin extends Ktai_Location_Info {

//static
function factory($url) {
	if (! preg_match('#^http://i\.i\.zenrin\.co\.jp/MapToLink/p1\?(\S+)#m', $url, $m)) {
		return NULL;
	}
	$loc = new Ktai_Location_iZenrin;
	$loc->url = $m[0];
	$params = parent::get_params($m[1]);
	$loc->lat = floatval(@$params['n']);
	$loc->lon = floatval(@$params['e']);
	$loc->tokyo2wgs84();
	if (! $loc->check_values()) {
		return NULL;
	}
	return $loc;
}

// ===== End of class ====================
}

/* ==================================================
 *   Ktai_Location_EXIF class
 * Document: http://www.kanzaki.com/docs/sw/geoinfo.html#geo-exif
   ================================================== */

class Ktai_Location_EXIF extends Ktai_Location_Info {

//static
function factory($content) {
	if (! preg_match_all('/(<a [^>]*href="(.*?)"[^>]*>)?<img [^>]*src="(.*?)"/', $content, $matches, PREG_SET_ORDER)) {
		return NULL;
	}
	$locations = array();
	foreach ($matches as $m) {
		$file = NULL;
		if ($m[1]) {
			$file = Ktai_Location_EXIF::decide_file_path($m[2]);
		}
		if (! $file) {
			$file = Ktai_Location_EXIF::decide_file_path($m[3]);
		}
		if ($file) {
			$exif = exif_read_data($file, 0, TRUE);
			if ($exif && array_key_exists('GPS', $exif)) {
				$loc = new Ktai_Location_EXIF;
				$loc->lat = $loc->decode_exif_location($exif['GPS']['GPSLatitude'], $exif['GPS']['GPSLatitudeRef']);
				$loc->lon = $loc->decode_exif_location($exif['GPS']['GPSLongitude'], $exif['GPS']['GPSLongitudeRef']);
				if (isset($exif['GPS']['GPSAltitude'])) {
					$loc->alt = (@$exif['GPS']['GPSAltitudeRef'] == '1' ? -1 : 1) * $exif['GPS']['GPSAltitude'];
				}
				$loc->geometory = @$exif['GPS']['GPSMapDatum'];
				if ($loc->lat && $loc->lon && ($loc->geometory == 'tokyo' || $loc->geometory == 'TOKYO')) {
					$loc->tokyo2wgs84();
				}
	            if ($loc->lat && $loc->lon) {
					$locations[] = $loc;
				};
			}
		}
	}
	return $locations;
}

/* ==================================================
 * Decide the filepath from html img src attributes.
 * @param	String $src
 * @return	String $path
*/

//private
function decide_file_path($src) {
	foreach (array(get_bloginfo('url'), get_option('siteurl')) as $url) {
		if (! $url) {
			continue;
		}
		$path = ABSPATH . preg_replace('#^' . quotemeta($url) . '/?#', '', $src, 1);
		if (file_exists($path)) {
			return $path;
		}
		$short_url = preg_replace('#^https?://[^/]*/?#', '/', $url, 1);
		$path = ABSPATH . preg_replace('#^' . quotemeta($short_url) . '/?#', '', $src, 1);
		if (file_exists($path)) {
			return $path;
		}
	}
	$path = ABSPATH . preg_replace('#^https?://[^/]*/#', '', $src, 1);
	if (file_exists($path)) {
		return $path;
	}
	$path = ABSPATH . $src;
	if (file_exists($path)) {
		return $path;
	}
	return NULL; // failed to decide file path...
}

/* ==================================================
 * Decode a location value of EXIF format.
 * @param	Array  $dms
 * @param   String $ref
 * @return	String $degree
*/

//private
function decode_exif_location($dms, $ref) {
	if (count($dms) != 3) {
		return NULL;
	}
    $deg = array_map('intval', explode('/', $dms[0]));
    $min = array_map('intval', explode('/', $dms[1]));
    $sec = array_map('intval', explode('/', $dms[2]));
	return ($ref == 'S' || $ref == 'W' ? -1 : 1) *
	  ( ($deg[1] != 0 ? $deg[0] / $deg[1] : 0)
	  + ($min[1] != 0 ? $min[0] / ($min[1] * 60) : 0)
	  + ($sec[1] != 0 ? $sec[0] / ($sec[1] * 3600) : 0));
}

// ===== End of class ====================
}

/* ==================================================
 *   Ktai_Location_Geocoding class
 * API: http://www.google.com/apis/maps/documentation/#Geocoding_HTTP_Request
   ================================================== */

class Ktai_Location_Geocoding extends Ktai_Location_Info {

//static 
function factory($content) {
	if (! preg_match_all('#\[geo\](.*?)\[/geo\]#', $content, $matches, PREG_SET_ORDER) 
	||  ! ini_get('allow_url_fopen')) {
		return NULL;
	}
	$api_key = get_option('googlemaps_api_key');
	if (! $api_key) {
		if (! defined('GOOGLEMAPS_API_KEY')) {
			return NULL;
		} else {
			$api_key = GOOGLEMAPS_API_KEY;
		}
	}
	$locations = array();
	$charset = get_option('blog_charset');
	$format = function_exists('simplexml_load_string') ? 'xml' : 'csv';
	foreach ($matches as $m) {
		$loc = new Ktai_Location_Geocoding;
		$loc->geotag = $m[0];
		$loc->place  = $m[1];
		$place_utf8  = ($charset != 'UTF-8') ? mb_convert_encoding($m[1], 'UTF-8', $charset) : $m[1];
		$response = @file_get_contents('http://maps.google.com/maps/geo?q=' . urlencode($place_utf8) . "&key=$api_key&output=$format");
		if (! $response) {
			$loc->place .= '(Geocoding no response) ';
		} elseif ($format == 'xml') {
			$geoxml = simplexml_load_string($response);
			$status = $geoxml->Response->Status->code;
			if ($status != 200) {
				$loc->place .= "(Geocoding error: $status) ";
			} else {
				$coords = explode(',', $geoxml->Response->Placemark->Point->coordinates);
				$loc->lat    = @$coords[1];
				$loc->lon    = @$coords[0];
				$loc->alt    = @$coords[2];
			}
		} elseif ($format == 'csv') {
			list($status, $accuracy, $lat, $lon) = explode(',', $response);
			if ($status != 200) {
				$loc->place .= "(Geocoding error: $status) ";
			} else {
				$loc->lat = @$lat;
				$loc->lon = @$lon;
			}
		} else {	// Invalid format
			continue;
		}
		$locations[] = $loc;
	}
	return $locations;
}

// ===== End of class ====================
}

?>