/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.typeconvert;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.Format;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.regex.Pattern;

import woolpack.bool.BoolUtils;
import woolpack.convert.ConvertUtils;
import woolpack.fn.Fn;
import woolpack.fn.FnUtils;
import woolpack.utils.Utils;

/**
 * 型変換のユーティリティです。
 * 型推論で表記を簡略するためのスタティックメソッドと変数を含みます。
 * 
 * @author nakamura
 * 
 */
public final class TypeConvertUtils {
	/**
	 * {@link ConvertContext#getToType()}を返す関数です。
	 */
	public static final Fn<ConvertContext, Class, RuntimeException> GET_TO_TYPE = new Fn<ConvertContext, Class, RuntimeException>() {
		public Class exec(final ConvertContext c) {
			return c.getToType();
		}
	};

	/**
	 * {@link ConvertContext#getValue()}の{@link Object#getClass()}を返す関数です。
	 */
	public static final Fn<ConvertContext, Class, RuntimeException> GET_FROM_TYPE = new Fn<ConvertContext, Class, RuntimeException>() {
		public Class exec(final ConvertContext c) {
			return c.getValue().getClass();
		}
	};
	
	/**
	 * {@link ConvertContext#getPropertyName()}を返す関数です。
	 */
	public static final Fn<ConvertContext, String, RuntimeException> GET_PROPERTY_NAME = new PropertyNameGetter();
	
	/**
	 * デフォルトの日付フォーマッタです。
	 * 以下のパターンを成功するまで順に試行します。
	 * <ol>
	 * <li>プロパティ名が正規表現「^.*date_?time$」に合致する場合はyyyyMMddHHmmss
	 * <li>プロパティ名が正規表現「^.*time$」に合致する場合はHHmmss
	 * <li>上記以外の場合はyyyyMMdd
	 * </ol>
	 * <br/>適用しているデザインパターン：{@link Fn}のCompositeを生成するBuilder。
	 */
	public static final Fn<ConvertContext, Void, RuntimeException> DATE_FORMATTER =
		FnUtils.exec(FnUtils.join(GET_PROPERTY_NAME, FnUtils.ifElse(Utils
			.linkedMap(
				BoolUtils.ifTrue(BoolUtils.not(BoolUtils.IS_NULL),
					BoolUtils.checkRegExp(Pattern.compile("^.*date_?time$", Pattern.CASE_INSENSITIVE))),
				format(new SimpleDateFormat("yyyyMMddHHmmss"))
			)
			.map(
				BoolUtils.ifTrue(BoolUtils.not(BoolUtils.IS_NULL),
						BoolUtils.checkRegExp(Pattern.compile("^.*time$", Pattern.CASE_INSENSITIVE))),
				format(new SimpleDateFormat("HHmmss"))
			)
			, FnUtils.fix(format(new SimpleDateFormat("yyyyMMdd"))))));
	
	/**
	 * デフォルトの日付パーサです。以下のパターンを成功するまで順に試行します。
	 * <ol>
	 * <li>yyyy-MM-dd
	 * <li>yyyy-MM-dd HH:mm:ss
	 * <li>yyyy-MM-dd HH:mm:ss z
	 * <li>yyyyMMddHHmmss
	 * <li>yyyyMMdd
	 * <li>yy/MM/dd(DateFormat.getDateInstance(SHORT))
	 * <li>yyyy/MM/dd(DateFormat.getDateInstance(MEDIUM))
	 * <li>yyyy/MM/dd HH:mm:ss(DateFormat.getDateTimeInstance(MEDIUM, MEDIUM))
	 * <li>HH:mm:ss(DateFormat.getTimeInstance(MEDIUM))
	 * <li>yyyy/MM/dd(DateFormat.getDateInstance(LONG))
	 * <li>yyyy/MM/dd HH:mm:ss z(DateFormat.getDateTimeInstance(LONG, LONG))
	 * <li>HH:mm:ss z(DateFormat.getTimeInstance(LONG))
	 * <li>yy/MM/dd HH:mm(DateFormat.getDateTimeInstance(SHORT, SHORT))
	 * <li>HH:mm(DateFormat.getTimeInstance(SHORT))
	 * <li>yyyy年MM月dd日(DateFormat.getDateInstance(FULL))
	 * <li>yyyy年MM月dd日 HH時mm分ss秒 z(DateFormat.getDateTimeInstance(FULL, FULL))
	 * <li>HH時mm分ss秒 z(DateFormat.getTimeInstance(FULL))
	 * </ol>
	 * <br/>適用しているデザインパターン：{@link Fn}のCompositeを生成するBuilder。
	 */
	public static final Fn<ConvertContext, Void, RuntimeException> DATE_PARSER =
		new Converter(FnUtils.join(
				ConvertUtils.TO_STRING,
				ConvertUtils.trys(
					Utils
					.list(parse(new SimpleDateFormat("yyyy-MM-dd")))
					.list(parse(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss")))
					.list(parse(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss z")))
					.list(parse(new SimpleDateFormat("yyyyMMddHHmmss")))
					.list(parse(new SimpleDateFormat("yyyyMMdd")))
					.list(parse(DateFormat.getDateInstance(DateFormat.SHORT)))
					.list(parse(DateFormat.getDateInstance(DateFormat.MEDIUM)))
					.list(parse(DateFormat.getDateTimeInstance(DateFormat.MEDIUM, DateFormat.MEDIUM)))
					.list(parse(DateFormat.getTimeInstance(DateFormat.MEDIUM)))
					.list(parse(DateFormat.getDateInstance(DateFormat.LONG)))
					.list(parse(DateFormat.getDateTimeInstance(DateFormat.LONG, DateFormat.LONG)))
					.list(parse(DateFormat.getTimeInstance(DateFormat.LONG)))
					.list(parse(DateFormat.getDateTimeInstance(DateFormat.SHORT, DateFormat.SHORT)))
					.list(parse(DateFormat.getTimeInstance(DateFormat.SHORT)))
					.list(parse(DateFormat.getDateInstance(DateFormat.FULL)))
					.list(parse(DateFormat.getDateTimeInstance(DateFormat.FULL, DateFormat.FULL)))
					.list(parse(DateFormat.getTimeInstance(DateFormat.FULL)))
					, FnUtils.<Exception, Void, IllegalStateException>fixThrows(null),
					new Fn<Exception, Void, RuntimeException>() {
						public Void exec(final Exception c) {
							throw new IllegalArgumentException(c);
						}
					}
				)));
	
	/**
	 * デフォルトの数値フォーマッタです。{@link DecimalFormat}を引数なしコンストラクタで使用します。
	 */
	public static final Fn<ConvertContext, Void, RuntimeException> NUMBER_FORMATTER =
		new Converter(ConvertUtils.format(ConvertUtils.formatFactory(new DecimalFormat())));
	
	/**
	 * デフォルトの数値パーサです。{@link DecimalFormat}を引数なしコンストラクタで使用します。
	 */
	public static final Fn<ConvertContext, Void, RuntimeException> NUMBER_PARSER =
		new Converter(FnUtils.join(
			ConvertUtils.TO_STRING,
			ConvertUtils.trys(
				Utils
				.list(parse(new DecimalFormat())),
				FnUtils.<Exception, Void, IllegalStateException>fixThrows(null),
				new Fn<Exception, Void, RuntimeException>() {
					public Void exec(final Exception c) {
						throw new IllegalArgumentException(c);
					}
				}
			)));
	
	/**
	 * デフォルト単純型変換の関数です。
	 */
	public static final Fn<ConvertContext, Void, RuntimeException> SIMPLE_CONVERTER = getSimpleConverter(
			DATE_FORMATTER,
			DATE_PARSER,
			NUMBER_FORMATTER,
			NUMBER_PARSER,
			BoolUtils.TO_BOOLEAN_VIEW,
			FnUtils.<ConvertContext, Void>fix(null));

	/**
	 * デフォルトの一覧型変換の関数です。
	 */
	public static final Fn<ConvertContext, Void, RuntimeException> COLLECTION_CONVERTER = getCollectionConverter(
			DATE_FORMATTER,
			DATE_PARSER,
			NUMBER_FORMATTER,
			NUMBER_PARSER,
			BoolUtils.TO_BOOLEAN_VIEW);
	
	private TypeConvertUtils() {
	}
	
	private static Fn<String, Object, Exception> parse(final Format f) {
		return ConvertUtils.parse(ConvertUtils.formatFactory(f));
	}
	
	private static Fn<ConvertContext, Void, RuntimeException> format(final Format f) {
		return new Converter(ConvertUtils.format(ConvertUtils.formatFactory(f)));
	}
	
	private static Fn<ConvertContext, Void, RuntimeException> getToDateAndAfter(
			final Fn<ConvertContext, Void, RuntimeException> successorFn,
			final Fn<Object, ?, RuntimeException> afterConverter) {
		return new Fn<ConvertContext, Void, RuntimeException>() {
			public Void exec(final ConvertContext c) throws RuntimeException {
				final Object before = c.getValue();
				final Object mid;
				if (before instanceof java.util.Date) {
					mid = before;
				} else if (before instanceof Calendar) {
					mid = afterConverter.exec(new java.util.Date(((Calendar) before).getTimeInMillis()));
				} else if (before instanceof Long) {
					mid = new java.util.Date((Long) before);
				} else {
					successorFn.exec(c);
					mid = c.getValue();
				}
				c.setValue(afterConverter.exec(mid));
				return null;
			}
		};
	}
	
	private static void riseCanntConvertDateToNumberException() {
		throw new IllegalArgumentException("cannot convert Date to Number which is not Long.");
	}

	private static Fn<ConvertContext, Void, RuntimeException> getToNumberAndAfter(
			final Fn<ConvertContext, Void, RuntimeException> successorFn,
			final Fn<Object, ?, RuntimeException> afterConverter) {
		return new Fn<ConvertContext, Void, RuntimeException>() {
			public Void exec(final ConvertContext c) throws RuntimeException {
				final Object before = c.getValue();
				final Object mid;
				if (before instanceof Number) {
					mid = before;
				} else if (before instanceof Boolean) {
					mid = ((Boolean) before) ? 1 : 0;
				} else if (before instanceof java.util.Date) {
					if (!Long.class.equals(c.getToType())) {
						riseCanntConvertDateToNumberException();
					}
					mid = ((java.util.Date) before).getTime();
				} else if (before instanceof Calendar) {
					if (!Long.class.equals(c.getToType())) {
						riseCanntConvertDateToNumberException();
					}
					mid = ((Calendar) before).getTimeInMillis();
				} else {
					successorFn.exec(c);
					mid = c.getValue();
				}
				c.setValue(afterConverter.exec(mid));
				return null;
			}
		};
	}
	
	private static Fn<ConvertContext, Void, RuntimeException> getSimpleConverterPrivate(
			final Fn<ConvertContext, Void, RuntimeException> dateFormatter,
			final Fn<ConvertContext, Void, RuntimeException> dateParser,
			final Fn<ConvertContext, Void, RuntimeException> numberFormatter,
			final Fn<ConvertContext, Void, RuntimeException> numberParser,
			final Fn<Object, Boolean, RuntimeException> booleanConverter,
			final Fn<ConvertContext, Void, RuntimeException> successorFn) {
		return FnUtils.exec(FnUtils.join(GET_TO_TYPE, FnUtils.switching(Utils.<Class, Fn<ConvertContext, Void, RuntimeException>>
			map(String.class, new Fn<ConvertContext, Void, RuntimeException>() {
				private final Fn<ConvertContext, Void, RuntimeException> defaultConverter = new Converter(ConvertUtils.TO_STRING);
				public Void exec(final ConvertContext c) {
					if (c.getValue() instanceof java.util.Date) {
						dateFormatter.exec(c);
					} else if (c.getValue() instanceof Number) {
						numberFormatter.exec(c);
					} else {
						defaultConverter.exec(c);
					}
					return null;
				}
			})
			.map(Character.class, new Converter(new Fn<Object, Character, RuntimeException>() {
				public Character exec(final Object c) {
					final String s = c.toString();
					if (s.length() == 0) {
						throw new IllegalArgumentException();
					}
					return s.charAt(0);
				}
			}))
			.map(Boolean.class, new Converter(booleanConverter))
			.map(java.util.Date.class, getToDateAndAfter(dateParser, FnUtils.echo()))
			.map(Calendar.class, getToDateAndAfter(dateParser, ConvertUtils.TO_CALENDAR))
			.map(java.sql.Date.class, getToDateAndAfter(dateParser, ConvertUtils.TO_SQL_DATE))
			.map(java.sql.Time.class, getToDateAndAfter(dateParser, ConvertUtils.TO_TIME))
			.map(java.sql.Timestamp.class, getToDateAndAfter(dateParser, ConvertUtils.TO_TIMESTAMP))
			.map(Byte.class, getToNumberAndAfter(numberParser, ConvertUtils.TO_BYTE))
			.map(Short.class, getToNumberAndAfter(numberParser, ConvertUtils.TO_SHORT))
			.map(Integer.class, getToNumberAndAfter(numberParser, ConvertUtils.TO_INTEGER))
			.map(Long.class, getToNumberAndAfter(numberParser, ConvertUtils.TO_LONG))
			.map(Float.class, getToNumberAndAfter(numberParser, ConvertUtils.TO_FLOAT))
			.map(Double.class, getToNumberAndAfter(numberParser, ConvertUtils.TO_DOUBLE))
			.map(BigInteger.class, getToNumberAndAfter(numberParser, ConvertUtils.TO_BIG_INTEGER))
			.map(BigDecimal.class, getToNumberAndAfter(numberParser, ConvertUtils.TO_BIG_DECIMAL))
			, successorFn)));
	}

	/**
	 * 単純型変換の関数を生成します。
	 * <br/>適用しているデザインパターン：{@link Fn}のCompositeを生成するBuilder。
	 * @param dateFormatter 日付フォーマッタ。
	 * @param dateParser 日付パーサ。
	 * @param numberFormatter 数値フォーマッタ。
	 * @param numberParser 数値パーサ。
	 * @param booleanConverter 真偽値へのコンバータ。
	 * @param successorFn この変換器で変換できない場合の委譲先。
	 * @return 関数。
	 */
	public static Fn<ConvertContext, Void, RuntimeException> getSimpleConverter(
			final Fn<ConvertContext, Void, RuntimeException> dateFormatter,
			final Fn<ConvertContext, Void, RuntimeException> dateParser,
			final Fn<ConvertContext, Void, RuntimeException> numberFormatter,
			final Fn<ConvertContext, Void, RuntimeException> numberParser,
			final Fn<Object, Boolean, RuntimeException> booleanConverter,
			final Fn<ConvertContext, Void, RuntimeException> successorFn) {
		return
		new ToPrimitiveConverter(
		new DelegationIfNecessityConverter(
		getSimpleConverterPrivate(
				dateFormatter,
				dateParser,
				numberFormatter,
				numberParser,
				booleanConverter,
				successorFn)));
	}

	/**
	 * 一覧型変換の関数を生成します。
	 * <br/>適用しているデザインパターン：{@link Fn}のCompositeを生成するBuilder。
	 * @param dateFormatter 日付フォーマッタ。
	 * @param dateParser 日付パーサ。
	 * @param numberFormatter 数値フォーマッタ。
	 * @param numberParser 数値パーサ。
	 * @param booleanConverter 真偽値へのコンバータ。
	 * @return 関数。
	 */
	public static Fn<ConvertContext, Void, RuntimeException> getCollectionConverter(
			final Fn<ConvertContext, Void, RuntimeException> dateFormatter,
			final Fn<ConvertContext, Void, RuntimeException> dateParser,
			final Fn<ConvertContext, Void, RuntimeException> numberFormatter,
			final Fn<ConvertContext, Void, RuntimeException> numberParser,
			final Fn<Object, Boolean, RuntimeException> booleanConverter) {
		return 
		new SettingFnConverter(
		new ToPrimitiveConverter(
		new RuntimeExceptionToNullConverter(
		new DelegationIfNecessityConverter(
		new ToArrayConverter(
		new ToCollectionViewConverter(
		new ToCollectionDecompositionConverter(
		new ToMapViewConverter(
		getSimpleConverterPrivate(
				dateFormatter,
				dateParser,
				numberFormatter,
				numberParser,
				booleanConverter,
				new ToBeanConverter())))))))));
	}
	
	/**
	 * BeanをMapに変換する関数を生成します。
	 * @return 関数。
	 */
	public static Fn<Object, BeanMap, RuntimeException> toMap() {
		return new Fn<Object, BeanMap, RuntimeException>() {
			public BeanMap exec(final Object c) {
				return new BeanMap(c);
			}
		};
	}
	
	/**
	 * BeanをMapに変換する関数を生成します。
	 * @param fn 型を変換する関数。
	 * @return 関数。
	 */
	public static Fn<Object, BeanMap, RuntimeException> toMap(
			final Fn<ConvertContext, Void, RuntimeException> fn) {
		return new Fn<Object, BeanMap, RuntimeException>() {
			public BeanMap exec(final Object c) {
				return new BeanMap(c, fn);
			}
		};
	}
}
