/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.el;

import java.util.ArrayList;
import java.util.Collection;
import java.util.StringTokenizer;

/**
 * ドット区切りのプロパティ名の順にたどっていく式言語です。
 * <br/>適用しているデザインパターン：内部で{@link EL}のCompositeを生成して使用するBuilder。
 * 
 * @author nakamura
 * 
 */
public class PathEL extends AbstractEL {
	private String expression;
	private PropertyELFactory factory;
	private EL el;
	private boolean notFoundIgnoreFlag;

	/**
	 * @param expression プロパティ名をドットでつなげた表現。
	 * @param factory 型を変換する関数。
	 * @param notFoundIgnoreFlag メンバが存在しない場合に何もしない場合はtrue。
	 */
	public PathEL(
			final String expression,
			final PropertyELFactory factory,
			final boolean notFoundIgnoreFlag) {
		super();
		this.expression = expression;
		this.factory = factory;
		this.notFoundIgnoreFlag = notFoundIgnoreFlag;
		generate();
	}
	
	private void generate() {
		final StringTokenizer st = new StringTokenizer(expression, ".");
		final Collection<EL> c = new ArrayList<EL>();
		while (st.hasMoreTokens()) {
			final EL tmpEL = new PropertyEL(st.nextToken(), factory);
			if (notFoundIgnoreFlag) {
				c.add(new MaybeEL(tmpEL));
			} else {
				c.add(tmpEL);
			}
		}
		el = new ArrayPathEL(c);
	}

	/**
	 * @param expression プロパティ名をドットでつなげた表現。
	 * @param notFoundIgnoreFlag メンバが存在しない場合に何もしない場合はtrue。
	 */
	public PathEL(
			final String expression,
			final boolean notFoundIgnoreFlag) {
		this(expression, ELUtils.getPropertyELFactory(notFoundIgnoreFlag), notFoundIgnoreFlag);
	}
	
	/**
	 * メンバが存在しない場合に何もしないように処理します。
	 * @param expression プロパティ名をドットでつなげた表現。
	 */
	public PathEL(final String expression) {
		this(expression, true);
	}

	@Override
	public Object getValue(final Object root, final Class clazz) {
		return el.getValue(root, clazz);
	}

	@Override
	public boolean setValue(final Object root, final Object value) {
		return el.setValue(root, value);
	}
	
	public String getExpression() {
		return expression;
	}
	public void setExpression(final String expression) {
		this.expression = expression;
		generate();
	}
	public PropertyELFactory getFactory() {
		return factory;
	}
	public void setFactory(final PropertyELFactory factory) {
		this.factory = factory;
		generate();
	}
	public boolean isNotFoundIgnoreFlag() {
		return notFoundIgnoreFlag;
	}
	public void setNotFoundIgnoreFlag(final boolean notFoundIgnoreFlag) {
		this.notFoundIgnoreFlag = notFoundIgnoreFlag;
		generate();
	}
}
