/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.el;

import woolpack.fn.Fn;
import woolpack.typeconvert.ConvertContext;
import woolpack.typeconvert.TypeConvertUtils;
import woolpack.utils.fn.UtilsFnUtils;

/**
 * 式言語のユーティリティです。
 * 
 * @author nakamura
 * 
 */
public final class ELUtils {

	/**
	 * ヌルオブジェクト役の式言語です。
	 * <br/>適用しているデザインパターン：Null Object。
	 */
	public static final EL NULL = new AbstractEL() {
		@Override
		public Object getValue(final Object root, final Class toType) {
			return null;
		}

		@Override
		public boolean setValue(final Object root, final Object value) {
			return false;
		}
	};
	
	/**
	 * 引数自身を表すプロパティ式言語のデフォルト値です。
	 */
	public static final EL THIS = new ThisEL(TypeConvertUtils.COLLECTION_CONVERTER, "");

	/**
	 * プロパティ式言語ファクトリのデフォルト値です。
	 */
	public static final PropertyELFactory PROPERTY_EL_FACTORY = getPropertyELFactory(true);

	/**
	 * プロパティ名にアクセスする式言語を生成する関数です。
	 * <br/>適用しているデザインパターン：{@link EL}のAbstract Factory。
	 */
	public static final Fn<String, EL, RuntimeException> EL_FACTORY = new Fn<String, EL, RuntimeException>() {
		public EL exec(final String c) {
			return new PathEL(c);
		}
	};

	private ELUtils() {
	}

	/**
	 * プロパティ式言語ファクトリを生成します。
	 * <br/>適用しているデザインパターン：{@link PropertyELFactory}のComposite。
	 * @param fn 型を変換する関数。
	 * @param guessFactory プロパティ名候補のファクトリ。
	 * @param notFoundIgnoreFlag メンバが存在しない場合に何もしない場合はtrue。
	 * @return プロパティ式言語ファクトリ。
	 */
	public static PropertyELFactory getPropertyELFactory(
			final Fn<ConvertContext, Void, ? extends RuntimeException> fn,
			final Fn<String, Iterable<String>, ? extends RuntimeException> guessFactory,
			final boolean notFoundIgnoreFlag) {
		return new MapSimilarPropertyELFactory(
			fn,
			new MaybePropertyELFactory(
				notFoundIgnoreFlag,
				new CachePropertyELFactory(
					new SimilarPropertyELFactory(
						new ReflectionPropertyELFactory(fn),
						guessFactory
					)
				)
			),
			guessFactory);
	}

	/**
	 * プロパティ式言語ファクトリを生成します。
	 * <br/>適用しているデザインパターン：{@link PropertyELFactory}のComposite。
	 * @param notFoundIgnoreFlag メンバが存在しない場合に何もしない場合はtrue。
	 * @return プロパティ式言語ファクトリ。
	 */
	public static PropertyELFactory getPropertyELFactory(
			final boolean notFoundIgnoreFlag) {
		return getPropertyELFactory(
				TypeConvertUtils.COLLECTION_CONVERTER,
				UtilsFnUtils.PROPERTY_GUESSING_FACTORY,
				notFoundIgnoreFlag);
	}
}
