/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.bool;

import java.util.Collection;
import java.util.regex.Pattern;

import woolpack.fn.Fn;
import woolpack.fn.FnUtils;

/**
 * 真偽値処理のユーティリティです。
 * 
 * @author nakamura
 *
 */
public final class BoolUtils {
	/**
	 * 引数が{@link Throwable}のサブクラスでないことを評価する関数です。
	 */
	public static final Fn<Object, Boolean, RuntimeException> NOT_THROWABLE = not(matchObject(Throwable.class));
	
	/**
	 * コンテキストを否定する関数です。
	 */
	public static final Fn<Boolean, Boolean, RuntimeException> NOT = new Fn<Boolean, Boolean, RuntimeException>() {
		public Boolean exec(final Boolean c) {
			return !c;
		}
	};
	
	/**
	 * null でない、かつ文字列の場合は空でないことをチェックする関数です。
	 * このクラスは LSP(The Liskov Substitution Principle) を満たしません。
	 */
	public static final Fn<Object, Boolean, RuntimeException> NOT_EMPTY = new Fn<Object, Boolean, RuntimeException>() {
		public Boolean exec(final Object c) {
			return c != null && (!(c instanceof String) || ((String) c).length() > 0);
		}
	};

	/**
	 * null であることをチェックする関数です。
	 */
	public static final Fn<Object, Boolean, RuntimeException> IS_NULL = new Fn<Object, Boolean, RuntimeException>() {
		public Boolean exec(final Object c) {
			return c == null;
		}
	};
	
	/**
	 * 引数が null または{@link Boolean#FALSE}の場合のみ
	 * {@link Boolean#FALSE}を返却する関数です。
	 */
	public static final Fn<Object, Boolean, RuntimeException> TO_BOOLEAN = new Fn<Object, Boolean, RuntimeException>() {
		public Boolean exec(final Object c) {
			return Boolean.valueOf(c != null && !Boolean.FALSE.equals(c));
		}
	};

	/**
	 * 引数が null または{@link Boolean#FALSE} または文字列"false"(大小を区別しない)の場合のみ
	 * {@link Boolean#FALSE}を返却する関数です。
	 */
	public static final Fn<Object, Boolean, RuntimeException> TO_BOOLEAN_VIEW = new Fn<Object, Boolean, RuntimeException>() {
		public Boolean exec(final Object c) {
			return c != null && !Boolean.FALSE.equals(c) && !"false".equalsIgnoreCase(c.toString());
		}
	};
	
	/**
	 * 論理AND演算子のファクトリです。
	 * <br/>適用しているデザインパターン：{@link BooleanState}のAbstract Factory。
	 */
	public static final Fn<Object, BooleanState, RuntimeException> ANDAND = new Fn<Object, BooleanState, RuntimeException>() {
		public BooleanState exec(final Object c) {
			return new BooleanState(true) {
				@Override
				public void in(final boolean flag) {
					final boolean newFlag = isCurrent() && flag;
					setCurrent(newFlag);
					setStopped(!newFlag);
				}
			};
		}
	};
	
	/**
	 * ビットAND演算子のファクトリです。
	 * <br/>適用しているデザインパターン：{@link BooleanState}のAbstract Factory。
	 */
	public static final Fn<Object, BooleanState, RuntimeException> AND = new Fn<Object, BooleanState, RuntimeException>() {
		public BooleanState exec(final Object c) {
			return new BooleanState(true) {
				@Override
				public void in(final boolean flag) {
					setCurrent(isCurrent() && flag);
				}
			};
		}
	};
	
	/**
	 * 論理同値演算子のファクトリです。
	 * <br/>適用しているデザインパターン：{@link BooleanState}のAbstract Factory。
	 */
	public static final Fn<Object, BooleanState, RuntimeException> EQEQ = new Fn<Object, BooleanState, RuntimeException>() {
		public BooleanState exec(final Object c) {
			return new BooleanState(true) {
				private boolean trueFlag = true;
				private boolean falseFlag = true;
				@Override
				public void in(final boolean flag) {
					trueFlag &= flag;
					falseFlag &= !flag;
					final boolean newFlag = trueFlag || falseFlag;
					setCurrent(newFlag);
					setStopped(!newFlag);
				}
			};
		}
	};
	
	/**
	 * ビット同値演算子のファクトリです。
	 * <br/>適用しているデザインパターン：{@link BooleanState}のAbstract Factory。
	 */
	public static final Fn<Object, BooleanState, RuntimeException> EQ = new Fn<Object, BooleanState, RuntimeException>() {
		public BooleanState exec(final Object c) {
			return new BooleanState(true) {
				private boolean trueFlag = true;
				private boolean falseFlag = true;
				@Override
				public void in(final boolean flag) {
					trueFlag &= flag;
					falseFlag &= !flag;
					setCurrent(trueFlag || falseFlag);
				}
			};
		}
	};
	
	/**
	 * ビットOR演算子のファクトリです
	 * <br/>適用しているデザインパターン：{@link BooleanState}のAbstract Factory。
	 */
	public static final Fn<Object, BooleanState, RuntimeException> OR = new Fn<Object, BooleanState, RuntimeException>() {
		public BooleanState exec(final Object c) {
			return new BooleanState(false) {
				@Override
				public void in(final boolean flag) {
					setCurrent(isCurrent() || flag);
				}
			};
		}
	};
	
	/**
	 * 論理OR演算子のファクトリです。
	 * <br/>適用しているデザインパターン：{@link BooleanState}のAbstract Factory。
	 */
	public static final Fn<Object, BooleanState, RuntimeException> OROR = new Fn<Object, BooleanState, RuntimeException>() {
		public BooleanState exec(final Object c) {
			return new BooleanState(false) {
				@Override
				public void in(final boolean flag) {
					final boolean newFlag = isCurrent() || flag;
					setCurrent(newFlag);
					setStopped(newFlag);
				}
			};
		}
	};
	
	private BoolUtils() {
	}
	
	/**
	 * 委譲先を順次呼び出して集計する関数を生成します。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <C>
	 * @param <E>
	 * @param factory 真偽集計器のファクトリ。
	 * @param iterable 委譲先の一覧。
	 * @return 関数。
	 */
	public static <C, E extends Exception> Fn<C, Boolean, E> boolSeq(
			final Fn<Object, ? extends BooleanState, ? extends RuntimeException> factory,
			final Iterable<? extends Fn<? super C, Boolean, ? extends E>> iterable) {
		return new Fn<C, Boolean, E>() {
			public Boolean exec(final C c) throws E {
				final BooleanState state = factory.exec(null);
				for (final Fn<? super C, Boolean, ? extends E> e : iterable) {
					state.in(e.exec(c));
					if (state.isStopped()) {
						break;
					}
				}
				return state.isCurrent();
			}
		};
	}
	
	/**
	 * 比較する関数を生成します。
	 * <br/>適用しているデザインパターン：二項演算子のCurrying。
	 * @param <C>
	 * @param value 比較元({@link Comparable#compareTo(Object)}の引数でないほう)。
	 * @return 関数。
	 */
	public static <C extends Comparable<C>> Fn<C, Integer, RuntimeException> compare(final C value) {
		return new Fn<C, Integer, RuntimeException>() {
			public Integer exec(final C c) {
				return value.compareTo(c);
			}
		};
	}
	
	/**
	 * コレクションがすべて初期コレクションに含まれることをチェックする関数を生成します。
	 * <br/>適用しているデザインパターン：変換ルールと変換対象のCurrying。
	 * @param collection 初期コレクション。
	 * @return 関数。
	 */
	public static Fn<Collection<?>, Boolean, RuntimeException> containsAll(final Collection<?> collection) {
		return new Fn<Collection<?>, Boolean, RuntimeException>() {
			public Boolean exec(final Collection<?> c) {
				return collection.containsAll(c);
			}
		};
	}
	
	/**
	 * コンテキスト役が初期コレクションに含まれることをチェックする関数を生成します。
	 * <br/>適用しているデザインパターン：変換ルールと変換対象のCurrying。
	 * @param collection 初期コレクション。
	 * @return 関数。
	 */
	public static Fn<Object, Boolean, RuntimeException> contains(final Collection<?> collection) {
		return new Fn<Object, Boolean, RuntimeException>() {
			public Boolean exec(final Object c) {
				return collection.contains(c);
			}
		};
	}
	
	/**
	 * {@link Object#equals(Object)}で同値関係をチェックする関数を生成します。
	 * <br/>適用しているデザインパターン：二項演算子のCurrying。
	 * @param value 初期値。
	 * @return 関数。
	 */
	public static Fn<Object, Boolean, RuntimeException> checkEquals(final Object value) {
		return new Fn<Object, Boolean, RuntimeException>() {
			public Boolean exec(final Object c) {
				return value == null ? c == null : value.equals(c);
			}
		};
	}
	
	/**
	 * 最大値をチェックする関数を生成します。
	 * <br/>適用しているデザインパターン：二項演算子のCurrying。
	 * @param <C>
	 * @param value 最大値。
	 * @return 関数。
	 */
	public static <C extends Comparable<C>> Fn<C, Boolean, RuntimeException> checkMax(final C value) {
		return new Fn<C, Boolean, RuntimeException>() {
			public Boolean exec(final C c) {
				return value.compareTo(c) >= 0;
			}
		};
	}
	
	/**
	 * 文字列の最大文字数をチェックする関数を生成します。
	 * <br/>適用しているデザインパターン：二項演算子のCurrying。
	 * @param value 最大文字数。
	 * @return 関数。
	 */
	public static Fn<String, Boolean, RuntimeException> checkMaxLength(final int value) {
		return new Fn<String, Boolean, RuntimeException>() {
			public Boolean exec(final String c) {
				return c.length() <= value;
			}
		};
	}
	
	/**
	 * 最小値をチェックする関数を生成します。
	 * <br/>適用しているデザインパターン：二項演算子のCurrying。
	 * @param <C>
	 * @param value 最小値。
	 * @return 関数。
	 */
	public static <C extends Comparable<C>> Fn<C, Boolean, RuntimeException> checkMin(final C value) {
		return new Fn<C, Boolean, RuntimeException>() {
			public Boolean exec(final C c) {
				return value.compareTo(c) <= 0;
			}
		};
	}
	
	/**
	 * 文字列の最小文字数をチェックする関数を生成します。
	 * <br/>適用しているデザインパターン：二項演算子のCurrying。
	 * @param value 最小文字数。
	 * @return 関数。
	 */
	public static Fn<String, Boolean, RuntimeException> checkMinLength(final int value) {
		return new Fn<String, Boolean, RuntimeException>() {
			public Boolean exec(final String c) {
				return c.length() >= value;
			}
		};
	}
	
	/**
	 * 委譲先の結果の否定する関数を生成します。
	 * <br/>適用しているデザインパターン：返却値のDecorator。
	 * @param <C>
	 * @param <E>
	 * @param fn 委譲先。
	 * @return 関数。
	 */
	public static <C, E extends Exception> Fn<C, Boolean, E> not(final Fn<? super C, Boolean, ? extends E> fn) {
		return new Fn<C, Boolean, E>() {
			public Boolean exec(final C c) throws E {
				return !fn.exec(c);
			}
		};
	}
	
	/**
	 * プロパティ値により以下の判定を行う関数を生成します。
	 * null の場合は、コンテキストが null であることを検証します。
	 * {@link Class}クラスのインスタンスの場合は、
	 * コンテキストがそのクラスの変数に代入できることを検証します。
	 * 上記以外の場合は、値が等しいことを検証します。
	 * このクラスは LSP(The Liskov Substitution Principle) を満たしません。
	 * <br/>適用しているデザインパターン：変換ルールと変換対象のCurrying。
	 * @param value 初期値。
	 * @return 関数。
	 */
	public static Fn<Object, Boolean, RuntimeException> matchObject(final Object value) {
		return new Fn<Object, Boolean, RuntimeException>() {
			public Boolean exec(final Object c) {
				if (value == null) {
					return c == null;
				} else if (value instanceof Class) {
					return ((Class) value).isInstance(c);
				} else {
					return value.equals(c);
				}
			}
		};
	}
	
	/**
	 * 正規表現でチェックする関数を生成します。
	 * <br/>適用しているデザインパターン：変換ルールと変換対象のCurrying。
	 * @param pattern 正規表現。
	 * @return 関数。
	 */
	public static Fn<String, Boolean, RuntimeException> checkRegExp(final Pattern pattern) {
		return new Fn<String, Boolean, RuntimeException>() {
			public Boolean exec(final String c) {
				return pattern.matcher(c).matches();
			}
		};
	}
	
	/**
	 * 評価結果が{@link Boolean#TRUE}の場合に委譲するを生成します。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <C>
	 * @param <E>
	 * @param ifFn 評価の委譲先。
	 * @param trueFn 評価結果が{@link Boolean#TRUE}の場合の委譲先。
	 * @return 関数。
	 */
	public static <C, E extends Exception> Fn<C, Boolean, E> ifTrue(
			final Fn<? super C, ?, ? extends E> ifFn,
			final Fn<? super C, ? extends Boolean, ? extends E> trueFn) {
		return FnUtils.ifTrue(ifFn, trueFn, FnUtils.<C, Boolean, E>fixThrows(false));
	}
	
	/**
	 * 評価結果が{@link Boolean#TRUE}でない場合に委譲する{@link Fn}を返します。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <C>
	 * @param <E>
	 * @param ifFn 評価の委譲先。
	 * @param falseFn 評価結果が{@link Boolean#TRUE}でない場合の委譲先。
	 * @return 関数。
	 */
	public static <C, E extends Exception> Fn<C, Boolean, E> ifNot(
			final Fn<? super C, ?, ? extends E> ifFn,
			final Fn<? super C, ? extends Boolean, ? extends E> falseFn) {
		return FnUtils.ifTrue(ifFn, FnUtils.<C, Boolean, E>fixThrows(true), falseFn);
	}
}
