/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.typeconvert;

import java.util.Arrays;
import java.util.Collection;

import junit.framework.TestCase;
import woolpack.fn.Fn;
import woolpack.test.TestUtils;

public class CollectionConverterTest extends TestCase {
	private final Fn<ConvertContext, Void> c = new CollectionConverter(TypeConvertUtils.SIMPLE_CONVERTER);
	
	private void scenario(final Object expected, final Object before, final Class toType, final String name) {
		final ConvertContext context = new ConvertContext();
		context.setPropertyName(name);
		context.setToType(toType);
		context.setValue(before);
		c.exec(context);
		assertTrue(TestUtils.equals(expected, context.getValue()));
	}

	public void testNormal() {
		scenario("1", "1", Object.class, null);
		scenario(Integer.valueOf(1), "1", Integer.class, null);
		scenario(new int[] { 1 }, "1", int[].class, null);
		scenario(new Integer[] { Integer.valueOf(1) }, "1", Integer[].class, null);
		scenario(Arrays.asList("1"), "1", Collection.class, null);

		scenario(new String[] { "1" }, new String[] { "1" }, Object.class, null);
		scenario(Integer.valueOf(1), new String[] { "1" }, Integer.class, null);
		scenario(new int[] { 1 }, new String[] { "1" }, int[].class, null);
		scenario(new Integer[] { Integer.valueOf(1) }, new String[] { "1" }, Integer[].class, null);
		scenario(Arrays.asList("1"), new String[] { "1" }, Collection.class, null);

		scenario("1", Arrays.asList("1"), Object.class, null);
		scenario(Integer.valueOf(1), Arrays.asList("1"), Integer.class, null);
		scenario(new int[] { 1 }, Arrays.asList("1"), int[].class, null);
		scenario(new Integer[] { Integer.valueOf(1) }, Arrays.asList("1"), Integer[].class, null);
		scenario(Arrays.asList("1"), Arrays.asList("1"), Collection.class, null);

		scenario(null, null, Object.class, null);
		scenario(null, null, Integer.class, null);
		scenario(new int[] { 0 }, null, int[].class, null);
		scenario(new Integer[] { null }, null, Integer[].class, null);
		scenario(Arrays.asList((Object) null), null, Collection.class, null);

		scenario(new int[] { 1, 2 }, new String[] { "1", "2" }, int[].class, null);
		{
			final ConvertContext context = new ConvertContext();
			context.setPropertyName(null);
			context.setToType(int[].class);
			context.setValue(new String[]{"1", "a"});
			try {
				c.exec(context);
				fail();
			} catch(final RuntimeException e) {
			}
			assertTrue(TestUtils.equals(new String[]{"1", "a"}, context.getValue()));
		}
	}
}
