/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import woolpack.el.EL;
import woolpack.fn.Fn;
import woolpack.xml.NodeContext;

/**
 * 各属性値の出現回数(1回か2回以上)により処理を分岐する{@link Fn}です。
 * 
 * @author nakamura
 * 
 */
public class BranchPropertyCounter<C extends NodeContext> extends AbstractCountProperty<C> {
	private Fn<? super C, Void> firstFn;
	private Fn<? super C, Void> pluralFn;

	/**
	 * @param el プロパティの出現回数を保持する位置。
	 * @param attrNames 属性名の一覧。
	 * @param firstFn 最初の検索結果に対する委譲先。
	 * @param pluralFn 2番目以降の検索結果に対する委譲先。
	 */
	public BranchPropertyCounter(
			final EL el,
			final Iterable<String> attrNames,
			final Fn<? super C, Void> firstFn,
			final Fn<? super C, Void> pluralFn) {
		super(el, attrNames);
		this.firstFn = firstFn;
		this.pluralFn = pluralFn;
	}

	@Override
	public void interpret(final C context, final PropertyCounter counter) {
		if (counter.getCount() == 1) {
			firstFn.exec(context);
		} else {
			pluralFn.exec(context);
		}
	}

	public Fn<? super C, Void> getFirstFn() {
		return firstFn;
	}
	public void setFirstFn(final Fn<? super C, Void> firstFn) {
		this.firstFn = firstFn;
	}
	public Fn<? super C, Void> getPluralFn() {
		return pluralFn;
	}
	public void setPluralFn(final Fn<? super C, Void> pluralFn) {
		this.pluralFn = pluralFn;
	}
}
