/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.regex.Pattern;

import junit.framework.TestCase;
import woolpack.bool.AndAnd;
import woolpack.bool.BoolUtils;
import woolpack.convert.ConvertUtils;
import woolpack.fn.Fn;
import woolpack.fn.FnUtils;
import woolpack.fn.ThrowFn;
import woolpack.utils.BuildableArrayList;
import woolpack.utils.Utils;

public class ValueLoopValidatorTest extends TestCase {

	public void testNull() {
		final List<ValidatorContext> contextList = new ArrayList<ValidatorContext>();
		final Fn<ValidatorContext, Boolean> expression = ValidatorUtils.loopValue(
				FnUtils.recode(FnUtils.fix(false), contextList, null));
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(new HashMap<String, List<Object>>());
		context.setKey("key0");

		assertFalse(expression.exec(context));
		assertEquals(1, contextList.size());
		assertEquals("key0", contextList.get(0).getKey());
		assertEquals(0, contextList.get(0).getIndex());
	}

	public void testException() {
		final Fn<ValidatorContext, Boolean> expression = ValidatorUtils.loopValue(
				new ThrowFn<ValidatorContext, Boolean>(new IllegalStateException()));
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(Utils.map("key0", Arrays.asList((Object) "a")));
		context.setKey("key0");
		context.setIndex(5);
		try {
			expression.exec(context);
			fail();
		} catch (final IllegalStateException e) {
		}
		assertEquals(5, context.getIndex());
	}

	public void testOne() {
		final Fn<ValidatorContext, Boolean> expression = ValidatorUtils.loopValue(
				ValidatorUtils.convertValue(FnUtils.join(
						ConvertUtils.TO_STRING,
						ConvertUtils.convertRegExp(Pattern.compile("(.+)"), "[$1]")
				)));
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(Utils.map("key0", Arrays.asList((Object) "a")));
		context.setKey("key0");
		
		assertTrue(expression.exec(context));
		assertEquals(Utils.map("key0", Arrays.asList("[a]")), context.getInputMap());
	}

	public void testAndAndFalse() {
		final Fn<ValidatorContext, Boolean> expression = ValidatorUtils.loopValue(
				new AndAnd<ValidatorContext>(), 
				BoolUtils.boolSeq(
						new BuildableArrayList<Fn<? super ValidatorContext, Boolean>>()
						.list(ValidatorUtils.convertValue(FnUtils.join(
										ConvertUtils.TO_STRING,
										ConvertUtils.convertRegExp(Pattern.compile("(.+)"), "[$1]")
								)))
						.list(FnUtils.fix(false))
				)
		);
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(Utils.map("key0", Arrays.asList((Object) "a", "b", "c")));
		context.setKey("key0");
		
		assertFalse(expression.exec(context));
		assertEquals(Utils.map("key0", Arrays.asList("[a]", "b", "c")), context.getInputMap());
	}

	public void testAndAndTrue() {
		final Fn<ValidatorContext, Boolean> expression = ValidatorUtils.loopValue(
				new AndAnd<ValidatorContext>(), 
				ValidatorUtils.convertValue(FnUtils.join(
						ConvertUtils.TO_STRING,
						ConvertUtils.convertRegExp(Pattern.compile("(.+)"), "[$1]")
				))
		);
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(Utils.map("key0", Arrays.asList((Object) "a", "b", "c")));
		context.setKey("key0");
		
		assertTrue(expression.exec(context));
		assertEquals(Utils.map("key0", Arrays.asList("[a]", "[b]", "[c]")), context.getInputMap());
	}

	public void testAndFalse() {
		final Fn<ValidatorContext, Boolean> expression = ValidatorUtils.loopValue(
				BoolUtils.boolSeq(
						new BuildableArrayList<Fn<? super ValidatorContext, Boolean>>()
						.list(ValidatorUtils.convertValue(FnUtils.join(
										ConvertUtils.TO_STRING,
										ConvertUtils.convertRegExp(Pattern.compile("(.+)"), "[$1]")
								)))
						.list(FnUtils.fix(false))
				)
		);
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(Utils.map("key0", Arrays.asList((Object) "a", "b", "c")));
		context.setKey("key0");
		
		assertFalse(expression.exec(context));
		assertEquals(Utils.map("key0", Arrays.asList("[a]", "[b]", "[c]")), context.getInputMap());
	}
}
