/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.util.Iterator;
import java.util.List;

import woolpack.bool.And;
import woolpack.bool.BooleanOperator;
import woolpack.fn.Delegator;
import woolpack.fn.Fn;

/**
 * {@link ValidatorContext#getInputMap()}の
 * {@link ValidatorContext#getKey()}をキーとした
 * 値一覧に対し順次委譲先を実行する{@link Fn}です。
 * 
 * @author nakamura
 * 
 */
public class ValueLoopValidator extends Delegator<ValidatorContext, Boolean> {
	private BooleanOperator<ValidatorContext> operator;

	/**
	 * @param operator 値検証一覧の呼び出し方。
	 * @param fn
	 */
	public ValueLoopValidator(
			final BooleanOperator<ValidatorContext> operator,
			final Fn<? super ValidatorContext, Boolean> fn) {
		super(fn);
		this.operator = operator;
	}

	/**
	 * {@link And}を使用します。
	 * @param fn
	 */
	public ValueLoopValidator(final Fn<? super ValidatorContext, Boolean> fn) {
		this(new And<ValidatorContext>(), fn);
	}

	@Override
	public Boolean exec(final ValidatorContext context) {
		final int length;
		{
			final List<Object> list = context.getInputMap().get(context.getKey());
			if (list == null) {
				length = 1;
			} else {
				length = list.size();
			}
		}

		return operator.exec(context,
			new Iterable<Fn<ValidatorContext, Boolean>>() {
				public Iterator<Fn<ValidatorContext, Boolean>> iterator() {
					return new Iterator<Fn<ValidatorContext, Boolean>>() {
						private int i = 0;

						public boolean hasNext() {
							return i < length;
						}

						public Fn<ValidatorContext, Boolean> next() {
							return new Fn<ValidatorContext, Boolean>() {
								public Boolean exec(
										final ValidatorContext context) {
									final int baseIndex = context.getIndex();
									try {
										context.setIndex(i++);
										return ValueLoopValidator.super.exec(context);
									} finally {
										context.setIndex(baseIndex);
									}
								}
							};
						}
						public void remove() {
							throw new UnsupportedOperationException();
						}
					};
				}
			}
		);
	}

	public BooleanOperator<ValidatorContext> getOperator() {
		return operator;
	}
	public void setOperator(final BooleanOperator<ValidatorContext> operator) {
		this.operator = operator;
	}
}
