/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.typeconvert;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;

import woolpack.convert.ConvertUtils;
import woolpack.fn.Fn;
import woolpack.fn.FnUtils;
import woolpack.fn.NullFn;
import woolpack.utils.BuildableArrayList;
import woolpack.utils.BuildableHashMap;

/**
 * ユーティリティです。
 * 型推論で表記を簡略するためのスタティックメソッドと変数を含みます。
 * 
 * @author nakamura
 * 
 */
public final class TypeConvertUtils {
	/**
	 * {@link ConvertContext#getToType()}を返す{@link Fn}です。
	 */
	public static final Fn<ConvertContext, Class> GET_TO_TYPE = new Fn<ConvertContext, Class>() {
		public Class exec(final ConvertContext c) {
			return c.getToType();
		}
	};

	/**
	 * {@link ConvertContext#getValue()}の{@link Object#getClass()}を返す{@link Fn}です。
	 */
	public static final Fn<ConvertContext, Class> GET_FROM_TYPE = new Fn<ConvertContext, Class>() {
		public Class exec(final ConvertContext c) {
			return c.getValue().getClass();
		}
	};
	
	private static final NumberFormat NUMBER_FORMAT = new DecimalFormat();
	private static final DateFormat DATE_FORMAT = new SimpleDateFormat("yyyyMMdd");

	/**
	 * 単純型変換器のデフォルト値です。
	 */
	public static final Fn<ConvertContext, Void> SIMPLE_CONVERTER = getSimpleConverter(NUMBER_FORMAT, DATE_FORMAT);

	/**
	 * 一覧型変換器のデフォルト値です。
	 */
	public static final Fn<ConvertContext, Void> COLLECTION_CONVERTER = getCollectionConverter(NUMBER_FORMAT, DATE_FORMAT);

	public static final Fn<ConvertContext, String> GET_PROPERTY_NAME = new PropertyNameGetter();
	
	private TypeConvertUtils() {
	}
	
	private static Fn<ConvertContext, Void> getSimpleConverterPrivate(
			final NumberFormat numberFormat,
			final DateFormat dateFormat) {
		return FnUtils.seq(
				new BuildableArrayList<Fn<ConvertContext, Void>>()
				.list(new ToTypeConverter(FnUtils.TO_WRAPPER))
				.list(
					FnUtils.exec(FnUtils.join(
						GET_TO_TYPE,
						FnUtils.switching(new BuildableHashMap<Class, Fn<ConvertContext, Void>>()
							.map(String.class,
								FnUtils.exec(FnUtils.join(
									GET_FROM_TYPE,
									FnUtils.switching(
										new BuildableHashMap<Class, Fn<ConvertContext, Void>>()
										.map(java.util.Date.class, new Converter(ConvertUtils.format(ConvertUtils.formatFactory(dateFormat))))
										.map(Number.class, new Converter(ConvertUtils.format(ConvertUtils.formatFactory(numberFormat))))
										.map(Byte.class)
										.map(Short.class)
										.map(Integer.class)
										.map(Long.class)
										.map(Float.class)
										.map(Double.class)
										.map(BigInteger.class)
										.map(BigDecimal.class),
										new Converter(ConvertUtils.TO_STRING)
									)
								))
							)
							.map(Boolean.class, new Converter(new Fn<Object, Boolean>() {
								public Boolean exec(final Object c) {
									return c != null && !"false".equals(c) && !Boolean.FALSE.equals(c);
								}
							}))
							.map(Character.class, new Converter(new Fn<Object, Character>() {
								public Character exec(final Object c) {
									final String s = c.toString();
									if (s.length() == 0) {
										throw new IllegalArgumentException();
									}
									return s.charAt(0);
								}
							}))
							.map(java.sql.Date.class, new Converter(FnUtils.join(
								ConvertUtils.TO_STRING,
								ConvertUtils.parse(ConvertUtils.formatFactory(dateFormat)),
								ConvertUtils.TO_SQL_DATE
							)))
							.map(java.util.Date.class, new Converter(FnUtils.join(
								ConvertUtils.TO_STRING,
								ConvertUtils.parse(ConvertUtils.formatFactory(dateFormat)))
							))
							.map(Number.class, new Fn<ConvertContext, Void>() {
								public Void exec(final ConvertContext c) {
									final Number number;
									if (Number.class.isAssignableFrom(c.getValue().getClass())) {
										number = (Number) c.getValue();
									} else {
										try {
											number = numberFormat.parse(c.getValue().toString());
										} catch (final ParseException e) {
											throw new IllegalArgumentException(e);
										}
									}
									c.setValue(AbstractNumberConverter.convert(number, c.getToType()));
									return null;
								}
							})
							.map(Byte.class)
							.map(Short.class)
							.map(Integer.class)
							.map(Long.class)
							.map(Float.class)
							.map(Double.class)
							.map(BigInteger.class)
							.map(BigDecimal.class),
							new NullFn<ConvertContext, Void>()
						)
					))
				)
			);
	}

	/**
	 * 単純型変換器を返します。
	 * 
	 * @param numberFormat 数値フォーマット。
	 * @param dateFormat 日付フォーマット。
	 * @return 単純型変換器。
	 */
	public static Fn<ConvertContext, Void> getSimpleConverter(
			final NumberFormat numberFormat,
			final DateFormat dateFormat) {
		return new DelegationIfNecessityConverter(
				getSimpleConverterPrivate(numberFormat, dateFormat));
	}

	/**
	 * 一覧型変換器を返します。
	 * 
	 * @param numberFormat 数値フォーマット。
	 * @param dateFormat 日付フォーマット。
	 * @return 一覧型変換器。
	 */
	public static Fn<ConvertContext, Void> getCollectionConverter(
			final NumberFormat numberFormat,
			final DateFormat dateFormat) {
		return new DelegationIfNecessityConverter(new CollectionConverter(
				getSimpleConverterPrivate(numberFormat, dateFormat)));
	}
}
