/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

import woolpack.el.EL;
import woolpack.fn.Fn;
import woolpack.xml.NodeContext;

/**
 * 各属性値の出現回数をカウントして
 * {@link #interpret(Object, PropertyCounter)}
 * に処理を委譲する{@link Fn}です。
 * <br/>適用しているデザインパターン：Template Method。
 * 
 * @author nakamura
 * 
 */
public abstract class AbstractCountProperty<C extends NodeContext> implements Fn<C, Void> {
	private EL el;
	private Iterable<String> attrNames;

	/**
	 * @param el プロパティの出現回数を保持する{@link NodeContext}上の位置。
	 * @param attrNames 属性名の一覧。本クラスはこの引数の状態を変化させない。
	 */
	public AbstractCountProperty(
			final EL el,
			final Iterable<String> attrNames) {
		this.el = el;
		this.attrNames = attrNames;
	}

	public Void exec(final C c) {
		if (c.getNode().getNodeType() != Node.ELEMENT_NODE) {
			return null;
		}
		PropertyCounter counter = (PropertyCounter) el.getValue(c);
		if (counter == null) {
			counter = new PropertyCounter();
			el.setValue(c, counter);
		}
		final Element element = (Element) c.getNode();
		for (final String attrName : attrNames) {
			if (element.hasAttribute(attrName)) {
				counter.put(attrName, element.getAttribute(attrName));
				interpret(c, counter);
			}
		}
		return null;
	}

	/**
	 * {@link #exec(Object)}から呼び出されます(called)。
	 * 実装クラスでは{@link PropertyCounter}の状態により処理を行う必要があります。
	 * 
	 * @param context コンテキスト。
	 * @param counter プロパティの出現回数をカウントするカウンタ。
	 */
	public abstract void interpret(C context, PropertyCounter counter);

	public Iterable<String> getAttrNames() {
		return attrNames;
	}
	public void setAttrNames(final Iterable<String> attrNames) {
		this.attrNames = attrNames;
	}
	public EL getEl() {
		return el;
	}
	public void setEl(final EL el) {
		this.el = el;
	}
}
