/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.util.Arrays;
import java.util.HashMap;

import javax.xml.xpath.XPathExpressionException;

import junit.framework.TestCase;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

import woolpack.TestUtils;
import woolpack.el.FixEL;
import woolpack.el.PathEL;

public class XPathTest extends TestCase {

	private Node node0;
	private Node node1;

	public void testConstructor(){
		try{
			new XPath((String)null, DomConstants.NULL, DomConstants.NULL);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new XPath("//*[@id=\"value\"", DomConstants.NULL, DomConstants.NULL);
			fail();
		}catch(final RuntimeException expected){
			assertTrue(expected.getCause() instanceof XPathExpressionException);
		}
		try{
			new XPath("//HOGE", null, DomConstants.NULL);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new XPath("//HOGE", DomConstants.NULL, null);
			fail();
		}catch(final NullPointerException expected){
		}
	}
	
	public void testNormal(){
		final DomExpression expression0 = new DomExpression(){
			public void interpret(DomContext context) {
				node0 = context.getNode();
			}
		};
		final DomExpression expression = new Serial(
			TestUtils.getToNodeForm("<INPUT type=\"text\" name=\"name0\" value=\"value0\" />"),
			new XPathAttrValue("name", "name0", expression0));
		final DomContext context = new DomContext();
		expression.interpret(context);
		assertEquals("INPUT", node0.getNodeName());
		assertEquals("name0", ((Element)node0).getAttribute("name"));
	}
	
	public void testFirstOther(){
		final DomExpression expression0 = new DomExpression(){
			public void interpret(DomContext context) {
				node0 = context.getNode();
			}
		};
		final DomExpression expression1 = new DomExpression(){
			public void interpret(DomContext context) {
				node1 = context.getNode();
			}
		};
		final DomExpression expression = new Serial(
			TestUtils.getToNodeForm(
					"<INPUT type=\"text\" name=\"name0\" value=\"value0\" />" +
					"<INPUT type=\"text\" name=\"name0\" value=\"value1\" />"),
			new XPathAttrValue("name", "name0", expression0, expression1));
		final DomContext context = new DomContext();
		expression.interpret(context);
		assertEquals("INPUT", node0.getNodeName());
		assertEquals("value0", ((Element)node0).getAttribute("value"));
		assertEquals("value1", ((Element)node1).getAttribute("value"));
	}
	
	public void testTipsLoop(){
		final DomExpression expression = new Serial(
			TestUtils.getToNodeForm(
					"<INPUT type=\"text\" name=\"name0\" value=\"value0\" class=\"hoge\" />" +
					"<INPUT type=\"text\" name=\"name0\" value=\"dummy\" />" +
					"<INPUT type=\"text\" name=\"name0\" value=\"dummy\" />"),
			new XPathAttrValue("name", "name0", 
					new Loop(new PathEL("local.list0"), new PathEL("local.bean"), new UpdateAttrValue("value", new FixEL("modified"))), 
					DomConstants.REMOVE_THIS));
		final DomContext context = new DomContext();
		context.getLocal().put("list0", Arrays.asList(new HashMap(), new HashMap()));
		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context, 
				"<INPUT type=\"text\" name=\"name0\" value=\"modified\" class=\"hoge\" />" + 
				"<INPUT type=\"text\" name=\"name0\" value=\"modified\" class=\"hoge\" />"));
	}
	
	public void testTipsRemove(){
		final DomExpression expression = new Serial(
				TestUtils.getToNodeForm(
						"<SPAN>a" +  
						"<SPAN id=\"dummy\">b" + 
						"</SPAN>" + 
						"<SPAN id=\"dummy\">c" +  
						"<SPAN id=\"dummy\">d" + 
						"</SPAN>" +
						"</SPAN>" +
						"</SPAN>" + 
						"<INPUT type=\"text\" name=\"name0\" value=\"value0\" />" +
						"<INPUT type=\"text\" name=\"dummy\" value=\"value1\" />" +
						"<INPUT type=\"text\" name=\"dummy\" value=\"value2\" />"),
				new XPath("//*[@id=\"dummy\" or @name=\"dummy\"]", DomConstants.REMOVE_THIS));
			final DomContext context = new DomContext();
			expression.interpret(context);

			assertTrue(TestUtils.equalsForm(context, 
					"<SPAN>a</SPAN><INPUT type=\"text\" name=\"name0\" value=\"value0\" />"));
	}
}
