/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;

import junit.framework.TestCase;
import woolpack.AllTests;
import woolpack.TestConcurrent;
import woolpack.utils.FixSwitch;
import woolpack.utils.SwitchBuilder;

public class DoSemaphoreTest extends TestCase {
	private final ExecutorService threadPool = Executors.newFixedThreadPool(AllTests.MAX_THREAD);

	public void testConstructor(){
		try{
			new DoSemaphore(null, DomConstants.NULL, DomConstants.NULL);
		}catch(final NullPointerException expected){
		}
		try{
			new DoSemaphore(new FixSwitch<String,Semaphore>(null), null, DomConstants.NULL);
		}catch(final NullPointerException expected){
		}
		try{
			new DoSemaphore(new FixSwitch<String,Semaphore>(null), DomConstants.NULL, null);
		}catch(final NullPointerException expected){
		}
	}
	
	private void scenario(
			final int testCount, 
			final int threshold) throws InterruptedException{
		final Object lock = new Object();
		final TestConcurrent concurrentTrue0 = new TestConcurrent("id0True", lock, 1, AllTests.MAX_CONCURRENT, AllTests.MAX_THREAD, threshold);
		final TestConcurrent concurrentFalse0 = new TestConcurrent("id0False", lock, 0, 0, 0, 0);
		final TestConcurrent concurrentTrue1 = new TestConcurrent("id1True", lock, 1, AllTests.MAX_THREAD, AllTests.MAX_THREAD, threshold);
		final TestConcurrent concurrentFalse1 = new TestConcurrent("id1False", lock, 0, 0, 0, 0);
		
		final DomExpression expression = new DoSemaphore(
				new SwitchBuilder<String,Semaphore>().put("id0", new Semaphore(AllTests.MAX_CONCURRENT, true)).get(), 
				new DomExpression(){
					public void interpret(final DomContext context) {
						if("id0".equals(context.getId())){
							concurrentTrue0.execute();
						}else{
							concurrentTrue1.execute();
						}
					}
				}, new DomExpression(){
					public void interpret(final DomContext context) {
						if("id0".equals(context.getId())){
							concurrentFalse0.execute();
						}else{
							concurrentFalse1.execute();
						}
					}
				});

		final CountDownLatch latch = new CountDownLatch(AllTests.MAX_THREAD);
		for(int i=0; i<AllTests.MAX_THREAD; i++){
			final int ii = i;
			threadPool.execute(new Runnable(){
				public void run() {
					try{
						concurrentTrue0.setThreadId(ii);
						concurrentFalse0.setThreadId(ii);
						concurrentTrue1.setThreadId(ii);
						concurrentFalse1.setThreadId(ii);
						for(int i=0; i<testCount; i++){
							Thread.yield();
							final DomContext context = new DomContext();
							Thread.yield();
							context.setId((Math.random() < 0.5)?"id0":"id1");
							Thread.yield();
							try{
								expression.interpret(context);
							}catch(final RuntimeException e){
							}
							Thread.yield();
						}
					}finally{
						latch.countDown();
					}
				}
			});
		}

//		System.out.println("-- DoSemaphoreTest start --");
//		concurrentTrue0.print();
//		concurrentFalse0.print();
//		concurrentTrue1.print();
//		concurrentFalse1.print();
//		System.out.println("-- DoSemaphoreTest end --");
		assertTrue(concurrentTrue0.assertValid());
		assertEquals(0, concurrentFalse0.getCount());
		assertTrue(concurrentTrue1.assertValid());
		assertEquals(0, concurrentFalse1.getCount());
	}
	
	public void testNormal() throws InterruptedException{
		scenario(AllTests.TEST_COUNT, AllTests.TEST_THRESHOLD);
	}
}
