/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.action;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import junit.framework.TestCase;
import woolpack.TestBean;
import woolpack.TestUtils;
import woolpack.el.ELConstants;
import woolpack.el.PathEL;
import woolpack.el.ToELTargetExceptionEL;
import woolpack.utils.FixSwitch;
import woolpack.utils.SwitchBuilder;

public class ActionInvokerTest extends TestCase {

	public void testConstructor(){
		try{
			new ActionInvoker(null, new ForwardDef("forwardId0", ELConstants.NULL, ActionConstants.ANY));
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new ActionInvoker(new FixSwitch<String,ActionDef>(null), (Iterable<ForwardDef>)null);
			fail();
		}catch(final NullPointerException expected){
		}
	}
	
	public void testFindId(){
		final ForwardDef[] forwardDefArray = new ForwardDef[]{
				new ForwardDef("class", ELConstants.NULL, new ForwardMatcher(Integer.class)),
				new ForwardDef("null", ELConstants.NULL, new ForwardMatcher(null)),
				new ForwardDef("value", ELConstants.NULL, new ForwardMatcher("myValue"))
		};
		{
			final ActionResult forwardResult = ActionInvoker.findId(Arrays.asList(forwardDefArray), new Integer(3));
			assertEquals(forwardDefArray[0], forwardResult.getForwardDef());
			assertEquals(new Integer(3), forwardResult.getReturnedObject());
		}
		{
			final ActionResult info = ActionInvoker.findId(Arrays.asList(forwardDefArray), null);
			assertEquals(forwardDefArray[1], info.getForwardDef());
			assertNull(info.getReturnedObject());
		}
		{
			final ActionResult info = ActionInvoker.findId(Arrays.asList(forwardDefArray), "myValue");
			assertEquals(forwardDefArray[2], info.getForwardDef());
			assertEquals("myValue", info.getReturnedObject());
		}
		try{
			ActionInvoker.findId(Arrays.asList(forwardDefArray), "yourValue");
			fail();
		}catch(final ForwardDefNotFoundException expected){
			assertEquals("yourValue", expected.getReturnedObject());
		}
	}
	
	public void testNormal() throws ForwardDefNotFoundException{
		final ForwardDef[] forwardDefs = {
				new ForwardDef("forwardId0", ELConstants.NULL, new ForwardMatcher(Integer.class)),
				new ForwardDef("forwardId1", ELConstants.NULL, new ForwardMatcher(Integer.class)),
				new ForwardDef("forwardId2", ELConstants.NULL, ActionConstants.ANY),
				new ForwardDef("forwardId0", ELConstants.NULL, new ForwardMatcher(IndexOutOfBoundsException.class))
		};
		final ActionInvoker defs = new ActionInvoker(new SwitchBuilder<String,ActionDef>()
				.put("id0", new ActionDef(new PathEL("component0"), new ToELTargetExceptionEL(){
					@Override public Object execute(Object root, Object value) throws Exception {
						// new OGE("component0.executeInt()")
						return ((TestBean)((Map)root).get("component0")).executeInt();
					}
				}, forwardDefs[0]))
				.put("id1", new ActionDef(new PathEL("component0"), new ToELTargetExceptionEL(){
					@Override public Object execute(Object root, Object value) throws Exception {
						// new OGE("component0.executeVoid()")
						((TestBean)((Map)root).get("component0")).executeVoid();
						return null;
					}
				}, forwardDefs[1]))
				.put("id2", new ActionDef(new PathEL("component0"), new ToELTargetExceptionEL(){
					@Override public Object execute(Object root, Object value) {
						// new OGE("component0.executeException()")
						((TestBean)((Map)root).get("component0")).executeException();
						return null;
					}
				}, forwardDefs[3]))
				.get(), forwardDefs[2]);
		{
			final Map<String, String> inputMap = new HashMap<String, String>();
			inputMap.put("myInt", "7");
			final TestBean bean = new TestBean();
			final ActionResult forwardResult = defs.invoke(Collections.singletonMap("component0", bean), "id0", inputMap);
			assertEquals(forwardDefs[0], forwardResult.getForwardDef());
			assertEquals(new Integer(1), forwardResult.getReturnedObject());
			assertEquals(7, bean.getMyInt());
		}
		{
			final Map<String, String> inputMap = new HashMap<String, String>();
			inputMap.put("yourInt", "7");
			final TestBean bean = new TestBean();
			final ActionResult forwardResult = defs.invoke(Collections.singletonMap("component0", bean), "id0", inputMap);
			assertEquals(forwardDefs[0], forwardResult.getForwardDef());
			assertEquals(new Integer(1), forwardResult.getReturnedObject());
			assertEquals(0, bean.getMyInt());
		}
		{
			final Map<String, String> inputMap = new HashMap<String, String>();
			inputMap.put("myInt", "7");
			final TestBean bean = new TestBean();
			final ActionResult forwardResult = defs.invoke(Collections.singletonMap("component0", bean), "id2", inputMap);
			assertEquals(forwardDefs[3], forwardResult.getForwardDef());
			assertTrue(forwardResult.getReturnedObject() instanceof IndexOutOfBoundsException);
			assertEquals(7, bean.getMyInt());
		}
		{
			final Map<String, String> inputMap = new HashMap<String, String>();
			inputMap.put("myInt", "7");
			final TestBean bean = new TestBean();
			final ActionResult forwardResult = defs.invoke(Collections.singletonMap("component0", bean), "id1", inputMap);
			assertEquals(forwardDefs[2], forwardResult.getForwardDef());
			assertNull(forwardResult.getReturnedObject());
			assertEquals(7, bean.getMyInt());
		}
	}
	
	public void testForwardDefNotFoundException(){
		final ForwardDef forwardDefs = new ForwardDef("forwardId0", ELConstants.NULL, new ForwardMatcher(Integer.class));
		final ActionInvoker defs = new ActionInvoker(new SwitchBuilder<String,ActionDef>()
				.put("id0", new ActionDef(new PathEL("component0"), new ToELTargetExceptionEL(){
					@Override public Object execute(Object root, Object value) {
						// new OGE("component0.executeException()")
						((TestBean)((Map)root).get("component0")).executeException();
						return null;
					}
				}, forwardDefs))
				.get(), forwardDefs);
		{
			final Map<String, String> inputMap = new HashMap<String, String>();
			inputMap.put("myInt", "7");
			final TestBean bean = new TestBean();
			try{
				defs.invoke(Collections.singletonMap("component0", bean), "id0", inputMap);
				fail();
			}catch(final ForwardDefNotFoundException expected){
				assertTrue(expected.getReturnedObject() instanceof IndexOutOfBoundsException);
			}
		}
	}
	

	
	public void testSetValuesToSingle(){
		final Object o = new TestBean();
		{
			final Map<String, Object> paramMap = new HashMap<String, Object>();
			paramMap.put("myInt", new Integer(3));
			ActionInvoker.setValuesTo(o, paramMap);
			assertEquals(new Integer(3), new PathEL("myInt").getValue(o));
		}
		{
			final Map<String, Object> paramMap = new HashMap<String, Object>();
			paramMap.put("myInt", "5");
			ActionInvoker.setValuesTo(o, paramMap);
			assertEquals(new Integer(5), new PathEL("myInt").getValue(o));
		}
	}
	
	public void testSetValuesToInteger(){
		final TestBean o = new TestBean();
		
		final Map<String, Object> paramMap = new HashMap<String, Object>();
		paramMap.put("myInt", new Integer[]{new Integer(3)});
		paramMap.put("myCollection", new Integer[]{new Integer(11), new Integer(13)});
		paramMap.put("myIntArray", new Integer[]{new Integer(5), new Integer(7)});

		ActionInvoker.setValuesTo(o, paramMap);

		assertEquals(3, o.getMyInt());
		assertEquals(2, o.getMyIntArray().length);
		assertEquals(5, o.getMyIntArray()[0]);
		assertEquals(7, o.getMyIntArray()[1]);
		assertTrue(TestUtils.equals(Arrays.asList(new Integer[]{new Integer(11), new Integer(13)}), o.getMyCollection()));
	}
	
	public void testSetValuesToInt(){
		final TestBean o = new TestBean();
		
		final Map<String, Object> paramMap = new HashMap<String, Object>();
		paramMap.put("myInt", new int[]{3});
		paramMap.put("myCollection", new int[]{11, 13});
		paramMap.put("myIntArray", new int[]{5, 7});

		ActionInvoker.setValuesTo(o, paramMap);

		assertEquals(3, o.getMyInt());
		assertEquals(2, o.getMyIntArray().length);
		assertEquals(5, o.getMyIntArray()[0]);
		assertEquals(7, o.getMyIntArray()[1]);
		assertTrue(TestUtils.equals(Arrays.asList(new Integer[]{new Integer(11), new Integer(13)}), o.getMyCollection()));
	}
	
	public void testSetValuesToString(){
		final TestBean o = new TestBean();
		
		final Map<String, Object> paramMap = new HashMap<String, Object>();
		paramMap.put("myInt", new String[]{"3"});
		paramMap.put("myIntArray", new String[]{"5", "7"});
		paramMap.put("myCollection", new String[]{"11", "13"});

		ActionInvoker.setValuesTo(o, paramMap);

		assertEquals(3, o.getMyInt());
		assertEquals(2, o.getMyIntArray().length);
		assertEquals(5, o.getMyIntArray()[0]);
		assertEquals(7, o.getMyIntArray()[1]);
		assertTrue(TestUtils.equals(Arrays.asList(new String[]{"11", "13"}), o.getMyCollection()));
	}
	
	public void testSetValuesToException1(){
		final Object o = new Object();

		final Map<String, Object> paramMap = new HashMap<String, Object>();
		paramMap.put("myInt", new Integer(3));
		ActionInvoker.setValuesTo(o, paramMap);
	}
}
