/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.utils;

import java.util.Collection;

/**
 * 値に インタープリタ デザインパターンの Expression 役を割り当てることにより
 * スイッチ(java 言語の switch のような機能)を表現する。
 * {@link java.util.Map}インターフェイスはデフォルトの処理ブロック(java 言語の default のような機能)を表現するには不適切なため、
 * 本インターフェイスを定義している。
 * {@link #keys()}の返却値に含まれないキーを{@link #get(K)}に指定した場合の返却値は{@link #getDefault()}と同一であるべき。
 * また実装クラスのインスタンスはリエントラントであるべき。
 * @author nakamura
 *
 * @param <K> 検索キー。
 * @param <V> キーに対応する値。インタープリタ デザインパターンの Expression 役が割り当てられることを想定している。
 */
public interface Switchable<K,V> {
	
	/**
	 * キーに対応する値を返す。
	 * @param key キー。
	 * @return キーに対応する値。
	 */
	public V get(K key);
	
	/**
	 * 登録されているキーの一覧を返す。
	 * @return 登録されているキーの一覧。
	 */
	public Collection<K> keys();
	
	/**
	 * デフォルト値を返す。設定されていない場合は null を返す。
	 * null オブジェクト役の expression 役を指定することにより利用クラスの処理が簡潔になる。
	 * @return デフォルト値。設定されていない場合は null。
	 */
	public V getDefault();
}
