/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.el;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;

class SimpleTypeConvertEL extends GettingELAdapter {
	private final DateFormat dateFormat = new SimpleDateFormat("yyyyMMdd");
	private final DateFormat dateTimeFormat = new SimpleDateFormat("yyyyMMddHHmmss");
	
	SimpleTypeConvertEL(){}
	
	private static boolean isToByte(final Class toType){
		return Byte.class.isAssignableFrom(toType) || byte.class.isAssignableFrom(toType);
	}
	
	private static boolean isToDouble(final Class toType){
		return Double.class.isAssignableFrom(toType) || double.class.isAssignableFrom(toType);
	}
	
	private static boolean isToFloat(final Class toType){
		return Float.class.isAssignableFrom(toType) || float.class.isAssignableFrom(toType);
	}
	
	private static boolean isToInteger(final Class toType){
		return Integer.class.isAssignableFrom(toType) || int.class.isAssignableFrom(toType);
	}
	
	private static boolean isToLong(final Class toType){
		return Long.class.isAssignableFrom(toType) || long.class.isAssignableFrom(toType);
	}
	
	private static boolean isToShort(final Class toType){
		return Short.class.isAssignableFrom(toType) || short.class.isAssignableFrom(toType);
	}

	@Override
	public Object getValue(final Object beforeValue, final Class toType)
			throws ELTargetRuntimeException {
		if(beforeValue == null){
			return null;
		}
		if(String.class.isAssignableFrom(toType)){
			return beforeValue.toString();
		}
		if(Boolean.class.isAssignableFrom(toType)){
			return new Boolean("true".equals(beforeValue));
		}
		if(Character.class.isAssignableFrom(toType)){
			return new Character(beforeValue.toString().charAt(0));
		}
		if(Number.class.isAssignableFrom(beforeValue.getClass())){
			final Number n = (Number)beforeValue;
			if(isToInteger(toType)){
				return new Integer(n.intValue());
			}
			if(isToLong(toType)){
				return new Long(n.longValue());
			}
			if(isToShort(toType)){
				return new Short(n.shortValue());
			}
			if(isToByte(toType)){
				return new Byte(n.byteValue());
			}
			if(isToDouble(toType)){
				return new Double(n.doubleValue());
			}
			if(isToFloat(toType)){
				return new Float(n.floatValue());
			}
		}
		if(String.class.isAssignableFrom(beforeValue.getClass())){
			final String s = (String)beforeValue;
			if(isToInteger(toType)){
				return new Integer(s);
			}
			if(isToLong(toType)){
				return new Long(s);
			}
			if(java.util.Date.class.isAssignableFrom(toType)){
				try{
					final java.util.Date date;
					if(s.length() == 8){
						date = ((DateFormat)dateFormat.clone()).parse(s);
					}else if(s.length() == 14){
						date = ((DateFormat)dateTimeFormat.clone()).parse(s);
					}else{
						date = null;
					}
					if(date == null){
						// do nothing.
					}else if(java.sql.Date.class.isAssignableFrom(toType)){
						return new java.sql.Date(date.getTime());
					}else{
						return date;
					}
				}catch(final ParseException e){
				}
			}
			if(isToShort(toType)){
				return new Short(s);
			}
			if(isToByte(toType)){
				return new Byte(s);
			}
			if(isToDouble(toType)){
				return new Double(s);
			}
			if(isToFloat(toType)){
				return new Float(s);
			}
		}
		return beforeValue;
	}
}
