/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.el;

import java.util.Map;

import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.commons.lang.builder.ToStringBuilder;

/**
 * {@link Map}のみをサポートする{@link PropertyELFactory}。
 * @author nakamura
 *
 */
public class MapPropertyELFactory implements PropertyELFactory {
	
	private final GettingEL convertEL;
	private final PropertyELFactory factory;
	
	/**
	 * コンストラクタ。
	 * @param convertEL 型変換器。
	 * @param factory 型が{@link Map}でない場合の委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public MapPropertyELFactory(final GettingEL convertEL, final PropertyELFactory factory){
		convertEL.getClass();
		factory.getClass();
		
		this.convertEL = convertEL;
		this.factory = factory;
	}

	public GettingEL newGetter(final Class clazz, final String propertyName) {
		if(Map.class.isAssignableFrom(clazz)){
			return new MapEL(propertyName, convertEL);
		}
		return factory.newGetter(clazz, propertyName);
	}

	public EL newSetter(final Class clazz, final String propertyName) {
		if(Map.class.isAssignableFrom(clazz)){
			return new MapEL(propertyName, convertEL);
		}
		return factory.newSetter(clazz, propertyName);
	}
	
	@Override public int hashCode(){
		return new HashCodeBuilder().append(convertEL).append(factory).toHashCode();
	}
	
	@Override public boolean equals(final Object obj){
		if(!(obj instanceof MapPropertyELFactory)){
			return false;
		}
		if(this == obj){
			return true;
		}
		final MapPropertyELFactory o = (MapPropertyELFactory)obj;
		return new EqualsBuilder()
		.append(this.convertEL, o.convertEL)
		.append(this.factory, o.factory)
		.isEquals();
	}
	
	@Override public String toString(){
		return new ToStringBuilder(this).append("convertEL", convertEL).append("factory", factory).toString();
	}
}
