package hiro.yoshioka.sql.util;

import hiro.yoshioka.sql.resource.DBColumn;
import hiro.yoshioka.sql.resource.DBSchema;
import hiro.yoshioka.sql.resource.DBTable;
import hiro.yoshioka.sql.resource.IDBColumn;
import hiro.yoshioka.sql.resource.IDBTable;
import hiro.yoshioka.util.StringUtil;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.math.BigDecimal;
import java.sql.CallableStatement;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class SQLUtil {
	public static final String FORMAT_YYYY_MM_DD_HH_MM_SS = "yyyy-MM-dd HH:mm:ss";

	// 1 [2006]
	// 2 [06]
	// 3 [08]
	public static final Pattern FORMAT_DATE = Pattern
			.compile("(\\d\\d\\d\\d).(\\d\\d).(\\d\\d)");

	// 1 [2006]
	// 2 [06]
	// 3 [08]
	// 4 [ 10:12:30.123456]
	// 5 [10:12:30]
	// 6 [10]
	// 7 [12]
	// 8 [30]
	// 9 [.123456]
	// 10 [123456]
	public static final Pattern FORMAT_TIMESTAMP = Pattern
			.compile("(\\d\\d\\d\\d).(\\d\\d).(\\d\\d)( +((\\d?\\d):(\\d?\\d):(\\d?\\d))([.](\\d+))?)?");

	public static final Pattern FORMAT_TIME = Pattern
			.compile("\\d?\\d:\\d?\\d:\\d?\\d");
	private static final Log fLogger = LogFactory.getLog(SQLUtil.class
			.getClass());

	public static final String createCallableSql(IDBTable dbTable) {
		// if (fLogger.isTraceEnabled()) {
		// fLogger.info("\n----- INPUTS PARAMETERS -----");
		// }

		StringBuffer sql = new StringBuffer();
		if (dbTable.isFunction()) {
			sql.append("{ ? = call ").append(dbTable.getName()).append("(");
		} else {
			sql.append("{ call ").append(dbTable.getName()).append("(");
		}
		IDBColumn[] cols = dbTable.getColumns();
		boolean existsParam = false;
		for (int i = 0; i < cols.length; i++) {
			if (cols[i].isColumnReturn()) {
				continue;
			}
			sql.append("?,");
			existsParam = true;
		}
		sql.setLength(sql.length() - 1);
		if (existsParam) {
			sql.append(") }");
		} else {
			sql.append(" }");
		}
		return sql.toString();
	}

	public static String getStringUsingType(CallableStatement st, int idx,
			int type) throws SQLException {
		switch (type) {
		case Types.TIMESTAMP:
			Timestamp t = st.getTimestamp(idx);
			if (t == null) {
				return null;
			}
			java.util.Date dd = new java.util.Date(t.getTime());
			return String.format("%tF %<tT", dd);
		case Types.DATE:
			Timestamp tt = st.getTimestamp(idx);
			if (tt == null) {
				return null;
			}
			return String.format("%tF", new java.util.Date(tt.getTime()));
		case Types.DECIMAL:
		case Types.NUMERIC:
			BigDecimal decimal = st.getBigDecimal(idx);
			if (decimal == null) {
				return StringUtil.EMPTY_STRING;
			} else {
				return decimal.toPlainString();
			}
		default:
			return st.getString(idx);
		}
	}

	public static boolean isNumericType(int type) {
		switch (type) {
		case Types.DECIMAL:
		case Types.NUMERIC:
		case Types.INTEGER:
		case Types.SMALLINT:
		case Types.FLOAT:
		case Types.TINYINT:
		case Types.DOUBLE:
			return true;
		}
		return false;
	}

	public static void setBinds(PreparedStatement st, int idx, BindObject o)
			throws SQLException {
		if (o.target == null) {
			if (fLogger.isTraceEnabled()) {
				fLogger.info((idx) + " : setNull[" + o + "]");
			}
			st.setNull(idx, o.type);
			return;
		}
		switch (o.type) {
		case Types.BIT:
		case Types.BOOLEAN:
			if (fLogger.isTraceEnabled()) {
				fLogger.trace((idx) + " : setBoolean[" + o + "]");
			}
			st.setBoolean(idx, (Boolean) o.target);
			break;
		case Types.TIME:
			if (fLogger.isTraceEnabled()) {
				fLogger.trace((idx) + " : setTime[" + o + "]");
			}
			st.setTime(idx, (Time) o.target);
			break;
		case Types.TIMESTAMP:
			if (fLogger.isTraceEnabled()) {
				fLogger.trace((idx) + " : setTimeStamp[" + o + "]");
			}
			st.setTimestamp(idx, (Timestamp) o.target);
			break;
		case Types.DATE:
			if (fLogger.isTraceEnabled()) {
				fLogger.trace((idx) + " : setDate[" + o + "]");
			}
			st.setDate(idx, (Date) o.target);
			break;
		case Types.SMALLINT:
			if (fLogger.isTraceEnabled()) {
				fLogger.trace((idx) + " : setShort[" + o + "]");
			}
			st.setShort(idx, (Short) o.target);
			break;
		case Types.INTEGER:
			if (fLogger.isTraceEnabled()) {
				fLogger.trace((idx) + " : setInt[" + o + "]");
			}
			st.setInt(idx, (Integer) o.target);
			break;
		case Types.REAL:
		case Types.FLOAT:
		case Types.BIGINT:
		case Types.NUMERIC:
		case Types.DOUBLE:
		case Types.DECIMAL:
			if (fLogger.isTraceEnabled()) {
				fLogger.trace((idx) + " : setBigDecimal[" + o + "]");
			}
			st.setBigDecimal(idx, (BigDecimal) o.target);
			break;
		case Types.CHAR:
		case Types.NCHAR:
		case Types.VARCHAR:
		case Types.NVARCHAR:
		case Types.LONGVARCHAR:
		case Types.LONGNVARCHAR:
			if (fLogger.isTraceEnabled()) {
				fLogger.trace((idx) + " : setString[" + o + "]");
			}
			st.setString(idx, (String) o.target);
			break;
		default:
			if (fLogger.isTraceEnabled()) {
				fLogger.trace((idx) + " : setString[" + o + "]");
			}
			st.setString(idx, (String) o.target);
			break;
		}
	}

	public static Integer getInteger(String value) {
		if (fLogger.isTraceEnabled()) {
			fLogger.trace("value[" + value + "]");
		}
		if (value.length() == 0) {
			return null;
		}
		return new Integer(value);
	}

	public static Short getShort(String value) {
		if (fLogger.isTraceEnabled()) {
			fLogger.trace("value[" + value + "]");
		}
		if (value.length() == 0) {
			return null;
		}
		return new Short(value);
	}

	public static BigDecimal getBigDecimal(String value) {
		if (fLogger.isTraceEnabled()) {
			fLogger.trace("value[" + value + "]");
		}
		if (value.length() == 0) {
			return null;
		}
		return new BigDecimal(value);
	}

	public static Date getDate(String value) {
		if (fLogger.isTraceEnabled()) {
			fLogger.trace("value[" + value + "]");
		}
		// 3/17/2005 18:40:21

		value = value.replaceAll("\\s*\\d+:\\d+:\\d+\\s*",
				StringUtil.EMPTY_STRING);
		if (value.length() == 0) {
			if (fLogger.isTraceEnabled()) {
				fLogger.trace("value length=0 and return null..");
			}
			return null;
		} else if (value.equalsIgnoreCase("SYSDATE")
				|| "NOW".equalsIgnoreCase(value)) {
			if (fLogger.isTraceEnabled()) {
				fLogger.trace("value is SYSDATE and return currentTimeStamp");
			}
			return new Date(System.currentTimeMillis());
		}

		if (value.matches("\\d+[ -/]+\\d+")) {
			int year = GregorianCalendar.getInstance().get(Calendar.YEAR);
			String dateVal = value.replaceAll("(\\d+)[ -/]+(\\d+)", year
					+ "-$1-$2");
			if (fLogger.isTraceEnabled()) {
				fLogger.trace(dateVal);
			}
			return Date.valueOf(dateVal);
		} else if (value.matches("\\d+[ -/]+\\d+[ -/]+\\d+")) {
			String dateVal = value.replaceAll("(\\d+)[ -/]+(\\d+)[ -/]+(\\d+)",
					"$1-$2-$3");
			if (fLogger.isTraceEnabled()) {
				fLogger.trace(dateVal);
			}
			return Date.valueOf(dateVal);
		} else if (value.matches("(\\d{4})(\\d{2})(\\d{2})")) {
			String dateVal = value.replaceAll("(\\d{4})(\\d{2})(\\d{2})",
					"$1-$2-$3");
			if (fLogger.isTraceEnabled()) {
				fLogger.trace(dateVal);
			}
			return Date.valueOf(dateVal);
		} else {
			if (fLogger.isWarnEnabled()) {
				fLogger.warn("no matches Date format[" + value + "]");
			}
			return null;
		}
	}

	public static java.sql.Timestamp getTimeStamp(String target) {
		if (target == null || target.length() == 0) {
			return null;
		} else if (target.equalsIgnoreCase("SYSDATE")
				|| "NOW".equalsIgnoreCase(target)) {
			return new Timestamp(System.currentTimeMillis());
		}
		Calendar cal = GregorianCalendar.getInstance();
		int year = cal.get(Calendar.YEAR);
		int month = cal.get(Calendar.MONTH) + 1;
		int day = cal.get(Calendar.DATE);
		int hour = 0;
		int min = 0;
		int sec = 0;
		String nano = "0";

		if (target.matches("\\d+[ -/]+\\d+")) {
			String[] v = target.split("[ -/]+");
			month = Integer.parseInt(v[0]);
			day = Integer.parseInt(v[1]);
		} else if (target.matches("(\\d{4})(\\d{2})(\\d{2})")) {
			year = Integer.parseInt(target.substring(0, 4));
			month = Integer.parseInt(target.substring(5, 6));
			day = Integer.parseInt(target.substring(7, 8));
		} else {
			Matcher m = FORMAT_TIMESTAMP.matcher(target);
			System.out.println("gc=" + m.groupCount());
			if (m.matches()) {
				year = Integer.parseInt(m.group(1));
				month = Integer.parseInt(m.group(2));
				day = Integer.parseInt(m.group(3));
			}
			if (m.group(5) != null) {
				hour = Integer.parseInt(m.group(6));
				min = Integer.parseInt(m.group(7));
				sec = Integer.parseInt(m.group(8));
			}
			if (m.group(10) != null) {
				nano = m.group(10);
			}
		}
		String s = String.format("%d-%02d-%02d %02d:%02d:%02d.%-9s", year,
				month, day, hour, min, sec, nano);
		java.sql.Timestamp ret = java.sql.Timestamp.valueOf(s);
		return ret;
	}

	public static Time getTime(String value) {
		Matcher m = FORMAT_TIME.matcher(value);
		if (m.matches()) {
			return Time.valueOf(value);
		} else if ("SYSDATE".equalsIgnoreCase(value)
				|| "NOW".equalsIgnoreCase(value)) {
			return new Time(System.currentTimeMillis());
		}
		return null;
	}

	public static Boolean getBoolean(String value) {
		if (StringUtil.isEmpty(value)) {
			return null;
		}
		char c = value.charAt(0);
		if (c == '0' || c == 'f' || c == 'F' || c == '\u00d7' || c == '-') {
			return Boolean.FALSE;
		}
		return Boolean.TRUE;
	}

	public static Object cnv(String input, int dataType) {
		switch (dataType) {
		case Types.CHAR:
		case Types.NCHAR:
		case Types.VARCHAR:
		case Types.NVARCHAR:
			return input;
		case Types.DECIMAL:
		case Types.NUMERIC:
		case Types.INTEGER:
		case Types.FLOAT:
			return getBigDecimal(input);
		case Types.DATE:
			return getDate(input);
		case Types.TIME:
			return getTime(input);
		case Types.TIMESTAMP:
			return getTimeStamp(input);
		case Types.BLOB:
			break;
		case Types.BINARY:
			break;
		default:
		}
		return null;
	}

	public static String getTypeString(int type) {
		SQLTypesNum ret = SQLTypesNum.parse(type);
		if (ret == null) {
			return "UnKnown[" + type + "]";
		}
		return ret.getMessage();
	}

	public static void main(String[] args) {
		DBTable table = new DBTable((DBSchema) null);
		DBColumn column = new DBColumn(table);
		table.putResource(column.getName(), column);
		column.setName("DATA");
		column.setDataType(Types.BLOB);

		// System.out.println(cnvEmptyBlob(table,
		// "update BL set data=? where pk=121"));
	}

	public static Object readObject(File f) {
		ObjectInputStream in = null;
		try {
			if (f.exists() && f.isFile()) {
				in = new ObjectInputStream(new FileInputStream(f));
				return in.readObject();

			} else {
				fLogger.info("File Not Found or Not File ["
						+ f.getAbsolutePath() + "]");
				return null;
			}
		} catch (Exception e) {
			fLogger.error("Deserialize error", e);
			return null;
		} finally {
			if (in != null) {
				try {
					in.close();
				} catch (IOException e1) {
					return null;
				}
			}
		}
	}
}
