package hiro.yoshioka.sql.params;

import hiro.yoshioka.ast.sql.DatabaseType;
import hiro.yoshioka.sql.resource.DBRoot;
import hiro.yoshioka.util.StringUtil;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.Collections;
import java.util.Properties;
import java.util.Set;
import java.util.TreeSet;

import com.mongodb.MongoOptions;

public class ConnectionProperties extends Properties implements Serializable,
		Cloneable {
	public static final String ADMIN_AUTHENTICATE = "admin";
	public static final String MONGO_OPTIONS = "MONGO_OPTIONS";
	public static final String HOST = "HOST";
	public static final String DRIVER_NAME = "DRIVER_NAME";
	public static final String DRIVER_FILE_PATH = "DRIVER_FILE_PATH";
	public static final String DISPLAY = "DISPLAY";
	public static final String PORT = "PORT";
	public static final String URL = "url";
	public static final String INTERNAEL_LOGIN = "INTERNAEL_LOGIN";
	public static final String MAX_ROW_NUM = "MAX_ROW_NUM";
	public static final String DBMS = "DBMS";
	private DBRoot root;
	private boolean doSerializableDBRoot;
	private boolean captureWithViewInfo;
	private boolean captureWithColumnInfo;
	private boolean captureWithDDL;
	private DBResourceCapturingFilter capturingFilter = DBResourceCapturingFilter.NOTHING;
	private static final String CATALOG_REGREX = "CATALOG_REGREX";
	private static final String TITLE_REGREX = "TITLE_REGREX";
	private static final String FILEPATH_REGREX = "FILEPATH_REGREX";

	private boolean captureResourceAfterTheConnectProcess;

	private transient boolean connected;
	private int recursiveSearchDepth;
	private long created = System.currentTimeMillis();

	public boolean isCaptureWithColumnInfo() {
		return captureWithColumnInfo;
	}

	public void setCaptureWithColumnInfo(boolean captureWithColumnInfo) {
		this.captureWithColumnInfo = captureWithColumnInfo;
	}

	public boolean isCaptureResourceAfterTheConnectProcess() {
		return captureResourceAfterTheConnectProcess;
	}

	public void setCaptureResourceAfterTheConnectProcess(
			boolean captureResourceAfterTheConnectProcess) {
		this.captureResourceAfterTheConnectProcess = captureResourceAfterTheConnectProcess;
	}

	public boolean isCaptureWithViewInfo() {
		return captureWithViewInfo;
	}

	public void setCaptureWithViewInfo(boolean captureWithViewInfo) {
		this.captureWithViewInfo = captureWithViewInfo;
	}

	public void setCaptureWithDDL(boolean captureWithDDL) {
		this.captureWithDDL = captureWithDDL;
	}

	public boolean isCaptureWithDDL() {
		return captureWithDDL;
	}

	public boolean isSerializableDBRoot() {
		return doSerializableDBRoot;
	}

	public void setSerializableDBRoot(boolean serializableDBRoot) {
		this.doSerializableDBRoot = serializableDBRoot;
	}

	public void setAdminAuthenticate(String user, String pass) {
		put(ADMIN_AUTHENTICATE, new DBUserPass(ADMIN_AUTHENTICATE, user, pass));
	}

	public void setAuthenticate(DBUserPass dup) {
		put("Authenticate", dup);
	}

	public void setAuthenticate(String dbName, String user, String pass) {
		this.setAuthenticate(new DBUserPass(dbName, user, pass));
	}

	public DBUserPass getAuthenticate() {
		return (DBUserPass) get("Authenticate");
	}

	public void setHost(String host) {
		setProperty(HOST, host);
	}

	public DBUserPass getAdminAuthenticate() {
		return (DBUserPass) get(ADMIN_AUTHENTICATE);
	}

	public String getHost() {
		return StringUtil.nvl(getProperty(HOST));
	}

	public void setMongoOptions(MongoOptions options) {
		put(MONGO_OPTIONS, options);
	}

	public void setDisplayString(String displayString) {
		setProperty(DISPLAY, displayString);
	}

	public String getDisplayString() {
		return StringUtil.nvl(getProperty(DISPLAY));
	}

	public void setURLString(String url) {
		setProperty(URL, url);
	}

	public String getURLString() {
		return StringUtil.nvl(getProperty(URL));
	}

	public void setDriverName(String driverName) {
		setProperty(DRIVER_NAME, driverName);
	}

	public String getDriverName() {
		return StringUtil.nvl(getProperty(DRIVER_NAME));
	}

	public void setDriverFilePath(String driverFilePath) {
		setProperty(DRIVER_FILE_PATH, driverFilePath);
	}

	public String getDriverFilePath() {
		return StringUtil.nvl(getProperty(DRIVER_FILE_PATH));
	}

	public MongoOptions getMongoOptions() {
		return (MongoOptions) get(MONGO_OPTIONS);
	}

	public void setDBRoot(DBRoot root) {
		this.root = root;
	}

	public DBRoot getDBRoot() {
		return root;
	}

	@Override
	public synchronized String toString() {
		return "Fielter[" + getCapturingFilter() + "] TITLE_REX["
				+ getTitleRegres() + "]" + super.toString() + " :DBRoot ("
				+ (root != null) + ")";
	}

	@Override
	public synchronized Object clone() {
		ConnectionProperties clone = (ConnectionProperties) super.clone();
		// long t = System.currentTimeMillis();
		// t = t + (t / 1000 % 10);
		// clone.created = t;
		return clone;
	}

	@Override
	public synchronized int hashCode() {
		return (int) created;
	}

	public boolean isConnected() {
		return connected;
	}

	public void setConnected(boolean connected) {
		this.connected = connected;
	}

	public void setRecursiveSearchDepth(int recursiveSearchDepth) {
		this.recursiveSearchDepth = recursiveSearchDepth;
	}

	public int getRecursiveSearchDepth() {
		return recursiveSearchDepth;
	}

	public boolean isOnlyConnectedUserInfo() {
		if (DBResourceCapturingFilter.MATCHES.equals(getCapturingFilter())) {
			Set<String> set = getCalalogRegrex();
			if (set.size() == 1) {
				for (String s : set) {
					return s.equalsIgnoreCase(getAuthenticate().user);
				}
			}
		}
		return false;
	}

	@Override
	public synchronized boolean equals(Object o) {
		if (o instanceof ConnectionProperties) {
			return this.created == ((ConnectionProperties) o).created;
		}
		return false;
	}

	public long getCreated() {
		return created;
	}

	private void readObject(ObjectInputStream in) throws IOException,
			ClassNotFoundException {
		in.defaultReadObject();
		connected = false;
	}

	private void writeObject(ObjectOutputStream stream) throws IOException {
		if (!doSerializableDBRoot) {
			root = null;
		}
		stream.defaultWriteObject();
	}

	public Properties getRelationalDBConnectionProperties() {
		Properties p = new Properties();
		DBUserPass dup = getAuthenticate();
		p.setProperty("user", dup.user);
		p.setProperty("password", dup.pass);
		p.setProperty("url", getURLString());
		return p;
	}

	public DBResourceCapturingFilter getCapturingFilter() {
		return capturingFilter;
	}

	public void setCapturingFilter(DBResourceCapturingFilter capturingFilter) {
		this.capturingFilter = capturingFilter;
		if (capturingFilter == DBResourceCapturingFilter.NOTHING) {
			remove(CATALOG_REGREX);
			remove(FILEPATH_REGREX);
			remove(TITLE_REGREX);
		}
	}

	public Set<String> getCalalogRegrex() {
		return getRegrexSet(CATALOG_REGREX);
	}

	public Set<String> getFilePathRegrex() {
		return getRegrexSet(FILEPATH_REGREX);
	}

	public Set<String> getTitleRegres() {
		return getRegrexSet(TITLE_REGREX);
	}

	public void addCalalogRegrex(String matchingString) {
		addRegrex(matchingString, CATALOG_REGREX);
	}

	public void addFilePathRegrex(String matchingString) {
		addRegrex(matchingString, FILEPATH_REGREX);
	}

	public void addTitleRegrex(String matchingString) {
		addRegrex(matchingString, TITLE_REGREX);
	}

	private Set<String> getRegrexSet(String key) {
		Set<String> ret = (Set<String>) get(key);
		if (ret == null) {
			return Collections.EMPTY_SET;
		}
		return ret;
	}

	private void addRegrex(String matchingString, String key) {
		if (StringUtil.isEmpty(matchingString)) {
			return;
		}
		Set<String> ret = (Set<String>) get(key);
		if (ret == null) {
			ret = new TreeSet<String>();
			put(key, ret);
		}
		ret.add(matchingString);
	}

	public void setMaxRowNum(int rownum) {
		put(MAX_ROW_NUM, rownum);
	}

	public int getMaxRowNum() {
		Integer max = (Integer) get(MAX_ROW_NUM);
		if (max == null) {
			return 100;
		}
		return max;
	}

	public boolean isCapturingTarget(String categoriesOrSchemaName) {
		switch (getCapturingFilter()) {
		case NOTHING:
			return true;
		case IGNORE:
			if (!StringUtil.isEmpty(categoriesOrSchemaName)) {
				for (String ignore : getCalalogRegrex()) {
					if (categoriesOrSchemaName.matches(ignore)) {
						return false;
					}
				}
			}
			return true;
		case MATCHES:
			return isAnyMatch(categoriesOrSchemaName, getCalalogRegrex());
		}
		return true;
	}

	public boolean isCapturingTarget(String categories, String filePath,
			String title) {
		switch (getCapturingFilter()) {
		case NOTHING:
			return true;
		case IGNORE:
			if (!StringUtil.isEmpty(categories)) {
				for (String ignore : getCalalogRegrex()) {
					if (categories.matches(ignore)) {
						return false;
					}
				}
			}
			if (!StringUtil.isEmpty(filePath)) {
				for (String ignore : getFilePathRegrex()) {
					if (filePath.matches(ignore)) {
						return false;
					}
				}
			}
			if (!StringUtil.isEmpty(title)) {
				for (String ignore : getTitleRegres()) {
					if (title.matches(ignore)) {
						return false;
					}
				}
			}
			return true;
		case MATCHES:
			return isAnyMatch(categories, getCalalogRegrex())
					&& isAnyMatch(filePath, getFilePathRegrex())
					&& isAnyMatch(title, getTitleRegres());
		}
		return true;
	}

	private boolean isAnyMatch(String compareTarget, Set<String> matchSet) {
		if (StringUtil.isEmpty(compareTarget)) {
			return true;
		}
		if (matchSet.size() == 0) {
			return true;
		}
		for (String ok : matchSet) {
			if (compareTarget.matches(ok)) {
				return true;
			}
		}
		return false;
	}

	public DatabaseType getDatabaseType() {
		return DatabaseType.parse(getDriverName());
	}

}
