package hiro.yoshioka.sql;

import hiro.yoshioka.ast.sql.oracle.WolfSQLParserConstants;
import hiro.yoshioka.sdh.ResultSetDataHolder;
import hiro.yoshioka.sdh2.ResultSetDataHolder2;
import hiro.yoshioka.sql.resource.DBColumn;
import hiro.yoshioka.sql.resource.DBSequence;
import hiro.yoshioka.sql.resource.IDBColumn;
import hiro.yoshioka.sql.resource.IDBResource;
import hiro.yoshioka.sql.resource.IDBSchema;
import hiro.yoshioka.sql.resource.IDBTable;
import hiro.yoshioka.util.StringUtil;

import java.io.File;
import java.sql.Driver;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class HSQL extends GeneralSQL {
	static final String SQL_EXISTS_SCHEMA = "SELECT COUNT(*) CNT FROM INFORMATION_SCHEMA.SYSTEM_SCHEMAS WHERE UPPER(TABLE_SCHEM) = ?";
	static final String SQL_CREATE_SCHEMA_PRE = "CREATE SCHEMA ";
	static final String SQL_CREATE_SCHEMA_POST = " AUTHORIZATION DBA";
	static final String SQL_EXISTS_TABLE = "SELECT COUNT(*) CNT FROM INFORMATION_SCHEMA.SYSTEM_TABLES WHERE upper(TABLE_SCHEM) = ? AND UPPER(TABLE_NAME) =?";
	static final String SQL_DROP_TABLE = "DROP TABLE ";

	static final String _SQL_COLUMN_DEF = "SELECT  "
			+ "  TABLE_SCHEM,TABLE_NAME,COLUMN_NAME,DATA_TYPE,TYPE_NAME "
			+ " ,COLUMN_SIZE,BUFFER_LENGTH,DECIMAL_DIGITS,NUM_PREC_RADIX,NULLABLE "
			+ " ,REMARKS,SQL_DATA_TYPE,CHAR_OCTET_LENGTH "
			+ " ,ORDINAL_POSITION,IS_NULLABLE "
			+ " FROM INFORMATION_SCHEMA.SYSTEM_COLUMNS  "
			+ " WHERE TABLE_SCHEM = ? AND TABLE_NAME=? ORDER BY TABLE_SCHEM,TABLE_NAME,ORDINAL_POSITION";

	static final String _SQL_COLUMN_PK = "SELECT COLUMN_NAME FROM INFORMATION_SCHEMA.SYSTEM_PRIMARYKEYS  "
			+ " WHERE TABLE_SCHEM = ? AND TABLE_NAME=?";

	static final String _SELECT_ALL_SEQUENCE = "SELECT  "
			+ "  SEQUENCE_SCHEMA, " + "  SEQUENCE_NAME, "
			+ "  DTD_IDENTIFIER, " + "  MAXIMUM_VALUE \"MAX\", "
			+ "  MINIMUM_VALUE \"MIN\", " + "  INCREMENT, "
			+ "  CYCLE_OPTION CYCLE, " + "  START_WITH "
			+ " FROM INFORMATION_SCHEMA.SYSTEM_SEQUENCES  "
			+ " WHERE SEQUENCE_SCHEMA = ? ";
	static final String _SELECT_SEQUENCE_CNT = "SELECT COUNT(*) AS \"CNT\" FROM INFORMATION_SCHEMA.SYSTEM_SEQUENCES WHERE UPPER(SEQUENCE_SCHEMA) = UPPER(?) AND UPPER(SEQUENCE_NAME) =UPPER(?)";

	public static String getSuggestURL() {
		return "jdbc:hsqldb:hsql://127.0.0.1:9003/hsql_db_yon";
	}

	protected HSQL(Driver ds) {
		super(ds);
	}

	public void createTable(IDBTable table) throws SQLException {
		IDBResource schema = table.getParent();
		if (!existsSchema(schema.getName())) {
			execute(SQL_CREATE_SCHEMA_PRE + schema.getName()
					+ SQL_CREATE_SCHEMA_POST);
		}
		if (existsTable(schema.getName(), table.getName())) {
			execute(SQL_DROP_TABLE + schema.getName() + "." + table.getName());
		}

		IDBColumn[] columns = table.getColumns();
		StringBuilder buff = new StringBuilder();
		buff.append("CREATE TABLE ");
		buff.append(table.getParent().getName());
		buff.append(".");
		buff.append(table.getName());
		buff.append(" ( \n");
		for (int i = 0; i < columns.length; i++) {
			if (i > 0) {
				buff.append(",");
			}
			buff.append("  ");
			buff.append(columns[i].getName());
			buff.append(" ");
			buff.append(dataType(columns[i]));
			if (columns[i].isNotNull()) {
				buff.append(" NOT NULL");
			}

			if (columns[i].isPkey()) {
				buff.append(" PRIMARY KEY");
			}
			buff.append("\n");
		}
		buff.append(") ");

		execute(buff.toString());

	}

	private String dataType(IDBColumn column) {
		switch (column.getDataType()) {
		case Types.TINYINT:
		case Types.SMALLINT:
		case Types.INTEGER:
		case Types.BIGINT:
		case Types.NUMERIC:
		case Types.DECIMAL:
			return "NUMERIC(" + column.getSize() + ")";

		case Types.FLOAT:
		case Types.REAL:
		case Types.DOUBLE:
			return "REAL";

		case Types.CHAR:
		case Types.NCHAR:
			return "CHAR(" + column.getSize() + ")";
		case Types.VARCHAR:
		case Types.LONGVARCHAR:
		case Types.NVARCHAR:
		case Types.LONGNVARCHAR:
			return "VARCHAR(" + column.getSize() + ")";
		case Types.DATE:
			return "DATE";
		case Types.TIME:
			return "TIME";
		case Types.TIMESTAMP:
			return "TIMESTAMP";
		case Types.BLOB:
			return "BLOB";
		case Types.CLOB:
			return "CLOB";
		case Types.BIT:
			return "BIT";
		case Types.BOOLEAN:
			return "BOOLEAN";
		case Types.BINARY:
			return "BINARY";

		case Types.VARBINARY:
			return "VARBINARY";
		case Types.LONGVARBINARY:
			return "LONGVARBINARY";
		case Types.NULL:
		case Types.OTHER:
		case Types.JAVA_OBJECT:
		case Types.DISTINCT:
		case Types.STRUCT:
		case Types.ARRAY:
		case Types.REF:
		case Types.DATALINK:
		case Types.ROWID:
		case Types.NCLOB:
		case Types.SQLXML:

			return "UNDEFINED";
		}
		return "UNDEFINED";
	}

	public boolean existsSchema(String name) throws SQLException {
		ResultSetDataHolder rdh = executePrepareQuery(SQL_EXISTS_SCHEMA,
				new String[] { name.toUpperCase() });
		return rdh.getIntData(0, "CNT") > 0;
	}

	public boolean existsTable(String schemaName, String tableName)
			throws SQLException {
		ResultSetDataHolder rdh = executePrepareQuery(
				SQL_EXISTS_TABLE,
				new String[] { schemaName.toUpperCase(),
						tableName.toUpperCase() });
		return rdh.getIntData(0, "CNT") > 0;
	}

	@Override
	protected void setTableColumns(String schema, IDBTable table)
			throws SQLException {
		ResultSetDataHolder rdh = null;
		IDBSchema[] schemas = _root.getSchemas();
		for (int i = 0; i < schemas.length; i++) {
			if (schemas[i].getName().length() == 0) {
				continue;
			}
			rdh = executePrepareQuery(_SQL_COLUMN_DEF, new String[] { schema,
					table.getName() });

			for (int j = 0; j < rdh.getRowCount(); j++) {
				DBColumn col = new DBColumn(table);
				col.setName(rdh.getStringData(j, "COLUMN_NAME"));
				col.setDataType((short) rdh.getIntData(j, "DATA_TYPE"));
				col.setSize(rdh.getIntData(j, "COLUMN_SIZE"));
				col.setDecimalDigits(rdh.getIntData(j, "DECIMAL_DIGITS"));

				col.setNotNull("NO".equalsIgnoreCase(rdh.getStringData(j,
						"IS_NULLABLE")));
				// col.setPKey(pkey);
				col.setDataTypeString(rdh.getStringData(j, "TYPE_NAME"));
				col.setMaxColumnNameLength(_con.getMetaData()
						.getMaxColumnNameLength());

				table.putResource(col.getUName(), col);
			}
			rdh = executePrepareQuery(_SQL_COLUMN_PK, new String[] { schema,
					table.getName() });
			for (int j = 0; j < rdh.getRowCount(); j++) {
				IDBResource res = table.getResource(rdh.getStringData(j,
						"COLUMN_NAME"));
				if (res != null) {
					((IDBColumn) res).setPKey(true);
				}
			}
		}
	}

	@Override
	protected void getSequence() throws SQLException {
		ResultSetDataHolder rdh = null;
		IDBSchema[] schemas = _root.getSchemas();
		for (int i = 0; i < schemas.length; i++) {
			if (schemas[i].getName().length() == 0) {
				continue;
			}
			rdh = executePrepareQuery(_SELECT_ALL_SEQUENCE,
					new String[] { schemas[i].getName() });
			DBSequence sequence = null;
			for (int j = 0; j < rdh.getRowCount(); j++) {
				sequence = new DBSequence(schemas[i]);
				String name = rdh.getStringData(j, "SEQUENCE_NAME");
				sequence.setName(name);
				// String text = _SELECT_SEQUENCE_BODY + schemas[i].getName()
				// + "." + name;
				// rdh2 = executePrepareQuery(text, EMPTY);
				if (rdh.getRowCount() > 0) {
					setResourceProperties(sequence, 0, rdh);
				}
				schemas[i].putSequence(sequence);
				// setViewText(trigger);
			}

		}
	}

	protected String getSupportToken() {
		StringBuffer buf = new StringBuffer();
		Pattern p = Pattern.compile("\"(\\w+)\"");
		String[] str = WolfSQLParserConstants.tokenImage;
		for (int i = 0; i < str.length; i++) {
			Matcher m = p.matcher(str[i]);
			if (m.matches()) {
				buf.append(m.group(1)).append(",");
			}
		}
		if (buf.length() > 0) {
			buf.setLength(buf.length() - 1);
		}
		return buf.toString();
	}

	public boolean load(File f) {
		boolean ret = super.load(f);
		_root.setDefaultSchema((IDBSchema) _root.getResource("PUBLIC"));
		return ret;
	}

	private void setBindString(PreparedStatement st, int idx, ResultSet all_rs,
			IDBColumn column) throws SQLException {
		switch (column.getDataType()) {
		case Types.TINYINT:
		case Types.SMALLINT:
		case Types.INTEGER:
		case Types.BIGINT:
		case Types.NUMERIC:
		case Types.DECIMAL:
			st.setString(idx, all_rs.getString(idx));
			break;
		case Types.FLOAT:
		case Types.REAL:
		case Types.DOUBLE:
			st.setString(idx, all_rs.getString(idx));
			break;
		case Types.CHAR:
		case Types.VARCHAR:
		case Types.LONGVARCHAR:
		case Types.NCHAR:
		case Types.NVARCHAR:
		case Types.LONGNVARCHAR:
			st.setString(idx, all_rs.getString(idx));
			break;
		case Types.DATE:
			st.setDate(idx, all_rs.getDate(idx));
			break;
		case Types.TIME:
			st.setTime(idx, all_rs.getTime(idx));
			break;
		case Types.TIMESTAMP:
			st.setTimestamp(idx, all_rs.getTimestamp(idx));
			break;
		case Types.BIT:
		case Types.BOOLEAN:
			st.setBoolean(idx, all_rs.getBoolean(idx));
			break;
		case Types.BLOB:
		case Types.CLOB:
		case Types.BINARY:
		case Types.VARBINARY:
		case Types.LONGVARBINARY:
			st.setBytes(idx, all_rs.getBytes(idx));
			break;
		case Types.NULL:
		case Types.OTHER:
		case Types.JAVA_OBJECT:
		case Types.DISTINCT:
		case Types.STRUCT:
		case Types.ARRAY:
		case Types.REF:
		case Types.DATALINK:
		case Types.ROWID:
		case Types.NCLOB:
		case Types.SQLXML:

		}
	}

	public ResultSetDataHolder2 executeBatUpdate(String[] statements)
			throws SQLException {
		Statement statement = null;
		ResultSetDataHolder2 sdh = null;
		if (_con == null) {
			return sdh;
		}
		try {
			boolean[] ret = new boolean[statements.length];
			long time = System.currentTimeMillis();
			for (int i = 0; i < ret.length; i++) {
				ret[i] = execute(statements[i]);
			}

			time = System.currentTimeMillis() - time;

			sdh = new ResultSetDataHolder2(new String[] { "Returns",
					"Statement" }, null);
			for (int i = 0; i < ret.length; i++) {
				sdh.addRow(new String[] { String.valueOf(ret[i]), statements[i] });
			}

			sdh.setWrapTime(time);
			return sdh;
		} finally {
			setTransactionTime(true);

			if (statement != null) {
				statement.close();
			}
		}
	}

	public boolean executePrepareStatement(String sql_statement,
			ResultSet all_rs, IDBColumn[] columns) throws SQLException {
		PreparedStatement st = null;
		try {
			st = getPrepareStatement(_con, sql_statement);
			for (int i = 0; i < columns.length; i++) {
				setBindString(st, i + 1, all_rs, columns[i]);
			}
			return st.execute();
		} finally {
			if (st != null) {
				st.close();
			}
		}
	}

	@Override
	public boolean migration(ITransactionSQL osql, IDBSchema schema,
			boolean drop, boolean cascade) throws SQLException {
		fLogger.info("migration DBSchema " + schema);
		String sn = schema.getName();
		boolean existsSchema = existsSchema(sn);
		if (existsSchema && drop) {
			String st = String.format("DROP SCHEMA %s", sn);
			if (cascade) {
				st += " CASCADE";
			}
			executePrepare(st, EMPTY);
		}
		if (!existsSchema || drop) {
			String st = String.format("CREATE SCHEMA %s AUTHORIZATION DBA", sn);
			executePrepare(st, EMPTY);
		}
		commit();

		return true;
	}

	@Override
	public boolean migration(ITransactionSQL osql, IDBTable table,
			boolean drop, boolean cascade, boolean noSchema)
			throws SQLException {
		fLogger.info("migration DBTable " + table);
		ResultSetDataHolder rdh = null;
		String sn = table.getParent().getName();
		String tableName = sn + "." + table.getName();
		if (noSchema) {
			sn = StringUtil.EMPTY_STRING;
			tableName = table.getName();
		}
		boolean existsTable = existsTable(sn, table.getName());
		if (existsTable && drop) {
			String st = null;
			if (table.isTable()) {
				st = String.format("DROP TABLE %s", tableName);
				if (cascade) {
					st += " CASCADE";
				}
			} else {
				st = String.format("DROP VIEW %s", tableName);
			}
			executePrepare(st, EMPTY);
		}
		if (!existsTable || drop) {
			if (createTableBody(sn, table)) {

			} else {
				return false;
			}
		}
		if (!migrateTableValue(osql, table, noSchema)) {
			return false;
		}
		commit();

		return true;
	}

	private boolean createTableBody(String schemaName, IDBTable table)
			throws SQLException {
		StringBuilder buf = new StringBuilder();
		if (table.isTable()) {
			buf.append("CREATE TABLE ");
			if (schemaName.length() > 0) {
				buf.append(schemaName).append(".");
			}
			buf.append(table.getName());
			buf.append(" ( ").append(StringUtil.LINE_SEPARATOR);
			IDBColumn[] columns = table.getColumns();
			for (int j = 0; j < columns.length; j++) {
				buf.append(String.format("  %s,%n", getTypeString(columns[j])));
			}
			if (columns.length > 0) {
				buf.setLength(buf.length() - 1
						- StringUtil.LINE_SEPARATOR.length());
			}
			buf.append(StringUtil.LINE_SEPARATOR);
			if (table.hasPk()) {
				buf.append("  ,PRIMARY KEY (");
				int[] pk = table.getPkPositions();
				for (int j = 0; j < pk.length; j++) {
					if (j > 0) {
						buf.append(",");
					}
					buf.append(table.getColumns()[pk[j]].getName());
				}
				buf.append(")").append(StringUtil.LINE_SEPARATOR);
			}
			buf.append(")");
		} else {
			if (StringUtil.nvl(table.getText()).length() == 0) {
				return true;
			}
			String vt = table.getText().replaceFirst(
					"([\"a-zA-Z_]+[.])?\"?([a-zA-Z_]+)\"? *[(]", "$2 (");
			buf.append(vt);
			buf.append(StringUtil.LINE_SEPARATOR);
		}
		fLogger.info("=== CREATE TABLE ===");
		fLogger.info(buf.toString());
		executePrepare(buf.toString(), EMPTY);
		return true;
	}

	private boolean migrateTableValue(ITransactionSQL osql, IDBTable table,
			boolean noSchema) {
		StringBuilder buf = new StringBuilder();
		PreparedStatement st = null;
		ResultSet rs = null;
		fLogger.info("" + table);
		try {
			buf.append("SELECT * FROM ");
			buf.append(table.getParent().getName()).append(".");
			buf.append(table.getName());
			AbsTransactionSQL asql = (AbsTransactionSQL) osql;

			st = asql.getPrepareStatement(asql._con, buf.toString());
			rs = st.executeQuery();

			doInsertFromRs(rs, table.getColumns(), noSchema);
		} catch (SQLException e) {
			fLogger.warn(e);
			return false;
		} finally {
			try {
				if (rs != null) {
					rs.close();
				}
				if (st != null) {
					st.close();
				}
			} catch (SQLException e) {
				fLogger.warn(e);
				return false;
			}
		}
		return true;

	}

	private String getTypeString(IDBColumn column) {
		StringBuilder buf = new StringBuilder();
		buf.append(column.getName()).append(" ");
		int size = column.getSize();
		if (size <= 0) {
			size = Integer.MAX_VALUE;
		}

		switch (column.getDataType()) {
		case Types.CHAR:
		case Types.NCHAR:
			buf.append(String.format("CHAR(%d)", size));
			break;
		case Types.VARCHAR:
		case Types.NVARCHAR:
			buf.append(String.format("VARCHAR(%d)", size));
			break;
		case Types.NUMERIC:
			buf.append(String.format("NUMERIC(%d)", size));
			break;
		case Types.DECIMAL:
			buf.append(String.format("DECIMAL(%d)", size));
			break;
		case Types.INTEGER:
			buf.append("INTEGER");
			break;
		case Types.FLOAT:
			buf.append("FLOAT");
			break;
		case Types.DOUBLE:
			buf.append("DOUBLE");
			break;
		case Types.BIGINT:
			buf.append("BIGINT");
			break;
		case Types.DATE:
			buf.append("DATE");
			break;
		case Types.TIMESTAMP:
			buf.append("TIMESTAMP");
			break;
		case Types.BLOB:
			buf.append("BLOB");
			break;
		case Types.BINARY:
			buf.append("BINARY");
			break;
		case Types.STRUCT:
			buf.append(String.format("VARCHAR(%d)", Integer.MAX_VALUE));
			break;
		default:
		}
		if (column.isNotNull() && !column.isPkey()) {
			buf.append(" NOT NULL");
		}
		return buf.toString();
	}
}