package hiro.yoshioka.sql;

import hiro.yoshioka.ast.sql.util.BindInfo;
import hiro.yoshioka.sdh.ResultSetMetaCopy;
import hiro.yoshioka.sdh2.ReflectionPreparedStatement;
import hiro.yoshioka.sdh2.ResultSetDataHolder2;
import hiro.yoshioka.sql.engine.CallRequest;
import hiro.yoshioka.sql.engine.Request;
import hiro.yoshioka.sql.engine.SQLOperationType;
import hiro.yoshioka.sql.engine.TransactionRequest;
import hiro.yoshioka.sql.params.OutPutParameter;
import hiro.yoshioka.sql.params.Parameter;
import hiro.yoshioka.sql.resource.DBCrossRefference;
import hiro.yoshioka.sql.resource.IDBColumn;
import hiro.yoshioka.sql.resource.IDBResource;
import hiro.yoshioka.sql.resource.IDBSchema;
import hiro.yoshioka.sql.resource.IDBSequence;
import hiro.yoshioka.sql.resource.IDBTable;
import hiro.yoshioka.sql.util.SQLUtil;
import hiro.yoshioka.util.StringUtil;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.Driver;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;

import jp.sf.orangesignal.csv.CsvWriter;
import jp.sf.orangesignal.csv.handlers.ResultSetHandler;

public abstract class AbsTransactionSQL extends AbsBasicSQL implements
		ITransactionSQL {
	static final String savepointName = "mySavePoint";
	static final String savepointNameExtra = "myExtraSavePoint";

	private boolean fTrunsactionTime;

	protected static final int DEFAULT_ROW_NUM = 1000;

	protected Statement fActiveStatement;

	private int queryTimeOut;

	protected long maxRow;

	protected AbsTransactionSQL(Driver ds) {
		super(ds);
	}

	public abstract ResultSetDataHolder2 getSessionInfo() throws SQLException;

	public abstract ResultSetDataHolder2 getLockInfo() throws SQLException;

	public boolean doOperation(SQLOperationType Operation, Request request)
			throws SQLException {
		boolean retCode = false;
		fLogger.info("doOperation:" + Operation);
		TransactionRequest treq = null;
		if (request instanceof TransactionRequest) {
			treq = (TransactionRequest) request;
			queryTimeOut = treq.getTimeOut();
			maxRow = treq.getMaxRownum();
		}
		try {
			switch (Operation) {
			case EXECUTE_BAT:
				treq.setRDH(executeBatUpdate(treq.getSQLStatements()));
				break;
			case PREPARED_EXECUTE_QUERY:
				treq.setRDH(executePrepareQuery(treq.getSQLStatement(),
						treq.getBindStrings()));
				break;
			case EXECUTE:
				treq.setResult(execute(treq.getSQLStatement()));
				break;
			case PREPARED_CALL:
				executePrepareCall((CallRequest) request);
				break;
			case CREATE_TRIG_FNC_PROC:
			case PREPARED_EXECUTE:
				treq.setRDH(executePrepare(treq.getSQLStatement(),
						treq.getBindObjects()));
				break;

			case COUNT:
				if (StringUtil.isEmpty(treq.getSQLStatement())) {
					IDBTable table = treq.getIDBTable();
					IDBResource schema = table.getParent();

					if (schema.getName().length() > 0) {
						treq.setSQLStatement("SELECT COUNT(*) CNT FROM "
								+ schema.getName() + "." + table.getName());
					} else {
						treq.setSQLStatement("SELECT COUNT(*) CNT FROM "
								+ table.getName());
					}
				}
				treq.setResultCount(count(treq.getSQLStatement()));
				break;

			case COUNTS:
				treq.setRDH(counts(treq.getSQLStatements()));
				break;

			case CANSEL:
				retCode = cansel();
				break;
			case COMMIT:
				retCode = commit();
				break;
			case ROLLBACK:
				retCode = rollback();
				break;
			case SELECT_ALL:
				treq.setRDH(getAllData2(treq.getIDBTable()));
				break;
			default:
				retCode = super.doOperation(Operation, request);
				break;
			}
		} catch (SQLException e) {
			notifyExecute(_con, SQLExecutionStatus.EXCEPTION, e.getMessage());
			throw e;
		} finally {
			queryTimeOut = 0;
			maxRow = 0;
		}

		return retCode;
	}

	@Override
	public ResultSet getAllData(IDBTable table) throws SQLException {
		IDBResource schema = table.getParent();

		getStatement(_con, ResultSet.TYPE_FORWARD_ONLY,
				ResultSet.CONCUR_READ_ONLY);

		if (schema.getName().length() > 0) {
			return fActiveStatement.executeQuery("SELECT * FROM "
					+ schema.getName() + "." + table.getName());
		} else {
			return fActiveStatement.executeQuery("SELECT * FROM "
					+ table.getName());
		}
	}

	public ResultSetDataHolder2 getAllData2(IDBTable table) throws SQLException {
		IDBResource schema = table.getParent();
		String sql_statement = null;
		getStatement(_con, ResultSet.TYPE_FORWARD_ONLY,
				ResultSet.CONCUR_READ_ONLY);

		if (schema.getName().length() > 0) {
			sql_statement = "SELECT * FROM " + schema.getName() + "."
					+ table.getName();
		} else {
			sql_statement = "SELECT * FROM " + table.getName();
		}
		return RS2RDH(fActiveStatement.executeQuery(sql_statement), true,
				sql_statement, null);
	}

	public boolean executePrepareCall(CallRequest request) throws SQLException {
		CallableStatement statement = null;
		boolean ret = false;
		if (_con == null) {
			return ret;
		}
		try {
			if (fLogger.isTraceEnabled()) {
				fLogger.trace("sql=" + request.getSQLStatement());
			}
			statement = _con.prepareCall(request.getSQLStatement());
			Parameter[] inputs = request.getInputParameters();
			for (int i = 0; i < inputs.length; i++) {
				if (fLogger.isWarnEnabled()) {
					fLogger.warn("in[" + inputs[i].index + "]="
							+ inputs[i].value);
				}
				SQLUtil.setBinds(statement, inputs[i].index, inputs[i].value);
			}
			OutPutParameter[] outputs = request.getOutPutParameters();
			for (int i = 0; i < outputs.length; i++) {
				if (fLogger.isTraceEnabled()) {
					fLogger.trace("registerOutParameter[" + outputs[i].index
							+ "]=" + outputs[i].datatype);
				}
				statement.registerOutParameter(outputs[i].index,
						outputs[i].datatype);
			}

			ret = statement.execute();

			for (int i = 0; i < outputs.length; i++) {
				outputs[i].setResult(SQLUtil.getStringUsingType(statement,
						outputs[i].index, outputs[i].datatype));
			}
		} finally {
			if (statement != null) {
				statement.close();
			}
			setTransactionTime(true);
		}
		return ret;
	}

	/**
	 * @param operationCode
	 * @return
	 */
	public boolean canDoOperation(SQLOperationType operation) {
		switch (operation) {
		case CANSEL:
			return _con != null;
		case COMMIT:
			return fTrunsactionTime;
		case ROLLBACK:
			return super.canDoOperation(SQLOperationType.CLOSE);
		default:
			return super.canDoOperation(operation);
		}
	}

	protected PreparedStatement getPrepareStatement(Connection connection,
			String sql_statement) throws SQLException {
		PreparedStatement statement = connection
				.prepareStatement(sql_statement);

		fActiveStatement = statement;
		statement.setQueryTimeout(getQueryTimeout());
		statement.setMaxRows((int) maxRow);

		return statement;
	}

	protected PreparedStatement getPrepareStatement(Connection connection,
			String sql_statement, int scroll, int read) throws SQLException {
		PreparedStatement statement = connection.prepareStatement(
				sql_statement, scroll, read);
		fActiveStatement = statement;
		statement.setQueryTimeout(getQueryTimeout());
		statement.setMaxRows((int) maxRow);

		return statement;
	}

	protected Statement getStatement(Connection connection, int scroll, int read)
			throws SQLException {
		Statement statement = connection.createStatement(scroll, read);
		fActiveStatement = statement;
		statement.setQueryTimeout(getQueryTimeout());
		statement.setMaxRows((int) maxRow);

		return statement;
	}

	public ResultSetDataHolder2 executeBatUpdate(String[] statements)
			throws SQLException {
		Statement statement = null;
		ResultSetDataHolder2 sdh = null;
		if (_con == null) {
			return sdh;
		}
		try {
			statement = getStatement(_con, ResultSet.TYPE_FORWARD_ONLY,
					ResultSet.CONCUR_READ_ONLY);

			for (int i = 0; i < statements.length; i++) {
				fLogger.trace(statements[i]);
				statement.addBatch(statements[i]);
			}

			long time = System.currentTimeMillis();
			notifyExecute(_con, SQLExecutionStatus.BEFORE_EXECUTE);
			int[] ret = statement.executeBatch();
			time = System.currentTimeMillis() - time;
			notifyExecute(_con, SQLExecutionStatus.AFTER_EXECUTE,
					String.valueOf(time));

			sdh = new ResultSetDataHolder2(new String[] { "Returns",
					"Statement" }, null);
			for (int i = 0; i < ret.length; i++) {
				sdh.addRow(new String[] { String.valueOf(ret[i]), statements[i] });
			}

			sdh.setWrapTime(time);
			return sdh;
		} finally {
			setTransactionTime(true);

			if (statement != null) {
				statement.close();
			}
		}
	}

	public boolean execute(Connection connection, String sql_statement,
			String[] args) throws SQLException {
		PreparedStatement statement = null;
		boolean ret = false;
		if (connection == null) {
			return false;
		}
		try {
			statement = getPrepareStatement(connection, sql_statement);

			fLogger.trace(sql_statement);
			if (args != null) {
				for (int i = 0; i < args.length; i++) {
					statement.setString(i + 1, args[i]);
				}
			}
			notifyExecute(connection, SQLExecutionStatus.BEFORE_EXECUTE);
			long time = System.currentTimeMillis();
			ret = statement.execute();
			time = System.currentTimeMillis() - time;
			notifyExecute(connection, SQLExecutionStatus.AFTER_EXECUTE,
					Boolean.toString(ret), String.valueOf(time));
			return ret;
		} finally {
			if (statement != null) {
				statement.close();
			}
		}
	}

	/**
	 * @return
	 */
	private int getQueryTimeout() {
		return queryTimeOut;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see hiro.yoshioka.sql.ITransactionSQL#count(java.lang.String)
	 */
	public int count(String sql_statement) throws SQLException {
		PreparedStatement statement = null;
		ResultSet rs = null;
		try {
			int ret = -1;
			statement = getPrepareStatement(_con, sql_statement);
			fLogger.trace(sql_statement);
			notifyExecute(_con, SQLExecutionStatus.BEFORE_EXECUTE);
			try {
				rs = statement.executeQuery();
				rs.next();
				ret = rs.getInt(1);
			} catch (SQLException e) {
				fLogger.warn(e);
			}
			notifyExecute(_con, SQLExecutionStatus.AFTER_EXECUTE,
					StringUtil.EMPTY_STRING + ret, "0");
			return ret;
		} finally {
			if (rs != null) {
				rs.close();
			}
			if (statement != null) {
				statement.close();
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see hiro.yoshioka.sql.ITransactionSQL#counts(java.lang.String[])
	 */
	public ResultSetDataHolder2 counts(String[] sql_statements)
			throws SQLException {
		ResultSetDataHolder2 ret = new ResultSetDataHolder2(
				new String[] { "CNT" }, null);
		for (int i = 0; i < sql_statements.length; i++) {
			PreparedStatement statement = null;
			ResultSet rs = null;
			try {
				statement = getPrepareStatement(_con, sql_statements[i]);
				fLogger.trace(sql_statements[i]);
				notifyExecute(_con, SQLExecutionStatus.BEFORE_EXECUTE);
				try {
					rs = statement.executeQuery();
					rs.next();
					ret.addRow(new String[] { String.valueOf(rs.getInt(1)) });
				} catch (SQLException e) {
					fLogger.warn(e);
					ret.addRow(new String[] { "-1" });
				}
				notifyExecute(_con, SQLExecutionStatus.AFTER_EXECUTE,
						StringUtil.EMPTY_STRING + ret, "0");
			} finally {
				if (rs != null) {
					rs.close();
				}
				if (statement != null) {
					statement.close();
				}
			}
		}
		return ret;
	}

	public int count(Connection con, String sql_statement, String[] args)
			throws SQLException {
		PreparedStatement statement = null;
		ResultSet rs = null;
		try {
			sql_statement = "SELECT COUNT(*) CNT FROM (" + sql_statement + ")";
			statement = getPrepareStatement(con, sql_statement);
			fLogger.trace(sql_statement);
			for (int i = 0; i < args.length; i++) {
				statement.setString(i + 1, args[i]);
			}
			notifyExecute(con, SQLExecutionStatus.BEFORE_EXECUTE);
			rs = statement.executeQuery();
			rs.next();
			int ret = rs.getInt(1);
			notifyExecute(con, SQLExecutionStatus.AFTER_EXECUTE,
					StringUtil.EMPTY_STRING + ret, "0");
			return ret;
		} finally {
			if (rs != null) {
				rs.close();
			}
			if (statement != null) {
				statement.close();
			}
		}
	}

	public boolean execute(ReflectionPreparedStatement ref) throws SQLException {
		PreparedStatement statement = null;
		boolean ret = false;

		try {
			String st = ref.getStatement();
			statement = getPrepareStatement(_con, st);

			fLogger.trace("STATEMENT[" + st + "]");
			ref.setBinds(statement);

			notifyExecute(_con, SQLExecutionStatus.BEFORE_EXECUTE);
			long time = System.currentTimeMillis();
			ret = statement.execute();
			time = System.currentTimeMillis() - time;
			notifyExecute(_con, SQLExecutionStatus.AFTER_EXECUTE,
					Boolean.toString(ret), String.valueOf(time));
			setTransactionTime(true);
			return ret;
		} finally {
			if (statement != null) {
				statement.close();
			}
		}
	}

	public ResultSetDataHolder2 executeQueryForBlob(
			ReflectionPreparedStatement ref) throws SQLException {
		PreparedStatement statement = null;
		ResultSetDataHolder2 rdh = null;

		try {
			String st = ref.getStatement();
			statement = getPrepareStatement(_con, st);

			fLogger.trace("STATEMENT[" + st + "]");
			ref.setBinds(statement);

			notifyExecute(_con, SQLExecutionStatus.BEFORE_EXECUTE);
			long time = System.currentTimeMillis();
			ResultSet rs = statement.executeQuery();
			time = System.currentTimeMillis() - time;
			rdh = RS2RDH(rs, true, st, null);
			notifyExecute(_con, SQLExecutionStatus.AFTER_EXECUTE,
					String.valueOf(rdh.getRowCount()), String.valueOf(time));

		} finally {
			if (statement != null) {
				statement.close();
			}
		}
		return rdh;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see hiro.yoshioka.sql.ITransactionSQL#execute(java.lang.String)
	 */
	public boolean execute(String sql_statement) throws SQLException {
		return execute(_con, sql_statement, null);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see hiro.yoshioka.sql.ITransactionSQL#execute(java.lang.String,
	 * java.lang.String[])
	 */
	public boolean execute(String sql_statement, String[] args)
			throws SQLException {
		return execute(_con, sql_statement, args);
	}

	// --------------------------------------------------------------------------
	// --
	/**
	 * 引数のSQL文を実行して結果を返す。 <BR>
	 * 
	 * @param connection
	 *            コネクション
	 * @param sql_statement
	 *            SQL文
	 * @param arg1
	 *            ?にセットする文字
	 * @return ResultSetDataHolder 結果セット connectionがNullの場合はNULLを返す。
	 * @throws SQLException
	 *             例外情報
	 */
	// --------------------------------------------------------------------------
	// --
	public ResultSetDataHolder2 executePrepareQuery(Connection connection,
			String sql_statement, String arg1) throws SQLException {
		return executePrepareQuery(connection, sql_statement,
				new String[] { arg1 });
	}

	public ResultSetDataHolder2 executePrepareQuery(String sql_statement,
			String[] arg1) throws SQLException {
		return executePrepareQuery(_con, sql_statement, arg1);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see hiro.yoshioka.sql.ITransactionSQL#executePrepare(java.lang.String,
	 * java.lang.Object[])
	 */
	public ResultSetDataHolder2 executePrepare(String sql_statement,
			Object[] arg1) throws SQLException {
		return executePrepare(_con, sql_statement, arg1);
	}

	public ResultSetDataHolder2 executePrepare(Connection connection,
			String sql_statement, Object[] args) throws SQLException {
		PreparedStatement statement = null;
		int rs = 0;
		if (connection == null) {
			fLogger.info("connection is null...");
			return null;
		}
		try {
			statement = getPrepareStatement(connection, sql_statement);
			fLogger.trace(sql_statement);
			for (int i = 0; i < args.length; i++) {
				if (args[i] instanceof BindInfo) {
					BindInfo bind = (BindInfo) args[i];
					try {
						if (bind.isBlob()) {
							if (bind.getValue() == null
									|| bind.getStringValue().length() == 0) {
								statement.setNull(i + 1, Types.BLOB);
							} else {
								File file = (File) bind.getValue();
								FileInputStream in = new FileInputStream(file);
								byte[] bytes = new byte[(int) file.length()];
								in.read(bytes, 0, (int) file.length());
								in.close();
								System.out.println("[" + (i + 1) + "] len["
										+ bytes.length + "]");
								statement.setBytes(i + 1, bytes);
								statement.setBinaryStream(i + 1, in,
										file.length());

							}
						} else if (bind.isBinary()) {
							if (bind.getValue() == null
									|| bind.getStringValue().length() == 0) {
								statement.setNull(i + 1, Types.BINARY);
							} else {
								File file = (File) bind.getValue();
								FileInputStream in = new FileInputStream(file);
								byte[] bytes = new byte[(int) file.length()];
								in.read(bytes, 0, (int) file.length());
								in.close();
								statement.setBytes(i + 1, bytes);
							}
						} else if (bind.isTimeStamp()) {
							if (bind.getValue() == null
									|| bind.getStringValue().length() == 0) {
								statement.setNull(i + 1, Types.TIMESTAMP);
							} else {
								statement.setTimestamp(i + 1, SQLUtil
										.getTimeStamp(bind.getStringValue()));
							}
						} else if (bind.isDate()) {
							if (bind.getValue() == null
									|| bind.getStringValue().length() == 0) {
								statement.setNull(i + 1, Types.DATE);
							} else {
								statement.setDate(i + 1,
										SQLUtil.getDate(bind.getStringValue()));
							}
						} else if (bind.isTime()) {
							if (bind.getValue() == null
									|| bind.getStringValue().length() == 0) {
								statement.setNull(i + 1, Types.TIME);
							} else {
								statement.setTime(i + 1,
										SQLUtil.getTime(bind.getStringValue()));
							}
						} else if (bind.isNumeric()) {
							if (bind.getValue() == null
									|| bind.getStringValue().length() == 0) {
								statement.setNull(i + 1, Types.NUMERIC);
							} else {
								statement.setBigDecimal(i + 1, new BigDecimal(
										bind.getStringValue()));
							}
						} else {
							statement.setString(i + 1, bind.getStringValue());
						}

					} catch (Throwable e) {
						fLogger.warn(e);
					}

				} else {
					statement.setString(i + 1, (String) args[i]);
				}
			}
			notifyExecute(connection, SQLExecutionStatus.BEFORE_EXECUTE);
			long time = System.currentTimeMillis();
			rs = statement.executeUpdate();
			time = System.currentTimeMillis() - time;
			notifyExecute(connection, SQLExecutionStatus.AFTER_EXECUTE,
					String.valueOf(rs), String.valueOf(time));

			ResultSetDataHolder2 rdh = new ResultSetDataHolder2(
					new String[] { "UpdateCount" }, null);
			rdh.addRow(new String[] { String.valueOf(rs) });
			rdh.setWrapTime(time);
			setTransactionTime(true);
			return rdh;
		} finally {
			if (statement != null) {
				statement.close();
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see hiro.yoshioka.sql.ITransactionSQL#setMaxRowNum(int)
	 */
	public void setMaxRowNum(int max) {
		maxRow = max;
	}

	// --------------------------------------------------------------------------
	// --
	/**
	 * 引数のSQL文を実行して結果を返す。 <BR>
	 * 
	 * @param connection
	 *            コネクション
	 * @param sql_statement
	 *            SQL文
	 * @param args
	 *            ?にセットする文字
	 * @return ResultSetDataHolder 結果セット connectionがNullの場合はNULLを返す。
	 * @throws SQLException
	 *             例外情報
	 */
	// --------------------------------------------------------------------------
	// --
	public ResultSetDataHolder2 executePrepareQuery(Connection connection,
			String sql_statement, String[] args) throws SQLException {
		PreparedStatement statement = null;
		ResultSet rs = null;
		if (connection == null) {
			return null;
		}
		try {

			statement = getPrepareStatement(connection, sql_statement);
			fLogger.trace(sql_statement);
			if (args != null) {
				for (int i = 0; i < args.length; i++) {
					statement.setString(i + 1, args[i]);
				}

			}
			notifyExecute(connection, SQLExecutionStatus.BEFORE_EXECUTE);
			long time = System.currentTimeMillis();
			rs = statement.executeQuery();
			time = System.currentTimeMillis() - time;

			ResultSetDataHolder2 rdh = RS2RDH(rs, true, sql_statement, args);
			notifyExecute(connection, SQLExecutionStatus.AFTER_EXECUTE,
					String.valueOf(rdh.getRowCount()), String.valueOf(time));

			rdh.setWrapTime(time);
			return rdh;
		} finally {
			if (statement != null) {
				statement.close();
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see hiro.yoshioka.sql.ITransactionSQL#saveCsv(java.lang.String,
	 * jp.sf.orangesignal.csv.handlers.ResultSetHandler,
	 * jp.sf.orangesignal.csv.CsvWriter)
	 */
	public boolean saveCsv(String sql_statement, ResultSetHandler handler,
			CsvWriter writer) throws SQLException {
		PreparedStatement statement = null;
		ResultSet rs = null;
		if (_con == null) {
			return false;
		}
		try {
			notifyExecute(_con, SQLExecutionStatus.BEFORE_EXECUTE);
			statement = _con
					.prepareStatement(sql_statement,
							ResultSet.TYPE_SCROLL_SENSITIVE,
							ResultSet.CONCUR_READ_ONLY);
			fLogger.trace(sql_statement);
			long time = System.currentTimeMillis();
			rs = statement.executeQuery();
			time = System.currentTimeMillis() - time;

			notifyExecute(_con, SQLExecutionStatus.AFTER_EXECUTE, "0",
					String.valueOf(time));
			try {
				handler.save(rs, writer);
				writer.flush();
				writer.close();
				return true;
			} catch (IOException e) {
				fLogger.error(e);
			}
			return false;
		} finally {
			if (rs != null) {
				rs.close();
			}
			if (statement != null) {
				statement.close();
			}
		}
	}

	public void executePrepareQueryByLine(String sql_statement,
			LineListener listner) throws SQLException {
		PreparedStatement statement = null;
		ResultSet rs = null;
		if (_con == null) {
			return;
		}
		try {
			notifyExecute(_con, SQLExecutionStatus.BEFORE_EXECUTE);

			statement = _con
					.prepareStatement(sql_statement,
							ResultSet.TYPE_SCROLL_SENSITIVE,
							ResultSet.CONCUR_READ_ONLY);
			fLogger.trace(sql_statement);

			long time = System.currentTimeMillis();

			rs = statement.executeQuery();

			listner.first(rs);
			rs.close();

			rs = statement.executeQuery();

			time = System.currentTimeMillis() - time;

			for (int i = 0; rs.next(); i++) {
				listner.next(rs);
			}
			notifyExecute(_con, SQLExecutionStatus.AFTER_EXECUTE, "0",
					String.valueOf(time));
		} finally {
			if (rs != null) {
				rs.close();
			}
			if (statement != null) {
				statement.close();
			}
		}
	}

	// --------------------------------------------------------------------------
	// --
	/*
	 * (non-Javadoc)
	 * 
	 * @see hiro.yoshioka.sql.ITransactionSQL#cansel()
	 */
	// --------------------------------------------------------------------------
	// --
	public boolean cansel() throws SQLException {
		boolean status = true;
		if (fActiveStatement != null) {
			fActiveStatement.cancel();
			fActiveStatement = null;
		}

		return true;
	}

	public boolean closeActiveStatement() throws SQLException {
		if (fActiveStatement != null) {
			fActiveStatement.close();
			fActiveStatement = null;
		}

		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see hiro.yoshioka.sql.ITransactionSQL#commit()
	 */
	public boolean commit() {
		try {
			_con.commit();
			setTransactionTime(false);
			for (int i = 0; i < fTransactionListenerList.size(); i++) {
				try {
					fTransactionListenerList.get(i).commited();
				} catch (RuntimeException e) {
					fLogger.error(e);
				}
			}
		} catch (SQLException e) {
			e.printStackTrace();
			return false;
		}
		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see hiro.yoshioka.sql.ITransactionSQL#rollback()
	 */
	public boolean rollback() {
		try {
			_con.rollback();
			for (int i = 0; i < fTransactionListenerList.size(); i++) {
				try {
					fTransactionListenerList.get(i).rollbacked();
				} catch (RuntimeException e) {
					fLogger.error(e);
				}
			}
			setTransactionTime(false);
		} catch (SQLException e) {
			e.printStackTrace();
			return false;
		}
		return true;
	}

	protected void setTransactionTime(boolean f) {
		boolean bf = fTrunsactionTime;
		fTrunsactionTime = f;
		if (bf != fTrunsactionTime) {
			for (int i = 0; i < fTransactionListenerList.size(); i++) {
				try {
					if (fTrunsactionTime) {
						fTransactionListenerList.get(i).transactionStarted();
					} else {
						fTransactionListenerList.get(i).transactionEnded();
					}
				} catch (RuntimeException e) {
					fLogger.error(e);
				}
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see hiro.yoshioka.sql.ITransactionSQL#trunsactionTime()
	 */
	public boolean trunsactionTime() {
		return fTrunsactionTime;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see hiro.yoshioka.sql.ITransactionSQL#getTableMetaCopy(java.lang.String)
	 */
	public ResultSetMetaCopy getTableMetaCopy(String name) {
		return getTableMetaCopy(name, StringUtil.EMPTY_STRING_ARRAY);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see hiro.yoshioka.sql.ITransactionSQL#getTableMetaCopy(java.lang.String,
	 * java.lang.String[])
	 */
	public ResultSetMetaCopy getTableMetaCopy(String name, String[] strings) {

		Statement st = null;
		try {
			st = _con.createStatement();

			StringBuffer selection = new StringBuffer("SELECT ");
			if (strings.length <= 0) {
				selection.append("*");
			} else {
				for (int i = 0; i < strings.length; i++) {
					selection.append(strings[i]).append(",");
				}
				selection.setLength(selection.length() - 1);
			}

			selection.append(" FROM ").append(name).append(" WHERE 1 = 2");

			ResultSet rs = st.executeQuery(selection.toString());

			ResultSetMetaData meta = rs.getMetaData();

			ResultSetMetaCopy copy = new ResultSetMetaCopy(meta);

			rs.close();

			return copy;
		} catch (SQLException e) {
			e.printStackTrace();
		} finally {
			if (st != null) {
				try {
					st.close();
				} catch (SQLException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
		}

		return null;
	}

	public int[] getSQLTypes(String select_statement) {

		Statement st = null;
		try {
			st = _con.createStatement();
			st.setMaxRows(1);
			ResultSet rs = st.executeQuery(select_statement);

			ResultSetMetaData meta = rs.getMetaData();

			int[] ret = new int[meta.getColumnCount()];

			for (int i = 1; i <= meta.getColumnCount(); i++) {
				ret[i - 1] = meta.getColumnType(i);
			}
			rs.close();

			return ret;
		} catch (SQLException e) {
			e.printStackTrace();
		} finally {
			if (st != null) {
				try {
					st.close();
				} catch (SQLException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
		}

		return null;
	}

	public boolean migration(ITransactionSQL osql, IDBSchema original,
			boolean drop, boolean cascade) throws SQLException {
		return false;
	}

	public boolean migration(ITransactionSQL osql, DBCrossRefference original,
			boolean drop, boolean cascade) throws SQLException {
		return false;
	}

	public boolean migration(ITransactionSQL osql, IDBTable original,
			boolean drop, boolean cascade, boolean noSchema)
			throws SQLException {
		return false;
	}

	public boolean migration(ITransactionSQL osql, IDBSequence original,
			boolean drop, boolean cascade, boolean noSchema)
			throws SQLException {
		return false;
	}

	private void setBindString(PreparedStatement st, int idx, ResultSet all_rs,
			IDBColumn column) throws SQLException {
		switch (column.getDataType()) {
		case Types.TINYINT:
		case Types.SMALLINT:
		case Types.INTEGER:
		case Types.BIGINT:
		case Types.NUMERIC:
		case Types.DECIMAL:
			st.setObject(idx, all_rs.getObject(idx), column.getDataType());
			break;
		case Types.FLOAT:
		case Types.REAL:
		case Types.DOUBLE:
			st.setObject(idx, all_rs.getObject(idx), column.getDataType());
			break;
		case Types.CHAR:
		case Types.VARCHAR:
		case Types.LONGVARCHAR:
		case Types.NCHAR:
		case Types.NVARCHAR:
		case Types.LONGNVARCHAR:
			st.setString(idx, all_rs.getString(idx));
			break;
		case Types.DATE:
			st.setDate(idx, all_rs.getDate(idx));
			break;
		case Types.TIME:
			st.setTime(idx, all_rs.getTime(idx));
			break;
		case Types.TIMESTAMP:
			st.setTimestamp(idx, all_rs.getTimestamp(idx));
			break;
		case Types.BIT:
		case Types.BOOLEAN:
			st.setBoolean(idx, all_rs.getBoolean(idx));
			break;
		case Types.BLOB:
		case Types.CLOB:
		case Types.BINARY:
		case Types.VARBINARY:
		case Types.LONGVARBINARY:
			st.setBytes(idx, all_rs.getBytes(idx));
			break;
		case Types.NULL:
		case Types.OTHER:
		case Types.JAVA_OBJECT:
		case Types.DISTINCT:
		case Types.STRUCT:
		case Types.ARRAY:
		case Types.REF:
		case Types.DATALINK:
		case Types.ROWID:
		case Types.NCLOB:
		case Types.SQLXML:
			st.setObject(idx, all_rs.getObject(idx), column.getDataType());
			break;

		}
	}

	public boolean doInsertFromRs(ResultSet all_rs, IDBColumn[] columns,
			boolean noSchema) throws SQLException {
		if (columns.length == 0) {
			return false;
		}
		PreparedStatement st = null;
		StringBuilder buf = new StringBuilder();
		buf.append("INSERT INTO ");
		IDBResource it = columns[0].getParent();
		IDBResource is = it.getParent();
		if (is != null && !noSchema) {
			if (is.getName().length() > 0) {
				buf.append(is.getName()).append(".");
			}
		}
		buf.append(it.getName()).append(" ");
		buf.append("(");
		for (int i = 0; i < columns.length; i++) {
			if (i > 0) {
				buf.append(",");
			}
			buf.append(columns[i].getName());
		}
		buf.append(")").append(StringUtil.LINE_SEPARATOR);
		buf.append("VALUES (");
		for (int i = 0; i < columns.length; i++) {
			if (i > 0) {
				buf.append(",");
			}
			buf.append("?");
		}
		buf.append(")");
		while (all_rs.next()) {
			try {
				st = getPrepareStatement(_con, buf.toString());
				for (int i = 0; i < columns.length; i++) {
					setBindString(st, i + 1, all_rs, columns[i]);
				}
				st.execute();
			} finally {
				if (st != null) {
					st.close();
				}
			}
		}
		return true;
	}

}