/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    server_c.c - TiMidity server written by Masanao Izumo <mo@goice.co.jp>
        Mon Apr 5 1999: Initial created.


    Launch TiMidity server: (example)
    % timidity -ir 7777

    Protcol note:
    The protocol is based on OSS interface.
    TiMidity server has 2 TCP/IP connection.  They are control port and
    data port.
    Control port:
	ASCII text protocol like FTP control port. See command_help for
	control protocol in this source code.
    Data port:
	One way Binary stream data from client to server. The format of
	stream is same as OSS sequencer stream.


    TODO:
	Protocol specification to be documented.
    */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#include <signal.h>

#ifdef HAVE_SYS_SOUNDCARD_H
#include <sys/soundcard.h>
#else
#include "server_defs.h"
#endif /* HAVE_SYS_SOUNDCARD_H */

#include "timidity.h"
#include "common.h"
#include "controls.h"
#include "instrum.h"
#include "playmidi.h"
#include "readmidi.h"
#include "recache.h"
#include "output.h"
#include "aq.h"
#include "timer.h"

#include "server_c.h"
#include "server_c_prv.h"

/* #define DEBUG_DUMP_SEQ 1 */
#define MIDI_COMMAND_PER_SEC	100
#define DEFAULT_LOW_TIMEAT	0.4
#define DEFAULT_HIGH_TIMEAT	0.6
#define DONT_STOP_AUDIO	1
#define DEFAULT_TIMEBASE	100 /* HZ? */
#define MAXTICKDIFF		150
#define SIG_TIMEOUT_SEC		3


static int cmd_help(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv);
static int cmd_open(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv);
static int cmd_close(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv);
static int cmd_timebase(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv);
static int cmd_reset(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv);
static int cmd_patch(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv);
static int cmd_quit(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv);
static int cmd_queue(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv);
static int cmd_maxqueue(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv);
static int cmd_time(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv);
static int cmd_nop(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv);
static int cmd_autoreduce(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv);
static int cmd_setbuf(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv);

struct
{
    char *cmd, *help;
    int minarg, maxarg;
    int (* proc)(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv);  /* argv[0] is command name
					   * argv[1..argc-1] is the arg.
					   * return: 0=success, -1=fatal-error,
					   *         1=connection-closed
					   */
} cmd_table[] =
{
    {"HELP",
	"HELP\tDisplay this message",
	1, 1, cmd_help},
    {"OPEN",
	"OPEN {lsb|msb}\n"
	"\tOpen data connection\n"
	"\tlsb: The byte order of data stream is LSB\n"
	"\tmsb: The byte order of data stream is MSB",
	2, 2, cmd_open},
    {"CLOSE",
	"CLOSE\tShutdown current data connection",
	1, 1, cmd_close},
    {"TIMEBASE",
	"TIMEBASE [timebase]\n\tSet time base",
	1, 2, cmd_timebase},
    {"RESET",
	"RESET\tInitialize all of MIDI status",
	1, 1, cmd_reset},
    {"PATCH",
	"PATCH {drumset|bank} <bank-no> <prog-no>\n\tLoad specified patch",
	4, 4, cmd_patch},
    {"QUIT",
	"QUIT\tClose control connection",
	1, 1, cmd_quit},
    {"QUEUE",
	"QUEUE\tTiMidity tells the time of audio buffer queue in second",
	1, 1, cmd_queue},
    {"MAXQUEUE",
	"MAXQUEUE\n"
	"\tTiMidity tells the maxmum time of audio buffer queue in second",
	1, 1, cmd_maxqueue},
    {"TIME",
	"TIME\tTiMidity tells the current playing time in second",
	1, 1, cmd_time},
    {"NOP",
	"NOP\tDo nothing",
	1, 1, cmd_nop},
    {"AUTOREDUCE",
	"AUTOREDUCE {on|off} [msec]\n\tEnable/Disable auto voice reduction",
	2, 3, cmd_autoreduce},
    {"SETBUF",
	"SETBUF low hi\n\tSpecify low/hi sec of buffer queue",
	3, 3, cmd_setbuf},

    {NULL, NULL, 0, 0, NULL} /* terminate */
};

/*
TEMPO [<tempo>]\n\
	Change the tempo.  If the argument is omitted, TiMidity tells the\n\
	current tempo.\n\
KEYSHIFT <{+|-}offs>\n\
	Change the base key. (0 to reset)\n\
SPEED <{+|-}offs>\n\
	Change the play speed. (0 to reset)\n\
MODE {gs|xg|gm}\n\
	Specify default MIDI system mode\n\
SYNTH [gus|awe]\n\
	Specify Synth type emulation. (no argument to default)\n\
*/


static int ctl_open(tmdy_struct_ex_t *tmdy_struct, int using_stdin, int using_stdout);
static void ctl_close(tmdy_struct_ex_t *tmdy_struct);
static int ctl_read(tmdy_struct_ex_t *tmdy_struct, int32 *valp);
static int cmsg(tmdy_struct_ex_t *tmdy_struct, int type, int verbosity_level, char *fmt, ...);
static void ctl_event(tmdy_struct_ex_t *tmdy_struct, CtlEvent *e);
static void ctl_pass_playing_list(tmdy_struct_ex_t *tmdy_struct, int n, char *args[]);

/**********************************/
/* export the interface functions */

#define ctl server_control_mode

ControlMode ctl=
{
    "remote interface", 'r',
    1,0,0,
    0,
    ctl_open,
    ctl_close,
    ctl_pass_playing_list,
    ctl_read,
    cmsg,
    ctl_event
};

/*
struct fd_read_buffer
{
    char buff[BUFSIZ];
    * count: beginning of read pointer
     * size:  end of read pointer
     * fd:    file descripter for input
     *
    int count, size, fd;
};
*/
static int fdgets(tmdy_struct_ex_t *tmdy_struct, char *buff, size_t buff_size, struct fd_read_buffer *p);
static int fdputs(char *s, int fd);
static uint32 data2long(tmdy_struct_ex_t *tmdy_struct, uint8* data);
static uint16 data2short(tmdy_struct_ex_t *tmdy_struct, uint8* data);
static int do_control_command_nonblock(tmdy_struct_ex_t *tmdy_struct);

/*
static struct fd_read_buffer control_fd_buffer;

static int data_fd = -1, control_fd = -1;
static int data_port, SRV->control_port;
static int SRV->is_lsb_data = 1;
static int curr_timebase = DEFAULT_TIMEBASE;
static int32 sample_correction;
static int32 sample_increment;
static int32 sample_cum;
static int32 curr_event_samples, event_time_offset;
static int32 curr_tick, SRV->tick_offs;
static double start_time;
static int tmr_running;

static int is_system_prefix = 0;
static struct sockaddr_in control_client;
static double low_time_at = 0.3;
static double high_time_at = 0.5;
static FILE *outfp;
*/
/*ARGSUSED*/
static int ctl_open(tmdy_struct_ex_t *tmdy_struct, int using_stdin, int using_stdout)
{
    ctl.opened = 1;
    ctl.flags &= ~(CTLF_LIST_RANDOM|CTLF_LIST_SORT);
    if(using_stdout)
	SRV->outfp = stderr;
    else
	SRV->outfp = stdout;
    return 0;
}

static void ctl_close(tmdy_struct_ex_t *tmdy_struct)
{
    if(!ctl.opened)
	return;
    if(SRV->data_fd != -1)
    {
	close(SRV->data_fd);
	SRV->data_fd = -1;
    }
    if(SRV->control_fd != -1)
    {
	close(SRV->control_fd);
	SRV->control_fd = -1;
    }
}

/*ARGSUSED*/
static int ctl_read(tmdy_struct_ex_t *tmdy_struct, int32 *valp)
{
    if(SRV->data_fd != -1)
	do_control_command_nonblock(tmdy_struct);
    return RC_NONE;
}

static int cmsg(tmdy_struct_ex_t *tmdy_struct, int type, int verbosity_level, char *fmt, ...)
{
    va_list ap;

    if((type==CMSG_TEXT || type==CMSG_INFO || type==CMSG_WARNING) &&
       ctl.verbosity < verbosity_level)
	return 0;

    if(SRV->outfp == NULL)
	SRV->outfp = stderr;

    va_start(ap, fmt);
    vfprintf(SRV->outfp, fmt, ap);
    fputs(NLS, SRV->outfp);
    fflush(SRV->outfp);
    va_end(ap);

    return 0;
}

static void ctl_event(tmdy_struct_ex_t *tmdy_struct, CtlEvent *e)
{
}

static int pasv_open(tmdy_struct_ex_t *tmdy_struct, int *port)
{
    int sfd;
    struct sockaddr_in server;

    if((sfd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
    {
	perror("socket");
	return -1;
    }

    memset(&server, 0, sizeof(server));
    server.sin_port        = htons(*port);
    server.sin_family      = AF_INET;
    server.sin_addr.s_addr = htonl(INADDR_ANY);

#ifdef SO_REUSEADDR
    {
	int on = 1;
	setsockopt(sfd, SOL_SOCKET, SO_REUSEADDR, (caddr_t)&on, sizeof(on));
    }
#endif /* SO_REUSEADDR */

    ctl.cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG, "Bind TCP/IP port=%d", *port);
    if(bind(sfd, (struct sockaddr *)&server, sizeof(server)) < 0)
    {
	perror("bind");
	close(sfd);
	return -1;
    }
    if(*port == 0)
    {
	int len = sizeof(server);
	if(getsockname(sfd, (struct sockaddr *)&server, &len) < 0)
	{
	    perror("getsockname");
	    close(sfd);
	    return -1;
	}
	*port = ntohs(server.sin_port);
    }

    /* Set it up to wait for connections. */
    if(listen(sfd, 1) < 0)
    {
	perror("listen");
	close(sfd);
	return -1;
    }

    return sfd;
}

static RETSIGTYPE sig_timeout(int sig)
{
    signal(SIGALRM, sig_timeout); /* For SysV base */
    /* Expect EINTR */
}

static void doit(tmdy_struct_ex_t *tmdy_struct);
static int send_status(tmdy_struct_ex_t *tmdy_struct, int status, char *message, ...);
static void compute_sample_increment(tmdy_struct_ex_t *tmdy_struct);
static void server_reset(tmdy_struct_ex_t *tmdy_struct);

static void ctl_pass_playing_list(tmdy_struct_ex_t *tmdy_struct, int n, char *args[])
{
    int sock;

    if(n != 2 && n != 1)
    {
	fprintf(stderr, "Usage: timidity -ir control-port [data-port]\n");
	return;
    }

#ifdef SIGPIPE
    signal(SIGPIPE, SIG_IGN);    /* Handle broken pipe */
#endif /* SIGPIPE */

    SRV->control_port = atoi(args[0]);
    if(n == 2)
	SRV->data_port = atoi(args[1]);
    else
	SRV->data_port = 0;

    if (SRV->control_port) {
	sock = pasv_open(tmdy_struct, &SRV->control_port);
	if(sock == -1)
	    return;
    }
    TMDY_PLAYMIDI->opt_realtime_playing = 1; /* Enable loading patch while playing */
    TMDY_RECACHE->allocate_cache_size = 0; /* Don't use pre-calclated samples */
/*  (TMDY_AQ->aq_set_soft_queue)(tmdy_struct, -1.0, 0.0); */
    alarm(0);
    signal(SIGALRM, sig_timeout);

    TMDY_OUTPUT->play_mode->close_output(tmdy_struct);
    while(1)
    {
	int addrlen;

	addrlen = sizeof(SRV->control_client);
	memset(&SRV->control_client, 0, addrlen);

	if (SRV->control_port) {
	    if((SRV->control_fd = accept(sock,
				(struct sockaddr *)&SRV->control_client,
				&addrlen)) < 0)
	    {
		if(errno == EINTR)
		    continue;
		perror("accept");
		close(sock);
		return;
	    }
	}
	else SRV->control_fd = 0;

	if(TMDY_OUTPUT->play_mode->open_output(tmdy_struct) < 0)
	{
	    ctl.cmsg(tmdy_struct, CMSG_FATAL, VERB_NORMAL,
		     "Couldn't open %s (`%c')",
		     TMDY_OUTPUT->play_mode->id_name, TMDY_OUTPUT->play_mode->id_character);
	    send_status(tmdy_struct, 510, "Couldn't open %s (`%c')",
			TMDY_OUTPUT->play_mode->id_name, TMDY_OUTPUT->play_mode->id_character);
	    if (SRV->control_port) {
		close(SRV->control_fd);
		SRV->control_fd = 1;
	    }
	    continue;
	}
	
	server_reset(tmdy_struct);

	ctl.cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY, "Connected");
	doit(tmdy_struct);
	ctl.cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY, "Connection closed");

	TMDY_OUTPUT->play_mode->close_output(tmdy_struct);

	if(SRV->control_fd != -1 && SRV->control_port)
	{
	    close(SRV->control_fd);
	    SRV->control_fd = -1;
	}
	if(SRV->data_fd != -1)
	{
	    close(SRV->data_fd);
	    SRV->data_fd = -1;
	}
	(TMDY_INSTRUM->free_instruments)(tmdy_struct, 0);
	TMDY_UTILS->mblock->free_global_mblock(tmdy_struct);
	if (!SRV->control_port)
	    break;
    }
}

#define MAX_GETCMD_PARAMS 8
/* return:
 * 0: success
 * 1: error
 *-1: fatal error (will be close the connection)
 */
static int control_getcmd(tmdy_struct_ex_t *tmdy_struct, char **params, int *nparams)
{
//    static char buff[BUFSIZ];
    int n;

    /* read line */
    n = fdgets(tmdy_struct, SRV->buff, sizeof(SRV->buff), &SRV->control_fd_buffer);
    if(n == -1)
    {
	perror("read");
	return -1;
    }
    if(n == 0)
	return 1;
    if((params[0] = strtok(SRV->buff, " \t\r\n\240")) == NULL)
	return 0;
    *nparams = 0;
    while(params[*nparams] && *nparams < MAX_GETCMD_PARAMS)
	params[++(*nparams)] = strtok(NULL," \t\r\n\240");
    return 0;
}

static int send_status(tmdy_struct_ex_t *tmdy_struct, int status, char *message, ...)
{
    va_list ap;
    char buff[BUFSIZ];

    va_start(ap, message);
    sprintf(buff, "%03d ", status);
    vsnprintf(buff + 4, sizeof(buff) - 5, message, ap);
    va_end(ap);
    strcat(buff, "\n");
    if(write(SRV->control_fd, buff, strlen(buff)) == -1)
	return -1;
    return 0;
}

static void seq_play_event(tmdy_struct_ex_t *tmdy_struct, MidiEvent *ev)
{
    if(SRV->tmr_running)
	ev->time = SRV->curr_event_samples;
    else
    {
	if(IS_STREAM_TRACE)
	{
	    SRV->event_time_offset += TMDY_OUTPUT->play_mode->rate / MIDI_COMMAND_PER_SEC;
	    ev->time = SRV->curr_event_samples;
	}
	else
	{
	    double past_time = TMDY_UTILS->timer->get_current_calender_time(tmdy_struct) - SRV->start_time;
	    if(TMDY_OUTPUT->play_mode->flag & PF_PCM_STREAM)
		past_time += SRV->high_time_at;
	    ev->time = (int32)(past_time * TMDY_OUTPUT->play_mode->rate);
	}
    }
    ev->time += SRV->event_time_offset;
    TMDY_PLAYMIDI->play_event(tmdy_struct, ev);
}

static void tmr_reset(tmdy_struct_ex_t *tmdy_struct)
{
    SRV->curr_event_samples =
	SRV->event_time_offset =
	    SRV->sample_cum = 0;
    TMDY_PLAYMIDI->playmidi_tmr_reset(tmdy_struct);
    SRV->curr_timebase = DEFAULT_TIMEBASE;
    SRV->curr_tick = SRV->tick_offs = 0;
    SRV->start_time = TMDY_UTILS->timer->get_current_calender_time(tmdy_struct);
}

static void compute_sample_increment(tmdy_struct_ex_t *tmdy_struct)
{
    double a;
    a = (double)TMDY_PLAYMIDI->current_play_tempo * (double)TMDY_OUTPUT->play_mode->rate
	* (65536.0/500000.0) / (double)SRV->curr_timebase,
    SRV->sample_correction = (int32)(a) & 0xFFFF;
    SRV->sample_increment = (int32)(a) >> 16;
}

static void add_tick(tmdy_struct_ex_t *tmdy_struct, int tick)
{
    int32 samples_to_do;
    MidiEvent ev;

    samples_to_do = SRV->sample_increment * tick;
    SRV->sample_cum += SRV->sample_correction * tick;
    if(SRV->sample_cum & 0xFFFF0000)
    {
	samples_to_do += ((SRV->sample_cum >> 16) & 0xFFFF);
	SRV->sample_cum &= 0x0000FFFF;
    }
    SRV->curr_event_samples += samples_to_do;
    SRV->curr_tick += tick;
    ev.type = ME_NONE;
    seq_play_event(tmdy_struct, &ev);
}

static int tick2sample(tmdy_struct_ex_t *tmdy_struct, int tick)
{
    int32 samples, cum;

    samples = SRV->sample_increment * tick;
    cum = SRV->sample_correction * tick;
    if(cum & 0xFFFF0000)
	samples += ((SRV->sample_cum >> 16) & 0xFFFF);
    return samples;
}

int time2tick(tmdy_struct_ex_t *tmdy_struct, double sec)
{
    return (int)(sec * SRV->curr_timebase);
}


static void stop_playing(tmdy_struct_ex_t *tmdy_struct)
{
    if(TMDY_PLAYMIDI->upper_voices)
    {
	MidiEvent ev;
	ev.type = ME_EOT;
	ev.a = 0;
	ev.b = 0;
	seq_play_event(tmdy_struct, &ev);
	(TMDY_AQ->aq_flush)(tmdy_struct, 0);
    }
}

static int do_control_command(tmdy_struct_ex_t *tmdy_struct);
static int do_control_command_nonblock(tmdy_struct_ex_t *tmdy_struct);
static int do_sequencer(tmdy_struct_ex_t *tmdy_struct);
static void do_chn_voice(tmdy_struct_ex_t *tmdy_struct, uint8 *);
static void do_chn_common(tmdy_struct_ex_t *tmdy_struct, uint8 *);
static void do_timing(tmdy_struct_ex_t *tmdy_struct, uint8 *);
static void do_sysex(tmdy_struct_ex_t *tmdy_struct, uint8 *, int len);
static void do_extended(tmdy_struct_ex_t *tmdy_struct, uint8 *);
static void do_timeout(tmdy_struct_ex_t *tmdy_struct);
static void server_seq_sync(tmdy_struct_ex_t *tmdy_struct, double tm);
/*
static uint8 data_buffer[BUFSIZ];
static int data_buffer_len;
*/
static void doit(tmdy_struct_ex_t *tmdy_struct)
{
    memset(&SRV->control_fd_buffer, 0, sizeof(SRV->control_fd_buffer));
    SRV->control_fd_buffer.fd = SRV->control_fd;

    send_status(tmdy_struct, 220, "TiMidity++ v%s ready", timidity_version);

/*    while(SRV->data_fd != -1 && SRV->control_fd != -1) */
    while(SRV->control_fd != -1)
    {
	fd_set fds;
	int n, maxfd;

	if(SRV->data_fd == -1)
	{
	    if(do_control_command(tmdy_struct))
		break;
	}
	else
	{
	    long usec;

	    FD_ZERO(&fds);
	    FD_SET(SRV->control_fd, &fds);
	    FD_SET(SRV->data_fd, &fds);
	    if(SRV->control_fd > SRV->data_fd)
		maxfd = SRV->control_fd;
	    else
		maxfd = SRV->data_fd;

	    if(SRV->data_fd != -1)
	    {
		    double wait_time;
		    int32 filled;

		    filled = (TMDY_AQ->aq_filled)(tmdy_struct);
		    if(!SRV->tmr_running && filled <= 0)
			usec = -1;
		    else
		    {
			wait_time = (double)filled / TMDY_OUTPUT->play_mode->rate
			    - SRV->low_time_at;
			if(wait_time <= 0)
			    usec = 0;
			else
			    usec = (long)(wait_time * 1000000);
		    }
	    }
	    else
		usec = -1;

	    if(usec >= 0)
	    {
		struct timeval timeout;
		timeout.tv_sec = usec / 1000000;
		timeout.tv_usec = usec % 1000000;
		n = select(maxfd + 1, &fds, NULL, NULL, &timeout);
	    }
	    else
		n = select(maxfd + 1, &fds, NULL, NULL, NULL);

	    if(n < 0)
	    {
		perror("select");
		break;
	    }

	    if(n == 0)
	    {
		if(ctl.verbosity >= VERB_DEBUG)
		{
		    putchar(',');
		    fflush(stdout);
		}
		do_timeout(tmdy_struct);
		continue;
	    }

	    if(SRV->control_fd != -1 && FD_ISSET(SRV->control_fd, &fds))
	    {
		if(do_control_command(tmdy_struct))
		{
		    close(SRV->control_fd);
		    SRV->control_fd = -1;
		}
	    }
	    else if(SRV->data_fd != -1 && FD_ISSET(SRV->data_fd, &fds))
	    {
		if(do_sequencer(tmdy_struct))
		{
		    close(SRV->data_fd);
		    SRV->data_fd = -1;
		    send_status(tmdy_struct, 403, "Data connection is closed");
		}
	    }
	}
    }

    if(SRV->data_fd != -1)
	stop_playing(tmdy_struct);
}

static void do_timeout(tmdy_struct_ex_t *tmdy_struct)
{
    double fill_time;

    if(SRV->data_fd == -1 || !IS_STREAM_TRACE)
	return;
    (TMDY_AQ->aq_add)(tmdy_struct, NULL, 0);
    fill_time = SRV->high_time_at - (double)(TMDY_AQ->aq_filled)(tmdy_struct) / TMDY_OUTPUT->play_mode->rate;
    if(fill_time <= 0)
	return;

    if(SRV->tmr_running)
	add_tick(tmdy_struct, time2tick(tmdy_struct, fill_time));
    else
    {
	MidiEvent ev;
	SRV->event_time_offset += (int32)(fill_time *
				     TMDY_OUTPUT->play_mode->rate);
	ev.time = SRV->curr_event_samples + SRV->event_time_offset;
	ev.type = ME_NONE;
	TMDY_PLAYMIDI->play_event(tmdy_struct, &ev);
    }
}

/* -1=error, 0=success, 1=connection-closed */
static int data_flush(tmdy_struct_ex_t *tmdy_struct, int discard)
{
    fd_set fds;
    char buff[BUFSIZ];
    struct timeval timeout;
    int n;

    while(1)
    {
	FD_ZERO(&fds);
	FD_SET(SRV->data_fd, &fds);
	timeout.tv_sec = 0;
	if(discard)
	    timeout.tv_usec = 100000;
	else
	    timeout.tv_usec = 0;
	if((n = select(SRV->data_fd + 1, &fds, NULL, NULL, &timeout)) < 0)
	{
	    perror("select");
	    return -1;
	}
	if(n == 0)
	    break;
	if(discard)
	{
	    if((n = read(SRV->data_fd, buff, sizeof(buff))) < 0)
	    {
		perror("read");
		return -1;
	    }
	    if(n == 0)
		return 1;
	}
	else
	{
	    int status;
	    if((status = do_sequencer(tmdy_struct)) != 0)
		return status;
	}
    }
    return 0;
}

static void server_seq_sync(tmdy_struct_ex_t *tmdy_struct, double tm)
{
    double t;
    (TMDY_AQ->aq_soft_flush)(tmdy_struct);
    t = (double)(TMDY_AQ->aq_filled)(tmdy_struct) / TMDY_OUTPUT->play_mode->rate;
    if(t > tm)
	usleep((unsigned long)((t - tm) * 1000000));
}

static void server_reset(tmdy_struct_ex_t *tmdy_struct)
{
    TMDY_PLAYMIDI->playmidi_stream_init(tmdy_struct);
    if((TMDY_INSTRUM->free_instruments_afterwards))
	(TMDY_INSTRUM->free_instruments)(tmdy_struct, 0);
    SRV->data_buffer_len = 0;
    do_sysex(tmdy_struct, NULL, 0); /* Initialize SysEx buffer */
    SRV->low_time_at = DEFAULT_LOW_TIMEAT;
    SRV->high_time_at = DEFAULT_HIGH_TIMEAT;
    TMDY_PLAYMIDI->reduce_voice_threshold = 0; /* Disable auto reduction voice */
    compute_sample_increment(tmdy_struct);
    tmr_reset(tmdy_struct);
    SRV->tmr_running = 0;
    SRV->start_time = TMDY_UTILS->timer->get_current_calender_time(tmdy_struct);
}

/* -1=error, 0=success, 1=connection-closed */
static int do_control_command(tmdy_struct_ex_t *tmdy_struct)
{
    int status;
    char *params[MAX_GETCMD_PARAMS];
    int nparams;
    int i;

    if((status = control_getcmd(tmdy_struct, params, &nparams)) == -1)
	return -1;
    if(status == 1)
    {
	send_status(tmdy_struct, 500, "Error");
	return 1;
    }

    if(nparams == 0 ||  *params == NULL || **params == '\0')
	return 0;

    for(i = 0; cmd_table[i].cmd; i++)
	if(strcasecmp(params[0], cmd_table[i].cmd) == 0)
	{
	    if(nparams < cmd_table[i].minarg)
		return send_status(tmdy_struct, 501, "'%s': Arguments is too few",
				   params[0]);
	    if(nparams > cmd_table[i].maxarg)
		return send_status(tmdy_struct, 501, "'%s': Arguments is too many",
				   params[0]);
	    return cmd_table[i].proc(tmdy_struct, nparams, params);
	}
    return send_status(tmdy_struct, 500, "'%s': command not understood.", params[0]);
}

static int cmd_help(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv)
{
    int i;

    if(send_status(tmdy_struct, 200, "Help ok"))
	return -1;

    for(i = 0; cmd_table[i].cmd; i++)
    {
	if(fdputs(cmd_table[i].help, SRV->control_fd))
	    return -1;
	if(fdputs("\n", SRV->control_fd))
	    return -1;
    }
    return fdputs(".\n", SRV->control_fd);
}

static int cmd_open(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv)
{
    int sock;
    struct sockaddr_in in;
    int addrlen;
    int port;

    if(SRV->data_fd != -1)
	return send_status(tmdy_struct, 125, "Data connection is already opened");

    if(strcasecmp(argv[1], "lsb") == 0)
	SRV->is_lsb_data = 1;
    else if(strcasecmp(argv[1], "msb") == 0)
	SRV->is_lsb_data = 0;
    else
	return send_status(tmdy_struct, 502, "OPEN: Invalid argument: %s", argv[1]);

    port = SRV->data_port;
    if((sock = pasv_open(tmdy_struct, &port)) == -1)
	return send_status(tmdy_struct, 511, "Can't open data connection");

    addrlen = sizeof(in);
    memset(&in, 0, addrlen);
    send_status(tmdy_struct, 200, "%d is ready acceptable", port);

    alarm(SIG_TIMEOUT_SEC);
    SRV->data_fd = accept(sock, (struct sockaddr *)&in, &addrlen);
    alarm(0);

    if(SRV->data_fd < 0)
    {
	send_status(tmdy_struct, 512, "Accept error");
	close(sock);
	return 0;
    }
    close(sock);

    if(SRV->control_client.sin_addr.s_addr != in.sin_addr.s_addr)
	return send_status(tmdy_struct, 513, "Security violation:  Address mismatch");

    send_status(tmdy_struct, 200, "Ready data connection");
    SRV->data_buffer_len = 0;
    do_sysex(tmdy_struct, NULL, 0); /* Initialize SysEx buffer */
    tmr_reset(tmdy_struct);

    return 0;
}

static int cmd_close(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv)
{
    if(SRV->data_fd != -1)
    {
	close(SRV->data_fd);
	SRV->data_fd = -1;
	return send_status(tmdy_struct, 302, "Data connection is closed");
    }
    return send_status(tmdy_struct, 302, "Data connection is already closed");
}

static int cmd_queue(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv)
{
    int32 qsamples;

    (TMDY_AQ->aq_add)(tmdy_struct, NULL, 0); /* Update software queue */
    if(!(TMDY_AQ->aq_fill_buffer_flag))
	qsamples = TMDY_AQ->aq_soft_filled(tmdy_struct) + (TMDY_AQ->aq_filled)(tmdy_struct);
    else
	qsamples = 0;
    return send_status(tmdy_struct, 200, "%f sec", (double)qsamples / TMDY_OUTPUT->play_mode->rate);
}

static int cmd_maxqueue(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv)
{
    return send_status(tmdy_struct, 200, "%f sec",
		       (double)(TMDY_AQ->aq_get_dev_queuesize)(tmdy_struct) / TMDY_OUTPUT->play_mode->rate);
}

static int cmd_time(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv)
{
    return send_status(tmdy_struct, 200, "%f sec", (double)(TMDY_AQ->aq_samples)(tmdy_struct));
}

static int cmd_quit(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv)
{
    send_status(tmdy_struct, 200, "Bye");
    return 1;
}

static int cmd_timebase(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv)
{
    int i;

    if(argc == 1)
	return send_status(tmdy_struct, 200, "%d OK", SRV->curr_timebase);
    i = atoi(argv[1]);
    if(i < 1)
	i = 1;
    else if(i > 1000)
	i = 1000;
    if(i != SRV->curr_timebase)
    {
	SRV->curr_timebase = i;
	compute_sample_increment(tmdy_struct);
	SRV->tick_offs = SRV->curr_tick;
	SRV->start_time = TMDY_UTILS->timer->get_current_calender_time(tmdy_struct);
    }
    return send_status(tmdy_struct, 200, "OK");
}

static int cmd_patch(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv)
{
    int dr, bank, prog;

    if(strcasecmp(argv[1], "drumset") == 0)
	dr = 1;
    else if(strcasecmp(argv[1], "bank") == 0)
	dr = 0;
    else
	return send_status(tmdy_struct, 502, "PATCH: Invalid argument: %s", argv[1]); 

    bank = atoi(argv[2]);
    prog = atoi(argv[3]);
    if(bank < 0 || bank > 127 || prog < 0 || prog > 127)
	return send_status(tmdy_struct, 502, "PATCH: Invalid argument"); 
    if(TMDY_PLAYMIDI->play_midi_load_instrument(tmdy_struct, dr, bank, prog) == NULL)
	return send_status(tmdy_struct, 514, "PATCH: Can't load the patch");
    return send_status(tmdy_struct, 200, "OK");
}

static int cmd_reset(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv)
{
    int status;

    if(SRV->data_fd >= 0)
    {
	stop_playing(tmdy_struct);
	if((status = data_flush(tmdy_struct, 1)) != 0)
	    return status;
    }
    server_reset(tmdy_struct);
    return send_status(tmdy_struct, 200, "OK");
}

static int cmd_autoreduce(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv)
{
    if(strcasecmp(argv[1], "on") == 0)
    {
	if(argc == 3)
	    TMDY_PLAYMIDI->reduce_voice_threshold = atoi(argv[2]);
	else
	    TMDY_PLAYMIDI->reduce_voice_threshold = -1;
    }
    else if(strcasecmp(argv[1], "off") == 0)
	TMDY_PLAYMIDI->reduce_voice_threshold = 0;
    else
	return send_status(tmdy_struct, 502, "AUTOREDUCE: Invalid argument: %s",
			   argv[1]);
    return send_status(tmdy_struct, 200, "OK");
}

static int cmd_setbuf(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv)
{
    SRV->low_time_at = atof(argv[1]);
    SRV->high_time_at = atof(argv[1]);
    return send_status(tmdy_struct, 200, "OK");
}

static int cmd_nop(tmdy_struct_ex_t *tmdy_struct, int argc, char **argv)
{
    return send_status(tmdy_struct, 200, "NOP OK");
}

static int do_control_command_nonblock(tmdy_struct_ex_t *tmdy_struct)
{
    struct timeval timeout;
    int n;
    fd_set fds;

    if(SRV->control_fd == -1)
	return 1;
    FD_ZERO(&fds);
    FD_SET(SRV->control_fd, &fds);
    timeout.tv_sec = 0;
    timeout.tv_usec = 0;
    n = select(SRV->control_fd + 1, &fds, NULL, NULL, &timeout);
    if(n > 0 && FD_ISSET(SRV->control_fd, &fds))
	return do_control_command(tmdy_struct);
    return 0;
}

static int fdgets(tmdy_struct_ex_t *tmdy_struct, char *buff, size_t buff_size, struct fd_read_buffer *p)
{
    int n, len, count, size, fd;
    char *buff_endp = buff + buff_size - 1, *pbuff, *beg;

    fd = p->fd;
    if(buff_size == 0)
	return 0;
    else if(buff_size == 1) /* buff == buff_endp */
    {
	*buff = '\0';
	return 0;
    }

    len = 0;
    count = p->count;
    size = p->size;
    pbuff = p->buff;
    beg = buff;
    do
    {
	if(count == size)
	{
	    if((n = read(fd, pbuff, BUFSIZ)) <= 0)
	    {
		*buff = '\0';
		if(n == 0)
		{
		    p->count = p->size = 0;
		    return buff - beg;
		}
		return -1; /* < 0 error */
	    }
	    count = p->count = 0;
	    size = p->size = n;
	}
	*buff++ = pbuff[count++];
    } while(*(buff - 1) != '\n' && buff != buff_endp);
    *buff = '\0';
    p->count = count;
    return buff - beg;
}

static int fdputs(char *s, int fd)
{
    if(write(fd, s, strlen(s)) == -1)
	return -1;
    return 0;
}

#ifdef LITTLE_ENDIAN
static uint32 data2long(tmdy_struct_ex_t *tmdy_struct, uint8* data)
{
    uint32 x;
    memcpy(&x, data, sizeof(x));
    if(!SRV->is_lsb_data)
	x = XCHG_LONG(x);
    return x;
}
static uint16 data2short(tmdy_struct_ex_t *tmdy_struct, uint8* data)
{
    uint16 x;
    memcpy(&x, data, sizeof(x));
    if(!SRV->is_lsb_data)
	x = XCHG_SHORT(x);
    return x;
}
#else
static uint32 data2long(tmdy_struct_ex_t *tmdy_struct, uint8* data)
{
    uint32 x;
    memcpy(&x, data, sizeof(x));
    if(SRV->is_lsb_data)
	x = XCHG_LONG(x);
    return x;
}
static uint16 do_chn_voice(tmdy_struct_ex_t *tmdy_struct, uint8* data)
{
    uint16 x;
    memcpy(&x, data, sizeof(x));
    if(SRV->is_lsb_data)
	x = XCHG_SHORT(x);
    return x;
}
#endif

static int do_sequencer(tmdy_struct_ex_t *tmdy_struct)
{
    int n, offset;
    MidiEvent ev;

    n = read(SRV->data_fd, SRV->data_buffer + SRV->data_buffer_len,
	     sizeof(SRV->data_buffer) - SRV->data_buffer_len);

    if(n <= 0)
    {
	stop_playing(tmdy_struct);
	return n;
    }

#ifdef DEBUG_DUMP_SEQ
    {
	int i;
	for(i = 0; i < n; i++)
	    printf("%02x", SRV->data_buffer[SRV->data_buffer_len + i]);
	putchar('\n');
    }
#endif /* DEBUG_DUMP_SEQ */

    SRV->data_buffer_len += n;
    offset = 0;
    while(offset < SRV->data_buffer_len)
    {
	int cmd;
	cmd = SRV->data_buffer[offset];

#define READ_NEEDBUF(x) if(offset + x > SRV->data_buffer_len) goto done;
#define READ_ADVBUF(x)  offset += x;
	switch(cmd)
	{
	  case EV_CHN_VOICE:
	    READ_NEEDBUF(8);
	    do_chn_voice(tmdy_struct, SRV->data_buffer + offset);
	    READ_ADVBUF(8);
	    break;
	  case EV_CHN_COMMON:
	    READ_NEEDBUF(8);
	    do_chn_common(tmdy_struct, SRV->data_buffer + offset);
	    READ_ADVBUF(8);
	    break;
	  case EV_TIMING:
	    READ_NEEDBUF(8);
	    do_timing(tmdy_struct, SRV->data_buffer + offset);
	    READ_ADVBUF(8);
	    break;
	  case EV_SYSEX:
	    READ_NEEDBUF(8);
	    do_sysex(tmdy_struct, SRV->data_buffer + offset + 2, 6);
	    READ_ADVBUF(8);
	    break;
	  case SEQ_MIDIPUTC:
	    if(SRV->is_system_prefix)
	    {
		READ_NEEDBUF(4);
		do_sysex(tmdy_struct, SRV->data_buffer + offset + 1, 1);
		if(SRV->data_buffer[offset + 1] == 0xf7)
		    SRV->is_system_prefix = 0;  /* End SysEX */
		READ_ADVBUF(4);
		break;
	    }
	    READ_NEEDBUF(2);
	    switch(SRV->data_buffer[offset + 1] & 0xf0)
	    {
	      case MIDI_NOTEON:
		READ_NEEDBUF(12);
		ev.channel = SRV->data_buffer[offset + 1] & 0x0f;
		ev.a       = SRV->data_buffer[offset + 5] & 0x7f;
		ev.b       = SRV->data_buffer[offset + 9] & 0x7f;
		if(ev.b != 0)
		    ev.type = ME_NOTEON;
		else
		    ev.type = ME_NOTEOFF;
		seq_play_event(tmdy_struct, &ev);
		READ_ADVBUF(12);
		break;

	      case MIDI_NOTEOFF:
		READ_NEEDBUF(12);
		ev.type    = ME_NOTEOFF;
		ev.channel = SRV->data_buffer[offset + 1] & 0x0f;
		ev.a       = SRV->data_buffer[offset + 5] & 0x7f;
		ev.b       = SRV->data_buffer[offset + 9] & 0x7f;
		seq_play_event(tmdy_struct, &ev);
		READ_ADVBUF(12);
		break;

	      case MIDI_KEY_PRESSURE:
		READ_NEEDBUF(12);
		ev.type    = ME_KEYPRESSURE;
		ev.channel = SRV->data_buffer[offset + 1] & 0x0f;
		ev.a       = SRV->data_buffer[offset + 5] & 0x7f;
		ev.b       = SRV->data_buffer[offset + 9] & 0x7f;
		seq_play_event(tmdy_struct, &ev);
		READ_ADVBUF(12);
		break;

	      case MIDI_CTL_CHANGE:
		READ_NEEDBUF(12);
		if(TMDY_READMIDI->convert_midi_control_change(tmdy_struct, SRV->data_buffer[offset + 1] & 0x0f,
					       SRV->data_buffer[offset + 5],
					       SRV->data_buffer[offset + 9],
					       &ev))
		    seq_play_event(tmdy_struct, &ev);
		READ_ADVBUF(12);
		break;

	      case MIDI_PGM_CHANGE:
		READ_NEEDBUF(8);
		ev.type    = ME_PROGRAM;
		ev.channel = SRV->data_buffer[offset + 1] & 0x0f;
		ev.a       = SRV->data_buffer[offset + 5] & 0x7f;
		ev.b       = 0;
		seq_play_event(tmdy_struct, &ev);
		READ_ADVBUF(8);
		break;

	      case MIDI_CHN_PRESSURE:
		READ_NEEDBUF(8);
		ev.type    = ME_CHANNEL_PRESSURE;
		ev.channel = SRV->data_buffer[offset + 1] & 0x0f;
		ev.a       = SRV->data_buffer[offset + 5] & 0x7f;
		ev.b       = 0;
		seq_play_event(tmdy_struct, &ev);
		READ_ADVBUF(8);
		break;

	      case MIDI_PITCH_BEND:
		READ_NEEDBUF(12);
		ev.type    = ME_PITCHWHEEL;
		ev.channel = SRV->data_buffer[offset + 1] & 0x0f;
		ev.a       = SRV->data_buffer[offset + 5] & 0x7f;
		ev.b       = SRV->data_buffer[offset + 9] & 0x7f;
		seq_play_event(tmdy_struct, &ev);
		READ_ADVBUF(12);
		break;

	      case MIDI_SYSTEM_PREFIX:
		READ_NEEDBUF(4);
		do_sysex(tmdy_struct, SRV->data_buffer + offset + 1, 1);
		SRV->is_system_prefix = 1; /* Start SysEX */
		READ_ADVBUF(4);
		break;

	      default:
		ctl.cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			 "Undefined SEQ_MIDIPUTC 0x%02x",
			 SRV->data_buffer[offset + 1]);
		send_status(tmdy_struct, 402, "Undefined SEQ_MIDIPUTC 0x%02x",
			    SRV->data_buffer[offset + 1]);
		return 1;
	    }
	    break;

	  case SEQ_FULLSIZE:
	    /* WARNING: This data may be devided into some socket fragments. */
	    offset = SRV->data_buffer_len;
	    ctl.cmsg(tmdy_struct, CMSG_WARNING, VERB_NORMAL,
		     "SEQ_FULLSIZE is received.  This command is not safety.");
	    break;

	  case SEQ_EXTENDED:
	    READ_NEEDBUF(8);
	    do_extended(tmdy_struct, SRV->data_buffer + offset);
	    READ_ADVBUF(8);
	    break;

	  case EV_SEQ_LOCAL:
	  case SEQ_PRIVATE:
	    READ_NEEDBUF(8);
	    /* Ignore */
	    READ_ADVBUF(8);
	    break;

	  default:
	    ctl.cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		     "Undefined data 0x%02x", SRV->data_buffer[offset - 1]);
	    send_status(tmdy_struct, 401, "Wrong data is recieved (seqcmd=0x%02x)",
			cmd);
	    stop_playing(tmdy_struct);
	    return 1;
	}
#undef READ_NEEDBUF
#undef READ_ADVBUF
    }

  done:
    if(offset)
    {
	SRV->data_buffer_len -= offset;
	memmove(SRV->data_buffer, SRV->data_buffer + offset, SRV->data_buffer_len);
    }
    return 0;
}


static void do_chn_voice(tmdy_struct_ex_t *tmdy_struct, uint8 *data)
{
    int type, chn, note, parm;
    MidiEvent ev;

    type = data[2];
    chn  = data[3] % MAX_CHANNELS;
    note = data[4] & 0x7f;
    parm = data[5] & 0x7f;

    ev.channel = chn;
    ev.a       = note;
    ev.b       = parm;
    switch(type)
    {
      case MIDI_NOTEON:
	if(parm > 0)
	{
	    ev.type = ME_NOTEON;
	    seq_play_event(tmdy_struct, &ev);
	    break;
	}
	/* FALLTHROUGH */
      case MIDI_NOTEOFF:
	ev.type = ME_NOTEOFF;
	seq_play_event(tmdy_struct, &ev);
	break;
      case MIDI_KEY_PRESSURE:
	ev.type = ME_KEYPRESSURE;
	seq_play_event(tmdy_struct, &ev);
	break;
    }
}

static void do_chn_common(tmdy_struct_ex_t *tmdy_struct, uint8 *data)
{
    int type, chn, p1, p2, w14;
    MidiEvent ev;

    type = data[2];
    chn  = data[3] % MAX_CHANNELS;
    p1 = data[4] & 0x7f;
    p2 = data[5] & 0x7f;
    w14 = data2short(tmdy_struct, data + 6) & 0x3fff;

    if(type == 0xff) /* Meta event */
    {
	/* This event is special event for timidity.  (not OSS compatible) */
	switch(data[3])
	{
	  case 0x2f: /* End of midi */
	    stop_playing(tmdy_struct);
	    tmr_reset(tmdy_struct);
	    break;

	  case 0x7f: /* Sequencer-Specific Meta-Event */
	    switch(data[4])
	    {
	      case 0x01: /* MIDI Reset */
		ev.type = ME_RESET;
		ev.a = DEFAULT_SYSTEM_MODE;
		ev.b = 0;
		seq_play_event(tmdy_struct, &ev);
		break;

	      case 0x02:  { /* Used to sync. */
		    double target_time, queue_time, sleep_time;
		    if(w14 == 0)
		    {
			(TMDY_AQ->aq_flush)(tmdy_struct, 0); /* wait until playout */
			send_status(tmdy_struct, 301, "0 Sync OK");
			break;
		    }

		    (TMDY_AQ->aq_soft_flush)(tmdy_struct);
		    target_time = (double)w14 / SRV->curr_timebase;
		    queue_time = (double)(TMDY_AQ->aq_filled)(tmdy_struct) / TMDY_OUTPUT->play_mode->rate;
		    sleep_time = queue_time - target_time;
		    if(sleep_time > 0)
		    {
			send_status(tmdy_struct, 301, "%g Sync OK", sleep_time);
			usleep((unsigned long)(sleep_time * 1000000));
		    }
		    else
			send_status(tmdy_struct, 301, "0 Sync OK");
		}
		break;
	    }
	}
	return;
    }

    ev.channel = chn;
    switch(type)
    {
      case MIDI_CTL_CHANGE:
	if(TMDY_READMIDI->convert_midi_control_change(tmdy_struct, chn, p1, w14, &ev))
	    seq_play_event(tmdy_struct, &ev);
	break;
      case MIDI_PGM_CHANGE:
	ev.type = ME_PROGRAM;
	ev.a    = p1;
	ev.b    = 0;
	seq_play_event(tmdy_struct, &ev);
	break;
      case MIDI_CHN_PRESSURE:
	ev.type = ME_CHANNEL_PRESSURE;
	ev.a    = p1;
	ev.b    = p2;
	seq_play_event(tmdy_struct, &ev);
	break;
      case MIDI_PITCH_BEND:
	ev.type = ME_PITCHWHEEL;
	ev.a    = w14 & 0x7f;
	ev.b    = (w14>>7) & 0x7f;
	seq_play_event(tmdy_struct, &ev);
	break;
    }
}


static void do_timing(tmdy_struct_ex_t *tmdy_struct, uint8 *data)
{
    int32 val;

    val = data2long(tmdy_struct, data + 4);
    switch(data[1])
    {
      case TMR_START:
	SRV->tmr_running = 1;
	tmr_reset(tmdy_struct);
	SRV->event_time_offset = (int32)(TMDY_OUTPUT->play_mode->rate * SRV->high_time_at);
	break;

      case TMR_STOP:
	SRV->tmr_running = 0;
	break;

      case TMR_CONTINUE:
	if(!SRV->tmr_running)
	{
	    SRV->tmr_running = 1;
	    SRV->tick_offs = SRV->curr_tick;
	    SRV->start_time = TMDY_UTILS->timer->get_current_calender_time(tmdy_struct);
	}
	break;

      case TMR_TEMPO:
#if 0 /* What should TMR_TEMPO work ? */
	if(val < 8)
	    val = 8;
	else if(val > 250)
	    val = 250;
	current_play_tempo = 60 * 1000000 / val;
	compute_sample_increment(tmdy_struct);
#endif
	break;

      case TMR_WAIT_ABS:

/*
   printf("## TMR_WAIT_ABS: %d %d %d %g\n",
       SRV->curr_tick,
       SRV->curr_tick - (time2tick(tmdy_struct, TMDY_UTILS->timer->get_current_calender_time(tmdy_struct)
			      - SRV->start_time) + SRV->tick_offs),
       SRV->event_time_offset,
       (double)(TMDY_AQ->aq_filled)(tmdy_struct) / TMDY_OUTPUT->play_mode->rate);
 */

	val -= SRV->curr_tick;
	/*FALLTHROUGH*/
      case TMR_WAIT_REL:
	if(val <= 0)
	    break;
	add_tick(tmdy_struct, val);
	break;

#if 0
      case TMR_ECHO:
      case TMR_SPP:
#endif
      default:
/* printf("## TMR=0x%02x is not supported\n", data[1]); */
	break;
    }
}


static void do_sysex(tmdy_struct_ex_t *tmdy_struct, uint8 *data, int n)
{
/*
    static uint8 sysexbuf[BUFSIZ];
    static int buflen;
    static int fillflag;
*/
    int i;

    if(data == NULL)
    {
	SRV->buflen = SRV->fillflag = 0;
	SRV->is_system_prefix = 0;
	return;
    }

    for(i = 0; i < n; i++)
    {
	/* SysEx := /\xf0([^\xf7]*\xf7)/ */
	if(!SRV->fillflag)
	{
	    if(data[i] == 0xf0)
	    {
		SRV->fillflag = 1;
		continue;
	    }
	}
	if(SRV->fillflag)
	{
	    if(SRV->buflen < sizeof(SRV->sysexbuf))
		SRV->sysexbuf[SRV->buflen++] = data[i];
	    if(data[i] == 0xf7)
	    {
		MidiEvent ev;
		if(TMDY_READMIDI->parse_sysex_event(tmdy_struct, SRV->sysexbuf, SRV->buflen, &ev))
		    seq_play_event(tmdy_struct, &ev);
		SRV->buflen = 0;
		SRV->fillflag = 0;
	    }
	}
    }
}

static void do_extended(tmdy_struct_ex_t *tmdy_struct, uint8 *data)
{
    int value;
    MidiEvent ev;

    value = data[5] | data[6] * 256;
    switch(data[1])
    {
      case SEQ_CONTROLLER:
	switch(data[4])
	{
	  case CTRL_PITCH_BENDER: /* ?? */
	    break;
	  case CTRL_PITCH_BENDER_RANGE: /* ?? */
	    ev.channel = data[3] % MAX_CHANNELS;
	    ev.b = 0;
	    ev.a = 0;

	    /* LSB */
	    ev.type = ME_NRPN_LSB;
	    seq_play_event(tmdy_struct, &ev);

	    /* MSB */
	    ev.type = ME_NRPN_MSB;
	    seq_play_event(tmdy_struct, &ev);

	    /* Data entry */
	    ev.type = ME_DATA_ENTRY_MSB;
	    ev.a = value / 100; /* ?? */
	    seq_play_event(tmdy_struct, &ev);

	    break;
	  default:
	    break;
	}
	break;
      default:
	break;
    }
}


/*
 * interface_<id>_loader();
 */
ControlMode *interface_r_loader(void)
{
    return &ctl;
}




server_c_ex_t* new_server_c(tmdy_struct_ex_t *tmdy_struct){
	int i;
	server_c_ex_t* server_c_ex;

	server_c_ex=(server_c_ex_t *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(server_c_ex_t));
	
	timidity_mutex_init(server_c_ex->busy);

	server_c_ex->data_fd = -1;
	server_c_ex-> control_fd = -1;
	server_c_ex->is_lsb_data = 1;
	server_c_ex->curr_timebase = DEFAULT_TIMEBASE;
	server_c_ex->is_system_prefix = 0;
	server_c_ex->low_time_at = 0.3;
	server_c_ex-> high_time_at = 0.5;
	
	return server_c_ex;
}
void destroy_server_c(server_c_ex_t* server_c){
	timidity_mutex_destroy(server_c->busy);
	free(server_c);
}
