/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    rtsyn_common.c
        Copyright (c) 2003  Keishi Suenaga <s_keishi@mutt.freemail.ne.jp>

    I referenced following sources.
        alsaseq_c.c - ALSA sequencer server interface
            Copyright (c) 2000  Takashi Iwai <tiwai@suse.de>
        readmidi.c
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#include "interface.h"

#include <stdio.h>

#include <stdarg.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <sys/types.h>
#ifdef TIME_WITH_SYS_TIME
#include <sys/time.h>
#endif
#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#include <math.h>
#include <signal.h>

#include "server_defs.h"

#ifdef __W32__
#include <windows.h>
#endif

#include "tmdy_struct.h"
#include "timidity.h"
#include "common.h"
#include "controls.h"
#include "instrum.h"
#include "playmidi.h"
#include "readmidi.h"
#include "recache.h"
#include "output.h"
#include "aq.h"
#include "timer.h"

#include "rtsyn.h"
#include "rtsyn_prv.h"




#define EX_RESET_NO 7
static const char sysex_resets[EX_RESET_NO][11]={
		'\xf0','\x7e','\x7f','\x09','\x00','\xf7','\x00','\x00','\x00','\x00','\x00',
		'\xf0','\x7e','\x7f','\x09','\x01','\xf7','\x00','\x00','\x00','\x00','\x00',
		'\xf0','\x7e','\x7f','\x09','\x03','\xf7','\x00','\x00','\x00','\x00','\x00',
		'\xf0','\x41','\x10','\x42','\x12','\x40','\x00','\x7f','\x00','\x41','\xf7',
		'\xf0','\x41','\x10','\x42','\x12','\x00','\x00','\x7f','\x00','\x01','\xf7',
		'\xf0','\x41','\x10','\x42','\x12','\x00','\x00','\x7f','\x01','\x00','\xf7',
		'\xf0','\x43','\x10','\x4c','\x00','\x00','\x7E','\x00','\xf7','\x00','\x00' };
/*
#define EX_RESET_NO 9
static const char sysex_resets[EX_RESET_NO][11]={
	'\xf0','\x7e','\x7f','\x09','\x00','\xf7','\x00','\x00','\x00','\x00','\x00', //gm off
	'\xf0','\x7e','\x7f','\x09','\x01','\xf7','\x00','\x00','\x00','\x00','\x00', //gm1
	'\xf0','\x7e','\x7f','\x09','\x02','\xf7','\x00','\x00','\x00','\x00','\x00', //gm off
	'\xf0','\x7e','\x7f','\x09','\x03','\xf7','\x00','\x00','\x00','\x00','\x00', //gm2
	'\xf0','\x41','\x10','\x42','\x12','\x40','\x00','\x7f','\x00','\x41','\xf7', //GS
	'\xf0','\x41','\x10','\x42','\x12','\x40','\x00','\x7f','\x7f','\x41','\xf7', //GS off
	'\xf0','\x41','\x10','\x42','\x12','\x00','\x00','\x7f','\x00','\x01','\xf7', //88
	'\xf0','\x41','\x10','\x42','\x12','\x00','\x00','\x7f','\x01','\x00','\xf7', //88
	'\xf0','\x43','\x10','\x4c','\x00','\x00','\x7E','\x00','\xf7','\x00','\x00'  //XG on
	};
*/

static void seq_set_time(tmdy_struct_ex_t *tmdy_struct,MidiEvent *ev);



void rtsyn_gm_reset(tmdy_struct_ex_t *tmdy_struct){
	rtsyn_server_reset(tmdy_struct);
	TMDY_READMIDI->change_system_mode(tmdy_struct, GM_SYSTEM_MODE);
}

void rtsyn_gs_reset(tmdy_struct_ex_t *tmdy_struct){
	rtsyn_server_reset(tmdy_struct);
	TMDY_READMIDI->change_system_mode(tmdy_struct, GS_SYSTEM_MODE);
}

void rtsyn_xg_reset(tmdy_struct_ex_t *tmdy_struct){
	rtsyn_server_reset(tmdy_struct);
	TMDY_READMIDI->change_system_mode(tmdy_struct, XG_SYSTEM_MODE);
}

void rtsyn_normal_reset(tmdy_struct_ex_t *tmdy_struct){
	rtsyn_server_reset(tmdy_struct);
	TMDY_READMIDI->change_system_mode(tmdy_struct,DEFAULT_SYSTEM_MODE);
}


void rtsyn_gm_modeset(tmdy_struct_ex_t *tmdy_struct){
	RTSYN->rtsyn_system_mode=GM_SYSTEM_MODE;
	rtsyn_server_reset(tmdy_struct);
	TMDY_READMIDI->change_system_mode(tmdy_struct, RTSYN->rtsyn_system_mode);
}

void rtsyn_gs_modeset(tmdy_struct_ex_t *tmdy_struct){
	rtsyn_server_reset(tmdy_struct);
	RTSYN->rtsyn_system_mode=GS_SYSTEM_MODE;
	TMDY_READMIDI->change_system_mode(tmdy_struct, RTSYN->rtsyn_system_mode);
}

void rtsyn_xg_modeset(tmdy_struct_ex_t *tmdy_struct){
	rtsyn_server_reset(tmdy_struct);
	RTSYN->rtsyn_system_mode=XG_SYSTEM_MODE;
	TMDY_READMIDI->change_system_mode(tmdy_struct, RTSYN->rtsyn_system_mode);
}

void rtsyn_normal_modeset(tmdy_struct_ex_t *tmdy_struct){
	rtsyn_server_reset(tmdy_struct);
	RTSYN->rtsyn_system_mode=DEFAULT_SYSTEM_MODE;
	TMDY_READMIDI->change_system_mode(tmdy_struct, RTSYN->rtsyn_system_mode);
}


static void timer_one_calc(tmdy_struct_ex_t * tmdy_struct){
	int32 time_div; 
	double currenttime;

	currenttime=TMDY_UTILS->timer->get_current_calender_time(tmdy_struct);
	if( RTSYN->active_sensing_flag==~0 && 
		( currenttime > RTSYN->active_sensing_time + 0.330) ){
		(TMDY_OUTPUT->play_mode)->close_output(tmdy_struct);
		(TMDY_OUTPUT->play_mode)->open_output(tmdy_struct);
		TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			"Active Sensing Expired\n");
		RTSYN->active_sensing_flag=0;
	}
	timidity_mutex_lock(RTSYN->busy);
	time_div=(double)((double)TMDY_OUTPUT->play_mode->rate * 
			(currenttime - RTSYN->starttime) + 0.5 );
	if( time_div > (double)(TMDY_OUTPUT->play_mode->rate)/(double)TICKTIME_HZ * 2.0){
		time_div = (double)(TMDY_OUTPUT->play_mode->rate) / (double)TICKTIME_HZ - 3.0 ;
	}
	RTSYN->starttime=currenttime;
	TMDY_PLAYMIDI->compute_data(tmdy_struct,time_div);
	(TMDY_AQ->aq_fill_nonblocking)(tmdy_struct);
	RTSYN->rtsyn_reachtime=currenttime + 1.0 / (double)TICKTIME_HZ ;
	timidity_mutex_unlock(RTSYN->busy);
}

#ifdef USE_WINSYN_TIMER_I

#ifdef __W32__

VOID CALLBACK timercalc(UINT uTimerID, UINT uMsg, DWORD dwUser, DWORD dummy1, DWORD dummy2){
	tmdy_struct_ex_t* tmdy_struct;
	tmdy_struct=(tmdy_struct_ex_t *)dwUser;	
	timer_one_calc(tmdy_struct);	
	return;
}


#else
void *timercalc(void *arg){
	double reachtime,delay;
	tmdy_struct_ex_t* tmdy_struct;

	tmdy_struct=(tmdy_struct_ex_t *)arg;	
	delay=(double)(1.0/TICKTIME_HZ);
	while(RTSYN->thread_on_f==1){		
		reachtime=TMDY_UTILS->timer->get_current_calender_time(tmdy_struct)+delay;
		timer_one_calc(tmdy_struct);
		do{
			sleep(0);
		}while(TMDY_UTILS->timer->get_current_calender_time(tmdy_struct)<reachtime);
	}
	return NULL;
}
#endif
#endif
void rtsyn_init(tmdy_struct_ex_t *tmdy_struct){
	int i,j;
	unsigned int port;
		/* set constants */
	TMDY_PLAYMIDI->opt_realtime_playing = 1; /* Enable loading patch while playing */
	TMDY_RECACHE->allocate_cache_size = 0; /* Don't use pre-calclated samples */
	TMDY_PLAYMIDI->auto_reduce_polyphony = 0;
	TMDY_PLAYMIDI->current_keysig = TMDY_PLAYMIDI->opt_init_keysig;
	TMDY_PLAYMIDI->note_key_offset = 0;

	if (TMDY_PLAYMIDI->opt_force_keysig != 8) {
		i = TMDY_PLAYMIDI->current_keysig + ((TMDY_PLAYMIDI->current_keysig < 8) ? 7 : -6);
		j = TMDY_PLAYMIDI->opt_force_keysig + ((TMDY_PLAYMIDI->current_keysig < 8) ? 7 : 10);
		while (i != j && i != j + 12) {
			if (++TMDY_PLAYMIDI->note_key_offset > 6)
				TMDY_PLAYMIDI->note_key_offset -= 12;
			i += (i > 10) ? -5 : 7;
		}
	}
	i = TMDY_PLAYMIDI->current_keysig + ((TMDY_PLAYMIDI->current_keysig < 8) ? 7 : -9), j = 0;
	while (i != 7 && i != 19)
		i += (i < 7) ? 5 : -7, j++;
	j += TMDY_PLAYMIDI->note_key_offset, j -= floor(j / 12.0) * 12;
	TMDY_PLAYMIDI->current_freq_table = j;

	RTSYN->rtsyn_system_mode=DEFAULT_SYSTEM_MODE;
	RTSYN->active_sensing_flag=0;
	RTSYN->active_sensing_time=0;
#ifdef USE_WINSYN_TIMER_I
#ifdef __W32__
		RTSYN->timerID=0;
#else 
		RTSYN->timer_thread=NULL;
		RTSYN->thread_on_f=0;
#endif
#endif
	for(port=0;port<MAX_PORT;port++){
		RTSYN->portID[port]=-1;
	}
//	}
	rtsyn_server_reset(tmdy_struct);
}

void rtsyn_close(tmdy_struct_ex_t *tmdy_struct)
{
#ifdef USE_WINSYN_TIMER_I
#ifdef __W32__
	timeKillEvent( RTSYN->timerID );
	RTSYN->timerID=0;
	timeEndPeriod(1);
#else
	RTSYN->thread_on_f=0;
	pthread_join(RTSYN->timer_thread, NULL);
	RTSYN->timer_thread=NULL;
#endif
#endif 
}

void rtsyn_play_event(tmdy_struct_ex_t *tmdy_struct, MidiEvent *ev)
{
  int gch;
  int32 cet;
	gch = GLOBAL_CHANNEL_EVENT_TYPE(ev->type);
//	if(gch || !IS_SET_CHANNELMASK(TMDY_READMIDI->quietchannels, ev->channel) ){
//    if ( !seq_quit ) {
			ev->time=0;
#ifdef USE_WINSYN_TIMER
			if(RTSYN->timerID!=0){
				TMDY_PLAYMIDI->play_event(tmdy_struct, ev);
			}
#else
			TMDY_PLAYMIDI->play_event(tmdy_struct, ev);
#endif
//		}
//	}
}

void rtsyn_server_reset(tmdy_struct_ex_t *tmdy_struct){
	int i;
	
//	timidity_mutex_lock(TMDY_PLAYMIDI->busy);
	(TMDY_OUTPUT->play_mode)->close_output(tmdy_struct);	// PM_REQ_PLAY_START wlll called in TMDY_PLAYMIDI->playmidi_stream_init()
	(TMDY_OUTPUT->play_mode)->open_output(tmdy_struct);	// but w32_a.c does not have it.
	TMDY_READMIDI->readmidi_read_init(tmdy_struct);
//	timidity_mutex_unlock(TMDY_PLAYMIDI->busy);
	TMDY_PLAYMIDI->playmidi_stream_init(tmdy_struct);
//	timidity_mutex_lock(TMDY_PLAYMIDI->busy);
	if ((TMDY_INSTRUM->free_instruments_afterwards))
		(TMDY_INSTRUM->free_instruments)(tmdy_struct, 0);

	TMDY_UTILS->mblock->free_global_mblock(tmdy_struct);
	RTSYN->starttime=TMDY_UTILS->timer->get_current_calender_time(tmdy_struct);
	TMDY_PLAYMIDI->reduce_voice_threshold = 0; // * Disable auto reduction voice *
	TMDY_PLAYMIDI->auto_reduce_polyphony = 0;
//	timidity_mutex_unlock(TMDY_PLAYMIDI->busy);
}

void rtsyn_stop_playing(tmdy_struct_ex_t *tmdy_struct)
{
	if(TMDY_PLAYMIDI->upper_voices) {
/*		MidiEvent ev;
		ev.type = ME_EOT;
		ev.a = 0;
		ev.b = 0;
		rtsyn_play_event(tmdy_struct, &ev);
*/
	(TMDY_AQ->aq_flush)(tmdy_struct, 1);
//	(TMDY_OUTPUT->play_mode)->close_output(tmdy_struct);
	}
}

void rtsyn_play_calculate(tmdy_struct_ex_t *tmdy_struct){
	MidiEvent ev;

#ifdef USE_WINSYN_TIMER_I

#ifdef __W32__

	if(RTSYN->timerID==0){
		timeBeginPeriod(1);
		RTSYN->starttime=0;
		{
			DWORD data;
			UINT delay;
			delay=(1000/TICKTIME_HZ+0.5);
			data=(DWORD)tmdy_struct;
		 
		
			RTSYN->timerID = timeSetEvent( delay, 0, timercalc, data,
				TIME_PERIODIC | TIME_CALLBACK_FUNCTION );
        	if( !RTSYN->timerID ){
        		TMDY_CONTROLS->ctl->cmsg(tmdy_struct,   CMSG_ERROR, VERB_NORMAL,"Fail to setup Timer Interrupt (winsyn) \n");
        	}
		}
	}
#else
	if(RTSYN->timer_thread==NULL){
		RTSYN->thread_on_f=1;
		if(0!=pthread_create(&RTSYN->timer_thread,NULL,timercalc,(void *)tmdy_struct)){
        		TMDY_CONTROLS->ctl->cmsg(tmdy_struct,   CMSG_ERROR, VERB_NORMAL,"Fail to setup Timer Interrupt (winsyn) \n");
		}
	}
#endif /* __W32__ */

#else
	timer_one_calc(tmdy_struct);

#endif /* USE_WINSYN_TIMER_I */
	
}
	
int rtsyn_play_one_data (tmdy_struct_ex_t *tmdy_struct, int port, int32 dwParam1){
	MidiEvent ev;

	ev.type = ME_NONE;
	ev.channel = dwParam1 & 0x0000000f;
	ev.channel = ev.channel+port*16;
	ev.a = (dwParam1 >> 8) & 0xff;
	ev.b = (dwParam1 >> 16) & 0xff;
	switch ((int) (dwParam1 & 0x000000f0)) {
	case 0x80:
		ev.type = ME_NOTEOFF;
//		rtsyn_play_event(tmdy_struct,&ev);
		break;
	case 0x90:
		ev.type = (ev.b) ? ME_NOTEON : ME_NOTEOFF;
//		rtsyn_play_event(tmdy_struct,&ev);
		break;
	case 0xa0:
		ev.type = ME_KEYPRESSURE;
//		rtsyn_play_event(tmdy_struct,&ev);
		break;
	case 0xb0:
		if (! TMDY_READMIDI->convert_midi_control_change(tmdy_struct, ev.channel, ev.a, ev.b, &ev))
		ev.type = ME_NONE;
		break;
	case 0xc0:
		ev.type = ME_PROGRAM;
//		rtsyn_play_event(tmdy_struct,&ev);
		break;
	case 0xd0:
		ev.type = ME_CHANNEL_PRESSURE;
//		rtsyn_play_event(tmdy_struct,&ev);
		break;
	case 0xe0:
		ev.type = ME_PITCHWHEEL;
//		rtsyn_play_event(tmdy_struct,&ev);
		break;
	case 0xf0:
//#ifdef IA_PORTMIDISYN
		if ( (dwParam1 & 0x000000ff) == 0xf0) {
			//SysEx
			return 1;
		}
//#endif
		if ((dwParam1 & 0x000000ff) == 0xf2) {
			ev.type = ME_PROGRAM;
//			rtsyn_play_event(tmdy_struct,&ev);
		}
#if 0
		if ((dwParam1 & 0x000000ff) == 0xf1)
			//MIDI Time Code Qtr. Frame (not need)
			printf("MIDI Time Code Qtr\n");
		if ((dwParam1 & 0x000000ff) == 0xf3)
			//Song Select(Song #) (not need)
		if ((dwParam1 & 0x000000ff) == 0xf6)
			//Tune request (not need)
			printf("Tune request\n");
		if ((dwParam1 & 0x000000ff) == 0xf8)
			//Timing Clock (not need)
			printf("Timing Clock\n");
		if ((dwParam1&0x000000ff)==0xfa)
			//Start
		if ((dwParam1 & 0x000000ff) == 0xfb)
			//Continue
		if ((dwParam1 & 0x000000ff) == 0xfc) {
			//Stop
			printf("Stop\n");
		}
#endif
		if ((dwParam1 & 0x000000ff) == 0xfe) {
			//Active Sensing
//			printf("Active Sensing\n");
			RTSYN->active_sensing_flag = ~0;
			RTSYN->active_sensing_time = TMDY_UTILS->timer->get_current_calender_time(tmdy_struct);
		}
		if ((dwParam1 & 0x000000ff) == 0xff) {
			//System Reset
			printf("System Reset\n");
		}
		break;
	default:
//		printf("Unsup/ed event %d\n", aevp->type);
		break;
	}
	if (ev.type != ME_NONE) {
		rtsyn_play_event(tmdy_struct,&ev);
	}
	return 0;
}


void rtsyn_play_one_sysex (tmdy_struct_ex_t *tmdy_struct, char *sysexbuffer, int exlen ){
	int i,j,chk,ne;
	MidiEvent ev;
	MidiEvent evm[260];

	if(sysexbuffer[exlen-1] == '\xf7'){            // I don't konw why this need
/*
		for(i=0;i<EX_RESET_NO;i++){
			chk=0;
			for(j=0;(j<exlen)&&(j<11);j++){
				if(chk==0 && sysex_resets[i][j]!=sysexbuffer[j]){
					chk=~0;
				}
			}
			if(chk==0){
				 rtsyn_server_reset(tmdy_struct);
				TMDY_READMIDI->change_system_mode(tmdy_struct,DEFAULT_SYSTEM_MODE);
			}
		}
*/
/*
		printf("SyeEx length=%x bytes \n", exlen);
		for(i=0;i<exlen;i++){
			printf("%x ",sysexbuffer[i]);
		}
		printf("\n");
*/
		if(TMDY_READMIDI->parse_sysex_event(tmdy_struct, sysexbuffer+1,exlen-1,&ev)){
			if(ev.type==ME_RESET){
				rtsyn_server_reset(tmdy_struct);
					TMDY_READMIDI->change_system_mode(tmdy_struct, RTSYN->rtsyn_system_mode);

			}else{
				rtsyn_play_event(tmdy_struct,&ev);
			}

		}
		if(ne=TMDY_READMIDI->parse_sysex_event_multi(tmdy_struct, sysexbuffer+1,exlen-1, evm)){
			for (i = 0; i < ne; i++){
					rtsyn_play_event(tmdy_struct,&evm[i]);
			}
		}
	}
}



void ts_rtsyn_gm_reset(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(RTSYN->busy);
	rtsyn_gm_reset(tmdy_struct);
	timidity_mutex_unlock(RTSYN->busy);
}
void ts_rtsyn_gs_reset(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(RTSYN->busy);
	rtsyn_gs_reset(tmdy_struct);
	timidity_mutex_unlock(RTSYN->busy);
}
void ts_rtsyn_xg_reset(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(RTSYN->busy);
	rtsyn_xg_reset(tmdy_struct);
	timidity_mutex_unlock(RTSYN->busy);
}
void ts_rtsyn_normal_reset(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(RTSYN->busy);
	rtsyn_normal_reset(tmdy_struct);
	timidity_mutex_unlock(RTSYN->busy);
}
void ts_rtsyn_gm_modeset(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(RTSYN->busy);
	rtsyn_gm_modeset(tmdy_struct);
	timidity_mutex_unlock(RTSYN->busy);
}
void ts_rtsyn_gs_modeset(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(RTSYN->busy);
	rtsyn_gs_modeset(tmdy_struct);
	timidity_mutex_unlock(RTSYN->busy);
}
void ts_rtsyn_xg_modeset(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(RTSYN->busy);
	rtsyn_xg_modeset(tmdy_struct);
	timidity_mutex_unlock(RTSYN->busy);
}
void ts_rtsyn_normal_modeset(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(RTSYN->busy);
	rtsyn_normal_modeset(tmdy_struct);
	timidity_mutex_unlock(RTSYN->busy);
}
void ts_rtsyn_init(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(RTSYN->busy);
	rtsyn_init(tmdy_struct);
	timidity_mutex_unlock(RTSYN->busy);
}
void ts_rtsyn_close(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(RTSYN->busy);
	rtsyn_close(tmdy_struct);
	timidity_mutex_unlock(RTSYN->busy);
}
void ts_rtsyn_play_event(tmdy_struct_ex_t *tmdy_struct, MidiEvent *ev){
	timidity_mutex_lock(RTSYN->busy);
	rtsyn_play_event(tmdy_struct, ev);
	timidity_mutex_unlock(RTSYN->busy);
}
void ts_rtsyn_server_reset(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(RTSYN->busy);
	rtsyn_server_reset(tmdy_struct);
	timidity_mutex_unlock(RTSYN->busy);
}
void ts_rtsyn_stop_playing(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(RTSYN->busy);
	rtsyn_stop_playing(tmdy_struct);
	timidity_mutex_unlock(RTSYN->busy);
}
int ts_rtsyn_play_one_data (tmdy_struct_ex_t *tmdy_struct, int port, int32 dwParam1){
	int ts_buf;
	timidity_mutex_lock(RTSYN->busy);
	ts_buf=rtsyn_play_one_data (tmdy_struct, port, dwParam1);
	timidity_mutex_unlock(RTSYN->busy);
	return ts_buf;
}
void ts_rtsyn_play_one_sysex (tmdy_struct_ex_t *tmdy_struct, char *sysexbuffer, int exlen ){
	timidity_mutex_lock(RTSYN->busy);
	rtsyn_play_one_sysex (tmdy_struct, sysexbuffer, exlen );
	timidity_mutex_unlock(RTSYN->busy);
}
void ts_rtsyn_play_calculate(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(RTSYN->busy);
	rtsyn_play_calculate(tmdy_struct);
	timidity_mutex_unlock(RTSYN->busy);
}
int ts_rtsyn_add_midi_port(tmdy_struct_ex_t *tmdy_struct, int port_id){
	int ts_buf;
	timidity_mutex_lock(RTSYN->busy);
	ts_buf=rtsyn_add_midi_port(tmdy_struct, port_id);
	timidity_mutex_unlock(RTSYN->busy);
	return ts_buf;
}
void ts_rtsyn_delete_midi_port(tmdy_struct_ex_t *tmdy_struct, int port_id){
	timidity_mutex_lock(RTSYN->busy);
	rtsyn_delete_midi_port(tmdy_struct, port_id);
	timidity_mutex_unlock(RTSYN->busy);
}

void ts_rtsyn_get_port_list(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(RTSYN->busy);
	rtsyn_get_port_list(tmdy_struct);
	timidity_mutex_unlock(RTSYN->busy);
}
int ts_rtsyn_synth_start(tmdy_struct_ex_t *tmdy_struct){
	int ts_buf;
	timidity_mutex_lock(RTSYN->busy);
	ts_buf=rtsyn_synth_start(tmdy_struct);
	timidity_mutex_unlock(RTSYN->busy);
	return ts_buf;
}
void ts_rtsyn_synth_stop(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(RTSYN->busy);
	rtsyn_synth_stop(tmdy_struct);
	timidity_mutex_unlock(RTSYN->busy);
}
int ts_rtsyn_play_some_data (tmdy_struct_ex_t *tmdy_struct){
	int ts_buf;
	timidity_mutex_lock(RTSYN->busy);
	ts_buf=rtsyn_play_some_data (tmdy_struct);
	timidity_mutex_unlock(RTSYN->busy);
	return ts_buf;
}
#if defined(IA_WINSYN) || defined(IA_W32G_SYN)
int ts_rtsyn_buf_check(tmdy_struct_ex_t *tmdy_struct){
	int ts_buf;
	timidity_mutex_lock(RTSYN->busy);
	ts_buf=rtsyn_buf_check(tmdy_struct);
	timidity_mutex_unlock(RTSYN->busy);
	return ts_buf;
}
#endif
void ts_rtsyn_midiports_close(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(RTSYN->busy);
	rtsyn_midiports_close(tmdy_struct);
	timidity_mutex_unlock(RTSYN->busy);
}



rtsyn_ex_t* new_rtsyn(tmdy_struct_ex_t *tmdy_struct){
	int i;
	rtsyn_ex_t* rtsyn_ex;

	rtsyn_ex=(rtsyn_ex_t *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(rtsyn_ex_t));
	
	timidity_mutex_init(rtsyn_ex->busy);
	
/* rtsyn_common.c */
rtsyn_ex->rtsyn_gm_reset=ts_rtsyn_gm_reset;
rtsyn_ex->rtsyn_gs_reset=ts_rtsyn_gs_reset;
rtsyn_ex->rtsyn_xg_reset=ts_rtsyn_xg_reset;
rtsyn_ex->rtsyn_normal_reset=ts_rtsyn_normal_reset;
rtsyn_ex->rtsyn_gm_modeset=ts_rtsyn_gm_modeset;
rtsyn_ex->rtsyn_gs_modeset=ts_rtsyn_gs_modeset;
rtsyn_ex->rtsyn_xg_modeset=ts_rtsyn_xg_modeset;
rtsyn_ex->rtsyn_normal_modeset=ts_rtsyn_normal_modeset;
rtsyn_ex->rtsyn_init=ts_rtsyn_init;
rtsyn_ex->rtsyn_close=ts_rtsyn_close;
rtsyn_ex->rtsyn_play_event=ts_rtsyn_play_event;
rtsyn_ex->rtsyn_server_reset=ts_rtsyn_server_reset;
rtsyn_ex->rtsyn_stop_playing=ts_rtsyn_stop_playing;
rtsyn_ex->rtsyn_play_one_data=ts_rtsyn_play_one_data;
rtsyn_ex->rtsyn_play_one_sysex=ts_rtsyn_play_one_sysex;
rtsyn_ex->rtsyn_play_calculate=ts_rtsyn_play_calculate;
rtsyn_ex->rtsyn_add_midi_port=ts_rtsyn_add_midi_port;
rtsyn_ex->rtsyn_delete_midi_port=ts_rtsyn_delete_midi_port;
/*rtsyn_(interface).c*/
rtsyn_ex->rtsyn_get_port_list=ts_rtsyn_get_port_list;
rtsyn_ex->rtsyn_synth_start=ts_rtsyn_synth_start;
rtsyn_ex->rtsyn_synth_stop=ts_rtsyn_synth_stop;
rtsyn_ex->rtsyn_play_some_data=ts_rtsyn_play_some_data;
#if defined(IA_WINSYN) || defined(IA_W32G_SYN)
rtsyn_ex->rtsyn_buf_check=ts_rtsyn_buf_check;
#endif
rtsyn_ex->rtsyn_midiports_close=ts_rtsyn_midiports_close;

	return rtsyn_ex;
}
void destroy_rtsyn(rtsyn_ex_t* rtsyn){	
	timidity_mutex_destroy(rtsyn->busy);
	free(rtsyn);
}

