#include "NetworkSpeed.h"
#include <string.h>
#include <iostream>

#if !defined(WIN32)
#include <sys/errno.h>
#include <unistd.h>
#endif // !WIN32

using namespace std;

NetworkSpeed::NetworkSpeed()
 : mTotalBytes(0), mTransferedBytes(0)
{
	memset(&mStartTime, 0, sizeof(struct timeval));
    memset(&mSendBeginTime, 0, sizeof(struct timeval));
	memset(&mUpdateTime, 0, sizeof(struct timeval));
}

void NetworkSpeed::Start(unsigned int inTotalBytes)
{
	mTotalBytes = inTotalBytes;
	if (gettimeofday(&mStartTime, NULL) < 0)
	{
		DEBUG_CALL(printf("gettimeofday error: %s\n", strerror(errno)); fflush(stdout));
	}
}

void NetworkSpeed::StartSendTimer()
{
    if (gettimeofday(&mSendBeginTime, NULL) < 0)
	{
		DEBUG_CALL(printf("gettimeofday error: %s\n", strerror(errno)); fflush(stdout));
	}
}

void NetworkSpeed::Update(unsigned int inTransferedBytes)
{
	if ((mTransferedBytes + inTransferedBytes) <= mTotalBytes)
		mTransferedBytes += inTransferedBytes;
	else
    {
        DEBUG_CALL(printf("*** transfered more than total size: %u > %u\n",
            (mTransferedBytes + inTransferedBytes), mTotalBytes); fflush(stdout));
		mTransferedBytes = mTotalBytes;
    }
	
	if (gettimeofday(&mUpdateTime, NULL) < 0)
	{
		DEBUG_CALL(printf("gettimeofday error: %s\n", strerror(errno)); fflush(stdout));
	}
}

void NetworkSpeed::LimitSpeed(unsigned int inTransferedBytes, unsigned int inMaxBps)
{
    if (inMaxBps != 0)
    {
        int sec, usec = 0;
        unsigned int elapsedTime = 0;
        if ((usec = mUpdateTime.tv_usec - mSendBeginTime.tv_usec) < 0)
        {
            usec += 1000000;
            sec = mUpdateTime.tv_sec - mSendBeginTime.tv_sec - 1;
        }
        else
            sec = mUpdateTime.tv_sec - mSendBeginTime.tv_sec;
        elapsedTime = usec + (sec * 1000000);
        
        double maxBytesPerUsec = inMaxBps/ 8 / 1000000.0;
        double bytesPerUsec = inTransferedBytes / (double)elapsedTime;
        //DEBUG_CALL(printf("maxBPUS: %f BPUS: %f\n", maxBytesPerUsec, bytesPerUsec); fflush(stdout));
        if (bytesPerUsec > maxBytesPerUsec)
        {
            unsigned int sleepTime =
                (unsigned int)((bytesPerUsec / maxBytesPerUsec) * (double)elapsedTime) - elapsedTime;
            usleep(sleepTime);
        }
    }
}

unsigned int NetworkSpeed::BytesPerSecond()
{
	struct timeval updateTime;
	if (gettimeofday(&updateTime, NULL) < 0)
	{
		DEBUG_CALL(printf("gettimeofday error: %s\n", strerror(errno)); fflush(stdout));
	}
	
	if ((updateTime.tv_sec - mUpdateTime.tv_sec) < 60)
	{
		// if no update has happened in 1 minute then we
		// use the current time to show the transfer has stalled
		updateTime = mUpdateTime;
	}
	
	unsigned int elapsedTime =
		(((updateTime.tv_sec - mStartTime.tv_sec) * 1000) +
		((updateTime.tv_usec - mStartTime.tv_usec) / 1000));
	if (elapsedTime == 0)
		return mTransferedBytes;
	else
		return ((mTransferedBytes / elapsedTime) * 1000);
}

bool NetworkSpeed::IsStalled()
{
    // first make sure the transfer has started
    if (mStartTime.tv_sec > 0)
    {
        struct timeval nowTime;
        if (gettimeofday(&nowTime, NULL) < 0)
        {
            DEBUG_CALL(printf("gettimeofday error: %s\n", strerror(errno)); fflush(stdout));
        }
        
        if (mUpdateTime.tv_sec > 0)
        {
            if ((nowTime.tv_sec - mUpdateTime.tv_sec) >= 300)
            {
                // if no update has happened in 5 minutes
                // then i consider the transfer stalled
                // and the server will cancel it
                return true;
            }
        }
        else if ((nowTime.tv_sec - mStartTime.tv_sec) >= 300)
        {
            // if an update hasn't occured 5 minutes
            // since the transfer started, then consider it
            // stalled and cancel it
            return true;
        }
	}
	return false;
}

