#include "BCondition.h"
#if !defined(WIN32)
#include <sys/time.h>
#include <sys/errno.h>
#endif

BCondition::BCondition(int inValue)
	: mValue(inValue)
{
#if defined(USE_WIN32THREADS)
	mSemHandle = CreateSemaphore(0, 1, 1, 0);
#elif defined(USE_PTHREADS)
	int err = pthread_mutex_init(&mMutex, NULL);
	if (err == 0)
	{
		err = pthread_cond_init(&mCond, NULL);
		if (err != 0)
		{
			DEBUG_CALL(printf("error initing condition: %d\n", err));
		}
	}
	else
	{
		DEBUG_CALL(printf("error initing cond mutex: %d\n", err));
	}
#endif
}

BCondition::~BCondition()
{
#if defined(USE_WIN32THREADS)
	CloseHandle(mSemHandle);
#elif defined(USE_PTHREADS)
	int err = pthread_mutex_destroy(&mMutex);
	if (err == 0)
	{
		err = pthread_cond_destroy(&mCond);
		if (err != 0)
		{
			DEBUG_CALL(printf("error destroying condition: %d\n", err));
		}
	}
	else
	{
		DEBUG_CALL(printf("error destroying cond mutex: %d\n", err));
	}
#endif
}

void BCondition::Wait(int inWaitValue, int inSetValue)
{
#if defined(USE_WIN32THREADS)
	
	bool done = false;
	do
	{
		WaitForSingleObject(mSemHandle, INFINITE);
		{ // begin lock scope
			StMutexLock lock(mMutex);
			if (mValue == inWaitValue)
			{
				mValue = inSetValue;
				done = true;
			}
		} // end locking scope
	} while (!done);

#elif defined(USE_PTHREADS)
	int err = pthread_mutex_lock(&mMutex);
	if (err == 0)
	{
		while (mValue != inWaitValue && err == 0)
		{
			err = pthread_cond_wait(&mCond, &mMutex);
		}
		
		if (err == 0)
		{
			mValue = inSetValue;
		}
		else
		{
			DEBUG_CALL(printf("error waiting cond: %d\n", err));
		}
        err = pthread_mutex_unlock(&mMutex);
		if (err != 0)
		{
			DEBUG_CALL(printf("error unlocking cond mutex: %d\n", err));
		}
	}
	else
	{
		DEBUG_CALL(printf("error locking cond mutex: %d\n", err));
	}
#endif
}

bool BCondition::TimedWait(int inWaitValue, int inSetValue, int inTimeout)
{
#if defined(USE_WIN32THREADS)
// unimplemented since i don't use it and i'm moving to boost threads soon
	return false;
#elif defined(USE_PTHREADS)
	int err = pthread_mutex_lock(&mMutex);
	bool rValue = false;
    if (err == 0)
	{
        struct timeval now;
        struct timespec timeout;
        gettimeofday(&now, NULL);
        timeout.tv_sec = now.tv_sec + inTimeout; /* timeout is in seconds */
        timeout.tv_nsec = now.tv_usec * 1000; 
        while (mValue != inWaitValue && err == 0)
		{
			err = pthread_cond_timedwait(&mCond, &mMutex, &timeout);
		}
		
		if (err == 0)
		{
			mValue = inSetValue;
            rValue = true;
        }
        else if (err == ETIMEDOUT)
        {
            // don't do anything, the wait timed out
            DEBUG_CALL(printf("cond wait timed out\n"));
        }
		else
		{
			DEBUG_CALL(printf("error waiting cond: %d\n", err));
		}
        err = pthread_mutex_unlock(&mMutex);
		if (err != 0)
		{
			DEBUG_CALL(printf("error unlocking cond mutex: %d\n", err));
		}
	}
	else
	{
		DEBUG_CALL(printf("error locking cond mutex: %d\n", err));
	}
    return rValue;
#endif
}

void BCondition::Signal(int inValue)
{
#if defined(USE_WIN32THREADS)
	
	StMutexLock lock(mMutex);
	mValue = inValue;
	ReleaseSemaphore(mSemHandle, 1, 0);

#elif defined(USE_PTHREADS)
	int err = pthread_mutex_lock(&mMutex);
	if (err == 0)
	{
		mValue = inValue;
		err = pthread_cond_signal(&mCond);
		if (err == 0)
		{
			err = pthread_mutex_unlock(&mMutex);
			if (err != 0)
			{
				DEBUG_CALL(printf("error unlocking cond mutex: %d\n", err));
			}
		}
		else
		{
			DEBUG_CALL(printf("error signaling cond: %d\n", err));
		}
	}
	else
	{
		DEBUG_CALL(printf("error locking cond mutex: %d\n", err));
	}
#endif
}

