#include "HLServer.h"
#include "HLChat.h"
#include "HLClient.h"
#include "HLProtocol.h"
#include "TCPSocket.h"
#include <algorithm>
#include "ServerLog.h"
#include "FileUtils.h"
#include "AsyncDNS.h"

#if !defined(WIN32)
#include <unistd.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <errno.h>
#endif // !WIN32

#if defined(_HTTPD_)
#include "WebServer.h"
#endif

using namespace FileUtils;

HLServer *gServer = NULL;

typedef struct
{
	u_int16_t version;
	u_int16_t port;
	u_int16_t nusers;
	u_int16_t reserved;
	u_int32_t id;
} TrackerPacket;

class HLTrackerRegister : public BThread, public AsyncDNSCallback
{
protected:
	void HostLookup(const string &inDomainName, const struct in_addr *inAddr);

private:
	void Run();
	int mSock;
};

void HLTrackerRegister::HostLookup(const string &inName, const struct in_addr *inAddr)
{
	if (inAddr)
	{
		char packetBuf[sizeof(TrackerPacket) + 1024];
		TrackerPacket *trackerPacket = (TrackerPacket *)packetBuf;

		{ // begin lock scope
			StMutexLock lock(gServer->mLock);

			TrackerList::iterator iter = gServer->mConf.trackerList.begin();
			while (iter != gServer->mConf.trackerList.end())
			{
				if ((*iter).addr == inName)
				{
					trackerPacket->version = htons(0x0001);
					trackerPacket->port = htons(gServer->mConf.serverPort);
					trackerPacket->nusers = htons(gServer->mClientList.size());
					trackerPacket->reserved = 0;
					//pkt.id = htonl(0xefe44d00); // ?? b20 ?
					//pkt.id = htonl(0xa67985b1); // 1.2.3
					//pkt.id = htonl(0x486d1f7c); // AfterBirth a2c3
					//pkt.id = htonl(0x7f23a207); // 1.8.4
					trackerPacket->id = htonl(0x74617261); // i <3 tara

					u_int16_t pos = sizeof(TrackerPacket);
					string tempString = gServer->mConf.serverName;
					FileUtils::HomeToNetworkString(tempString);
					packetBuf[pos++] = tempString.length();
					memcpy(&packetBuf[pos], tempString.c_str(), tempString.length());
					pos += tempString.length();
					tempString = gServer->mConf.serverDescription;
					FileUtils::HomeToNetworkString(tempString);
					packetBuf[pos++] = tempString.length();
					memcpy(&packetBuf[pos], tempString.c_str(), tempString.length());
					pos += tempString.length();

					int sendSize = pos;
					// add password for tracker to packet
					if ((*iter).pass.empty())
					{
						packetBuf[sendSize++] = 0;
					}
					else
					{
						packetBuf[sendSize++] = (*iter).pass.length();
						memcpy(&packetBuf[sendSize], (*iter).pass.c_str(), (*iter).pass.length());
						sendSize += (*iter).pass.length();
					}

					// send packet to tracker
					struct sockaddr_in saddr;
					saddr.sin_family = AF_INET;
					saddr.sin_port = htons(5499);
					saddr.sin_addr = (*inAddr);
					sendto(mSock, packetBuf, sendSize, 0, (struct sockaddr *)&saddr, sizeof(saddr));
					return;
				}
				iter++;
			}
		} // end lock scope
	}
}

void HLTrackerRegister::Run()
{
	struct sockaddr_in saddr;

	if ((mSock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) == -1)
	{
		ServerLog::ErrorLog(__FILE__, __LINE__,
			"unable to create tracker socket: %s", strerror(errno));
		return;
	}

	saddr.sin_family = AF_INET;
	saddr.sin_port = 0; /* htons(32769); */
	saddr.sin_addr.s_addr = 0;

	if (bind(mSock, (struct sockaddr *)&saddr, sizeof saddr) == -1)
	{
		ServerLog::ErrorLog(__FILE__, __LINE__,
			"unable to bind tracker socket: %s", strerror(errno));
		return;
	}

	while(true)
	{

		{ // begin lock scope
			StMutexLock lock(gServer->mLock);
			TrackerList::iterator iter = gServer->mConf.trackerList.begin();
			while (iter != gServer->mConf.trackerList.end())
			{
				AsyncDNS dns;
				dns.HostByName((*iter).addr, this);
				// code continues in callback...
				iter++;
			}
		} // end lock scope

		sleep(300);
	}

	closesocket(mSock);
}

class HLFileServer : public AsyncTCPSocket
{
protected:
	void Start(u_int16_t inPort);
	void OnAccept();

	friend class HLServer;
};



void HLFileServer::Start(u_int16_t inPort)
{
	Open();
	Bind(inPort);
	SetSelector(gServer->mSelector);
	Listen();
}

void HLFileServer::OnAccept()
{
	// accept a connection on a new socket
	struct sockaddr_in remoteAddr;
	int fd = Accept( remoteAddr );
	HLTransferSocket *clientSocket = new HLTransferSocket( fd, &remoteAddr );

	// the socket is created by accept from a non-blocking socket
	// it will be non-blocking also, so we change it to blocking
	//clientSocket->SetNonBlock(false);

	// first we make sure this host is connected to the server and logged in
	StMutexLock lock(gServer->mLock);
	bool foundHost = false;
	ClientList::iterator iter = gServer->mClientList.begin();
	while (iter != gServer->mClientList.end())
	{
		if ((*iter)->RemoteHost() == clientSocket->RemoteHost() &&
			(*iter)->User().IsLoggedIn())
		{
			foundHost = true;
			break;
		}
		iter++;
	}

	if (foundHost)
	{
		new HLTransfer(clientSocket);
		clientSocket->SetSelector(gServer->mSelector);
	}
	else
	{
		delete clientSocket;
	}
}

HLServer::HLServer(ServerConf &inConf)
: mTotalConnections(0), mStartTime(0), mRunning(false),
mFileServer(NULL), mTransAM(NULL), mTrackerReg(NULL), mConf(inConf)
{
	assert(mConf.serverPort != 0);

	//2003/08/01 added by ortana.
#ifdef WIN32
	this->hlBan.load();
#endif//WIN32

	gServer = this;
}


HLServer::~HLServer()
{
	Stop();
	gServer = NULL;
}

//2003/07/31 added by ortana,
HLDataBase * HLServer::getDataBase()
{
	return &mDataBase;

}

ClientList& HLServer::getClentList()
{
	return mClientList;
}

void HLServer::ReloadConf()
{
	StMutexLock lock(mLock);
	mConf.ReloadConf();
#if !defined(_MYSQL_)
	// reload the news file also
	mDataBase.Init();
#endif // _MYSQL_

	// reload all accounts for connected users
	list<string> accountList;
	ClientList::iterator iter = mClientList.begin();
	while (iter != mClientList.end())
	{
		if (find(accountList.begin(), accountList.end(),
			(*iter)->User().Login()) == accountList.end())
		{
			HLAccount modAccount;
			modAccount.SetLogin((*iter)->User().Login());
			if (mDataBase.FetchAccount(modAccount, false))
				UpdateAccountForClients(modAccount);
			// add the login to a list, so we don't update the same account twice
			accountList.push_back((*iter)->User().Login());
		}
		iter++;
	}
}

void HLServer::Start()
{
	if (!mRunning)
	{
		FileUtils::Init();

		mStartTime = time(0);
		mRunning = true;

		mUserIDCounter = 1;
#if defined(_MYSQL_)
		mDataBase.Connect(mConf.databaseAddress, mConf.databaseUser,
			mConf.databasePassword, mConf.databaseName);
#endif
		mDataBase.Init();

		// new transfer manager
		mTransAM = new HLTransferManager();

		// start file server
		mFileServer = new HLFileServer();
		mFileServer->Start(mConf.serverPort + 1);

		// start tracker register
		mTrackerReg = new HLTrackerRegister();
		mTrackerReg->Create();

#if defined(_HTTPD_)
		// start embedded web server
		mWebServer = new WebServer();
		mWebServer->Create();
#endif

		Open();
		Bind(mConf.serverPort);
		SetSelector(mSelector);
		Listen();

#ifdef WIN32
		//2003/07/30 added by ortana.
		ServerLog::OtherLog( "info" , "WinTerra Server started" );
#endif//WIN32

#if defined(CONFIG_RENDEZVOUS)
		{
			StMutexLock lock(mLock);
			mDNSServiceRegistrar.RegisterService(mConf.serverName,
				string("_hotline._tcp"), string(""), mConf.serverPort, this);
			// register the service
#if defined(_HTTPD_)
			// don't forget the embedded web server
			mDNSServiceRegistrar.RegisterService(mConf.serverName,
				string("_http._tcp"), string(""), mConf.serverPort + 2, this);
#endif // _HTTPD_
		}
#endif // CONFIG_RENDEZVOUS


#if defined(CONFIG_LINKING)
		if (!mConf.linkServer.empty())
			mLinkManager.EstablishLinkWithHub(mConf.linkServer, 6969);
#endif

		time_t sleepTime = SLEEP_TIME;
		time_t elapsedTime = 0;
		time_t keepaliveTimer = 0;
		while (mRunning)
		{
			try
			{
				// this time shit needs to be made more accurate with
				// gettimeofday or something
				time_t startTime = time(0);
				mSelector.WaitEvent(sleepTime);
				time_t currentTime = time(0);

				if ((currentTime - startTime) > 0)
					elapsedTime += (currentTime - startTime);
				else
					elapsedTime++;

				StMutexLock lock(mLock);
				ClientList::iterator iter = mClientList.begin();

				if (elapsedTime >= SLEEP_TIME)
				{
					sleepTime = SLEEP_TIME;
					elapsedTime = 0;
					while (iter != mClientList.end())
					{
						if ((*iter)->User().IsLoggedIn() && (*iter)->User().LoginTime())
						{
							if (!((*iter)->User().Status() & IDLE_MASK) &&
								(currentTime - (*iter)->User().LastRecvTime()) >= mConf.idleTimeout)
							{
								(*iter)->User().SetStatus((*iter)->User().Status() | IDLE_MASK);
								HLPacket userChangePacket(HTLS_HDR_USER_CHANGE, 0, 0);
								userChangePacket.AddStringObject(HTLS_DATA_NAME, (*iter)->User().Name());
								userChangePacket.AddUInt16Object(HTLS_DATA_UID, (*iter)->User().ID());
								userChangePacket.AddUInt16Object(HTLS_DATA_ICON, (*iter)->User().Icon());
								userChangePacket.AddUInt16Object(HTLS_DATA_COLOUR, (*iter)->User().Status());
								SendToAll(userChangePacket);
							}
						}
						else if ((currentTime - (*iter)->User().LastRecvTime()) >= LOGIN_TIMEOUT)
						{
							// disconnect clients that don't login
							// adding the client to the kick list will slow down
							// brute force password guessing bots because they will get temp
							// banned if they incorrectly login too many times
							AddClientToKickList(*(*iter));
							(*iter)->Disconnect();
						}
						iter++;
					}

					// clean up stalled transfers
					mTransAM->CancelStalledTransfers();

					keepaliveTimer += elapsedTime;
					// every 4 minutes send an empty packet to all queued transfers
					// and clients this prevents timeout for masqueraded connections
					if (keepaliveTimer >= (4 * 60))
					{
						mTransAM->KeepAliveQueuedTransfers();
						KeepAliveConnections();
						keepaliveTimer = 0;
					}
				}
				else
				{
					sleepTime = (SLEEP_TIME - elapsedTime);
				}

				// delete and remove disconnected clients
				iter = mClientList.begin();
				while (iter != mClientList.end())
				{
					if ((*iter)->IsConnected())
					{
						iter++;
					}
					else
					{
						delete (*iter);
						iter = mClientList.erase(iter);
						DEBUG_CALL(printf("%u clients connected\n",
							(unsigned int)mClientList.size()); fflush(stdout));
					}
				}
			}
			catch (socket_error &exp)
			{
				//2003/07/27 Added by Ortana
				ServerLog::ErrorLog(__FILE__, __LINE__, "socket error in server loop");
				DEBUG_CALL(printf("socket error in server loop: %s\n", exp.what()); fflush(stdout));
			}
			catch (logic_error &exp)
			{
				//2003/07/27 Added by Ortana
				ServerLog::ErrorLog(__FILE__, __LINE__, "logic error in server loop");
				DEBUG_CALL(printf("logic error in server loop: %s\n", exp.what()); fflush(stdout));
			}
			catch (runtime_error &exp)
			{
				//2003/07/27 Added by Ortana
				ServerLog::ErrorLog(__FILE__, __LINE__, "runtime error in server loop");
				DEBUG_CALL(printf("runtime error in server loop: %s\n", exp.what()); fflush(stdout));
			}
			catch (exception &exp)
			{
				//2003/07/27 Added by Ortana
				ServerLog::ErrorLog(__FILE__, __LINE__, "other exception in server loop");
				DEBUG_CALL(printf("other exception in server loop: %s\n", exp.what()); fflush(stdout));
			}
			catch (...)
			{
				//2003/07/27 Added by Ortana
				ServerLog::ErrorLog(__FILE__, __LINE__, "unknown exception in server loop");
				DEBUG_CALL(printf("error unknown exception in server loop."); fflush(stdout));
			}
		}

		// clean up server for exit
		// BEGIN CLEAN UP
		DEBUG_CALL(printf("server is exiting\n"); fflush(stdout));

#if defined(CONFIG_RENDEZVOUS)
		mDNSServiceRegistrar.UnregisterService(string("Terra"));
#if defined(_HTTPD_)
		// don't forget the embedded web server
		mDNSServiceRegistrar.UnregisterService(string("Terra-web"));
#endif // _HTTPD_
#endif // CONFIG_RENDEZVOUS

		delete mFileServer;
		mFileServer = NULL;

		delete mTrackerReg;
		mTrackerReg = NULL;

#if defined(_HTTPD_)
		delete mWebServer;
		mWebServer = NULL;
#endif // _HTTPD_


		DEBUG_CALL(printf("disconnecting all clients\n"); fflush(stdout));
		{
			StMutexLock lock(mLock);
			// tell all clients to disconnect
			ClientList::iterator iter = mClientList.begin();
			while (iter != mClientList.end())
			{
				(*iter)->Disconnect();
				delete (*iter);
				iter = mClientList.erase(iter);
			}
		}

		DEBUG_CALL(printf("deleting transfer manager\n"); fflush(stdout));
		delete mTransAM;
		mTransAM = NULL;
	}
}

void HLServer::Stop()
{
	if (mRunning)
	{
		mRunning = false;
		Close();
	}
}

void HLServer::KeepAliveConnections()
{
	// called every 4 minutes by HLServer::Start
	ClientList::iterator iter = mClientList.begin();
	while (iter != mClientList.end())
	{
		(*iter)->SendKeepAlive();
		iter++;
	}	
}

bool HLServer::IsRunning()
{
	return mRunning;
}

void HLServer::FoundSOCKSHost(const string &inAddress, int inVersion)
{
	StMutexLock lock(mLock);
	ClientList::iterator iter = mClientList.begin();
	while (iter != mClientList.end())
	{
		if ((*iter)->Address() == inAddress)
		{
			if (mConf.socksDetectLevel == 1)
			{
				HLPacket chatPacket(HTLS_HDR_CHAT, 0, 0);

				//2003/07/30 added by ortana.
#ifdef _JAPAN_
				string chatString("\r               <<");
				chatString.append((*iter)->User().Name());
				chatString.append(" ̓vNV[gpĂ܂B");
				chatString.append(">>");
#else
				string chatString("\r<");
				chatString.append((*iter)->User().Name());
				chatString.append(" is using a SOCKS proxy");
				chatString.append(">");
#endif

				chatPacket.AddStringObject(HTLS_DATA_CHAT, chatString);
				gServer->SendToAll(chatPacket, CanKickFilter);
			}
			else if (mConf.socksDetectLevel == 2)
			{
				if (inVersion == 4)
					(*iter)->Ban(serverUser, kBanReasonSOCKSv4);
				else if (inVersion == 5)
					(*iter)->Ban(serverUser, kBanReasonSOCKSv5);
			}
		}
		iter++;
	}
}

void HLServer::OnAccept()
{
	StMutexLock lock(mLock);

	struct sockaddr_in remoteAddr;
	int fd = Accept(remoteAddr);
	HLSocket *clientSocket = new HLSocket( fd, &remoteAddr );

//2003/08/01 added by ortana.
#ifdef WIN32
	if( hlBan.isBan( (clientSocket->RemoteHostName()).c_str() ) )
	{
		DEBUG_CALL(printf("host is banned\n"));
		delete clientSocket;
		return;
	}
#endif

	if (!mDataBase.IsHostBanned(clientSocket->RemoteHostName()))
	{

		u_int16_t clientCounter = 0;

		ClientList::iterator iter = mClientList.begin();
		while (iter != mClientList.end())
		{
			if ((*iter)->RemoteHost() == clientSocket->RemoteHost())
				clientCounter++;
			iter++;
		}

		if (clientCounter < mConf.maxClientsPerAddress)
		{
			DEBUG_CALL(printf("new connection accepted\n"));
			clientSocket->SetSelector(mSelector);
			mTotalConnections++;
			new HLClient(clientSocket);
		}
		else
		{
			DEBUG_CALL(printf("host has too many clients\n"));
			delete clientSocket;
		}
	}
	else
	{
		DEBUG_CALL(printf("host is banned\n"));
		delete clientSocket;
	}
}

void HLServer::SendToAll(HLPacket &inPacket, PacketFilterProc inFilter)
{
	ClientList::iterator iter = mClientList.begin();
	while (iter != mClientList.end())
	{
		if (inFilter)
		{	
			if ((*inFilter)((*iter)->User()))
				(*iter)->SendPacket(inPacket);
		}
		else if ((*iter)->User().IsLoggedIn() && (*iter)->User().LoginTime())
			(*iter)->SendPacket(inPacket);
		iter++;
	}
}

void HLServer::SendToOthers(HLPacket &inPacket, HLClient &inClient, PacketFilterProc inFilter)
{
	ClientList::iterator iter = mClientList.begin();
	while (iter != mClientList.end())
	{
		if ((*iter) != (&inClient))
		{
			if (inFilter)
			{	
				if ((*inFilter)((*iter)->User()))
					(*iter)->SendPacket(inPacket);
			}
			else if ((*iter)->User().IsLoggedIn() && (*iter)->User().LoginTime())
				(*iter)->SendPacket(inPacket);
		}
		iter++;
	}
}

void HLServer::SendTo(HLPacket &inPacket, u_int16_t inUserID)
{
	ClientList::iterator iter = mClientList.begin();
	while (iter != mClientList.end())
	{
		if ((*iter)->User().ID() == inUserID &&
			(*iter)->User().IsLoggedIn() && (*iter)->User().LoginTime())
		{
			(*iter)->SendPacket(inPacket);
			return;
		}
		iter++;
	}
}

bool HLServer::ReadNewsFilter(HLUser &inUser)
{
	if (inUser.IsLoggedIn() && inUser.LoginTime() && inUser.Access().read_news)
		return true;
	else
		return false;
}

bool HLServer::ReadChatFilter(HLUser &inUser)
{
	if (inUser.IsLoggedIn() && inUser.LoginTime() && inUser.Access().read_chat)
		return true;
	else
		return false;
}

bool HLServer::CanKickFilter(HLUser &inUser)
{
	if (inUser.IsLoggedIn() && inUser.LoginTime() && inUser.Access().disconnect_users)
		return true;
	else
		return false;
}

u_int16_t HLServer::CreateNewUserID()
{
#define MAX_USER_ID 0xFFFF
	//#define MAX_USER_ID 100 // this is for testing
	u_int16_t newID = mUserIDCounter;
	ClientList::iterator iter = mClientList.begin();
	while (iter != mClientList.end())
	{
		if ((*iter)->User().ID() == newID)
		{
			if (newID < MAX_USER_ID)
				newID++;
			else
				newID = 1;
			// start over and see if anyone is using this new id
			iter = mClientList.begin();
		}
		else
			iter++;
	}
	if (newID < MAX_USER_ID)
		mUserIDCounter = (newID + 1);
	else
		mUserIDCounter = 1;
	return newID;
}

HLClient *HLServer::GetClientForID(u_int16_t inUserID)
{
	// must lock until you finish using the pointer returned here
	ClientList::iterator iter = mClientList.begin();
	while (iter != mClientList.end())
	{
		if ((*iter)->User().ID() == inUserID && (*iter)->User().IsLoggedIn())
		{
			return (*iter);
		}
		iter++;
	}
	return NULL;
}

#if defined(CONFIG_LINKING)
void HLServer::RemoveClientWithGlobalID(u_int32_t inGlobalID)
{
	ClientList::iterator iter = mClientList.begin();
	while (iter != mClientList.end())
	{
		if ((*iter)->User().GlobalID() == inGlobalID)
		{
			// have to call OnClose here because it does special stuff
			(*iter)->Cleanup();
			delete (*iter);
			mClientList.erase(iter);
			return;
		}
		iter++;
	}
}

HLClient *HLServer::GetClientWithGlobalID(u_int32_t inGlobalID)
{
	// must lock until you finish using the pointer returned here
	ClientList::iterator iter = mClientList.begin();
	while (iter != mClientList.end())
	{
		if ((*iter)->User().GlobalID() == inGlobalID)
		{
			return (*iter);
		}
		iter++;
	}
	return NULL;
}

void HLServer::RemoveRemoteClients()
{
	ClientList::iterator iter = mClientList.begin();
	while (iter != mClientList.end())
	{
		if ((*iter)->IsRemote())
		{
			// have to call OnClose here because it does special stuff
			(*iter)->Cleanup();
			delete (*iter);
			iter = mClientList.erase(iter);
		}
		else
			iter++;
	}
}

#endif

bool HLServer::IsValidChat(HLChat &inChat)
{
	// must lock mChatListLock around use of this function
	ChatList::iterator iter	=
		find(mChatList.begin(), mChatList.end(), &inChat);
	return (iter != mChatList.end());
}

bool HLServer::IsClientInChat(HLClient &inClient, HLChat &inChat)
{
	// the client uses this to make sure a user isn't invited to
	// a chat it is already in and to make sure the inviting client
	// is actually in the chat
	ChatList::iterator iter	=
		find(mChatList.begin(), mChatList.end(), &inChat);
	if (iter != mChatList.end())
	{
		return (*iter)->IsClientInChat(inClient);
	}
	return false;
}

u_int8_t HLServer::CountChatsForClient(HLClient &inClient)
{
	// the client uses this to make sure the max number of chats
	// per client has not been reached for this client
	u_int8_t chatCount = 0;
	ChatList::iterator iter	= mChatList.begin();
	while (iter != mChatList.end())
	{
		if ((*iter)->IsClientInChat(inClient))
			chatCount++;
		iter++;
	}
	return chatCount;
}

u_int8_t HLServer::TotalChats()
{
	// the client checks this to make sure the max number of chats
	// for the server has not been reached
	return (u_int8_t)mChatList.size();
}

void HLServer::SetSubject(const string &inSubject)
{
	mSubject = inSubject;
}

void HLServer::GetSubject(string &outSubject)
{
	outSubject = mSubject;
}

void HLServer::UpdateAccountForClients(const HLAccount &inAccount)
{
	// when an account is modified
	// this updates the account for the user
	ClientList::iterator iter = mClientList.begin();
	while (iter != mClientList.end())
	{
		if ((*iter)->User().IsLoggedIn())
		{
			if ((*iter)->User().Login() == inAccount.Login())
			{
				if (inAccount.Access().disconnect_users && !((*iter)->User().Status() & ADMIN_MASK))
					(*iter)->User().SetStatus((*iter)->User().Status() | ADMIN_MASK);
				else if (!inAccount.Access().disconnect_users && ((*iter)->User().Status() & ADMIN_MASK))
					(*iter)->User().SetStatus((*iter)->User().Status() ^ ADMIN_MASK);

				(*iter)->User().SetAccount(inAccount);

				// modify user in database
				mDataBase.UpdateUser(((*iter)->User()));

				HLPacket userChangePacket(HTLS_HDR_USER_CHANGE, 0, 0);
				userChangePacket.AddStringObject(HTLS_DATA_NAME, (*iter)->User().Name());
				userChangePacket.AddUInt16Object(HTLS_DATA_UID, (*iter)->User().ID());
				userChangePacket.AddUInt16Object(HTLS_DATA_ICON, (*iter)->User().Icon());
				userChangePacket.AddUInt16Object(HTLS_DATA_COLOUR, (*iter)->User().Status());

				// need to use SendToOthers if it is 185 (and this still applies)
				SendToAll(userChangePacket);

				// notify the client of their new privileges
				(*iter)->SendSelfInfo();
			}
		}
		iter++;
	}
}

void HLServer::KickClientsWithAccount(const string &inAccount)
{
	ClientList::iterator iter = mClientList.begin();
	while (iter != mClientList.end())
	{
		if ((*iter)->User().IsLoggedIn())
		{
			if ((*iter)->User().Login() == inAccount)
				(*iter)->Disconnect();
		}
		iter++;
	}
}

void HLServer::AddClientToKickList(HLClient &inClient)
{
	KickList::iterator iter = mKickList.begin();
	time_t currentTime = time(0);
	while (iter != mKickList.end())
	{
		// if the ip or the nick name match
		if (inClient.RemoteHost() == (*iter).host ||
			inClient.User().RealName() == (*iter).nick)
		{
			if (currentTime - (*iter).time <= mConf.banTimeout)
			{
				(*iter).host = inClient.RemoteHost();
				(*iter).nick = inClient.User().RealName();
				(*iter).count++;
			}
			else
			{
				// restart the counter
				(*iter).count = 1;
				(*iter).nick = inClient.User().RealName();
				(*iter).time = currentTime;
				(*iter).host = inClient.RemoteHost();
			}
			break;
		}
		iter++;
	}

	if (iter == mKickList.end())
	{
		// not found so create a new entry and append it
		kick_entry kickEntry;
		kickEntry.host = inClient.RemoteHost();
		kickEntry.nick = inClient.User().RealName();
		kickEntry.time = currentTime;
		kickEntry.count = 1;
		mKickList.push_back(kickEntry);
	}
	else if ((*iter).count >= KICK_STRIKES)
	{
		// he's ooooouutttaaa hhheeerrree!!!!!!!
		inClient.Ban(serverUser, kBanReasonStrikes, mConf.banTimeout);
	}

	// clean up stale entries in the list
	if (mKickList.size() >= 1000)
	{
		iter = mKickList.begin();
		while (iter != mKickList.end())
		{
			if (currentTime - (*iter).time > mConf.banTimeout)
				iter = mKickList.erase(iter);
			else
				iter++;
		}
	}
}

#if defined(CONFIG_RENDEZVOUS)
void HLServer::RegisterReply(const string &inServiceName, int inErrorCode)
{
	DEBUG_CALL(
		printf("Rendezvous RegisterReply(%s): %d\n", inServiceName.c_str(), inErrorCode);
	fflush(stdout));
}
#endif // CONFIG_RENDEZVOUS

void HLServer::FormatChat(HLClient &inClient, unsigned int inStyle,
													string &ioChatString, unsigned int &outLineCount)
{
	const char *chatStyleString1 = inStyle == 1 ? " -> %s " : " %11.11s:  ";
	const char *chatStyleString2 = inStyle == 1 ? "\r -> %s %s" : "\r %11.11s:  %s";
	char chatBuf[CHAT_LIMIT + USERNAME_MAXLEN + 32];
	snprintf(chatBuf, CHAT_LIMIT + USERNAME_MAXLEN + 31,
		chatStyleString1, inClient.User().Name().c_str());
	string::size_type pos = 0;
	u_int16_t len = strlen(chatBuf);

	outLineCount = 1;
	while ((pos = ioChatString.find('\r', pos)) != string::npos)
	{
		ioChatString.insert(pos + 1, chatBuf);
		outLineCount++;
		pos += len;
	}
	snprintf(chatBuf, CHAT_LIMIT + USERNAME_MAXLEN + 31,
		chatStyleString2, inClient.User().Name().c_str(), ioChatString.c_str());
	ioChatString.assign(chatBuf);
}

void HLServer::FormatNewsPost(const string &inName, time_t inTime,
															const string &inPost, string &outFormattedPost)
{
	struct tm tm;
	char timeBuf[256];
	string thePost = inPost;

	localtime_r(&inTime, &tm);

	//2003/07/30 added by ortana.
#ifndef _JAPAN_
	strftime(timeBuf, 255, " %a %m/%d/%Y %I:%M:%S %p\r\r", &tm);
	outFormattedPost.assign("Posted by ");
	outFormattedPost.append(inName);
	outFormattedPost.append(timeBuf);
	fixLineEndings(thePost);
	outFormattedPost.append(thePost);
	outFormattedPost.append("\r\r_________________________________________________________\r");
#else
	strftime(timeBuf, 255, "%y/%m/%d/ %I:%M %p\r\r", &tm);
	outFormattedPost.assign("O: ");
	outFormattedPost.append(inName);
	outFormattedPost.append(" e:");
	outFormattedPost.append(timeBuf);
	fixLineEndings(thePost);
	outFormattedPost.append(thePost);
	outFormattedPost.append("\r\r_________________________________________________________\r");
#endif//_JAPAN_
}
