#include "AsyncDNS.h"
#include "BThread.h"
#include "BMutex.h"
#include <list>
#include "BCondition.h"

#if !defined(WIN32)
#include <netdb.h>
#include <arpa/inet.h>
#endif

class AsyncDNSLock
{
public:
	AsyncDNSLock() { mLock.Lock(); }
	~AsyncDNSLock() { mLock.Unlock(); }
protected:
	static BMutex mLock;
};
BMutex AsyncDNSLock::mLock;

template <typename T> class ProtectedStack
{
public:
	typedef list<T> list_type;
	
	ProtectedStack() {}
	~ProtectedStack() {}
	
	void Push(const T &inValue)
	{
		StMutexLock lock(mLock);
		mStack.push_back(inValue);
	}
	
	void Pop(T &outValue)
	{
		StMutexLock lock(mLock);
		outValue = mStack.front();
		mStack.pop_front();
	}
	
	bool Empty()
	{
		StMutexLock lock(mLock);
		return mStack.empty();
	}
	
protected:
	list_type mStack;
	BMutex mLock;
};

typedef struct
{
	AsyncDNSCallback *callback;
	string name;
	string host;
} AsyncLookupData;

class LookupStack : public ProtectedStack<AsyncLookupData>
{
public:
	void RemoveCallback(const AsyncDNSCallback *inCallback)
	{
		StMutexLock lock(mLock);
		list_type::iterator iter = mStack.begin();
		while (iter != mStack.end())
		{
			if (iter->callback == inCallback)
				iter = mStack.erase(iter);
			else
				iter++;
		}
	}
};

class AsyncDNSLookupThread : public BThread
{
public:
	AsyncDNSLookupThread() {}
	virtual ~AsyncDNSLookupThread() {}
	
	void AddHostLookup(const string &inName, AsyncDNSCallback *inCallback)
	{
		AsyncLookupData lookupData;
		lookupData.name = inName;
		lookupData.callback = inCallback; 
		mStack.Push(lookupData);
		mCondition.Signal(true);
	}
	
	void AddNameLookup(const string &inHost, AsyncDNSCallback *inCallback)
	{
		DEBUG_CALL(printf("adding name lookup: %s\n", inHost.c_str()));
		AsyncLookupData lookupData;
		lookupData.host = inHost;
		lookupData.callback = inCallback;
		mStack.Push(lookupData);
		mCondition.Signal(true);
	}
	
	void RemoveCallback(const AsyncDNSCallback *inCallback)
	{
		AsyncDNSLock alock();
		mStack.RemoveCallback(inCallback);
	}

private:
	void Run();
	
	LookupStack mStack;
	BCondition mCondition;
};

void AsyncDNSLookupThread::Run()
{
	while (true)
	{
		try
		{
			mCondition.Wait(true, false);
			while (!mStack.Empty())
			{
				{ // begin lock scope
					AsyncDNSLock lock();
					AsyncLookupData lookupData;
					mStack.Pop(lookupData);
					
					if (lookupData.name.size())
					{
						// this is a name lookup
						DEBUG_CALL(printf("looking up host for name: %s\n", lookupData.name.c_str()));
						struct hostent *he = gethostbyname(lookupData.name.c_str());
						if (lookupData.callback)
						{
							if (he)
							{
								struct in_addr addr;
								memcpy(&addr, he->h_addr_list[0],
									((u_int32_t)he->h_length) > sizeof(addr) ? sizeof(addr) : he->h_length);
								lookupData.callback->HostLookup(lookupData.name, &addr);
							}
							else
								lookupData.callback->HostLookup(lookupData.name, NULL);
						}
					}
					else if (lookupData.host.size())
					{
						// this is a host lookup
						struct in_addr addr;
						string hostName;
						DEBUG_CALL(printf("looking up name for host: %s\n", lookupData.host.c_str()));
						if (inet_aton(lookupData.host.c_str(), &addr))
						{
							struct hostent *he = gethostbyaddr((char *)&addr, sizeof(addr), AF_INET);
							if (he)
								hostName = he->h_name;
						}
						lookupData.callback->NameLookup(lookupData.host, hostName);
					}
				} // end lock scope
			}
		}
		catch (...)
		{
			DEBUG_CALL(printf("uncaught exception in async lookup\n"));
		}
	}
}

AsyncDNSCallback::~AsyncDNSCallback()
{
	if (AsyncDNS::mThread)
		AsyncDNS::mThread->RemoveCallback(this);
}

// -------------------------------------------- AsyncDNS public API

AsyncDNSLookupThread *AsyncDNS::mThread = NULL;

AsyncDNS::AsyncDNS()
{
	if (mThread == NULL)
	{
		AsyncDNSLock lock();
		mThread = new AsyncDNSLookupThread();
		mThread->Create();
	}
}

void AsyncDNS::HostByName(const string &inName, AsyncDNSCallback *inCallback)
{
	struct in_addr addr;
	if (!inet_aton(inName.c_str(), &addr))
		// have to use gethostbyname
		mThread->AddHostLookup(inName, inCallback);
	else
	{
		try
		{
			if (inCallback)
				inCallback->HostLookup(inName, &addr);
		}
		catch (...)
		{
			DEBUG_CALL(printf("uncaught exception in HostByName\n"));
		}
	}
}

void AsyncDNS::ResolveHost(const string &inHost, AsyncDNSCallback *inCallback)
{
	mThread->AddNameLookup(inHost, inCallback);
}
