// --------------------------------------------------------------------
// wm3d - A Flash Molecular Viewer
//
// Copyright (c) 2011-2013, tamanegi (tamanegi@users.sourceforge.jp)
// All rights reserved.
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.
// --------------------------------------------------------------------

/**
  Container for Alias

  In wm3d, an alias is an xml element. Aliases can have their own default values.
  For example, an alias of &lt;HELIX elem="RIBBON" color="yellow" /$gt; is
  RIBBON element with yellow default color.
**/

class WMAlias {
  /**
    XML element name of this alias; no default
  **/
  @:isVar public var name( get, set ):String;
    /**
      getter for `name`
    **/
    public function get_name():String { return( name ); }
    /**
      setter for `name`
    **/
    public function set_name( n:String ):String {
      name = n;
      return( name );
    }
  /**
    target element name of this alias; no default
  **/
  @:isVar public var elem( get, set ):String;
    /**
      getter for `elem`
    **/
    public function get_elem():String { return( elem ); }
    /**
      setter for `elem`
    **/
    public function set_elem( e:String ):String {
      elem = e;
      return( elem );
    }
  /**
    actual instance of this alias; no default.
    Classes such as WMAtom, WMBond, WMRibbon will be used here.
  **/
  @:isVar public var mybase( get, null ):Dynamic;
    /**
      getter of `mybase`
    **/
    public function get_mybase():Dynamic { return( mybase ); }

  // ####################################################################

  /**
    Constructor. Use `register` function to generate alias.
    This constructor cannot set aliases.
  **/
  public function new() {
    name = "";
    elem = "";
    mybase = null;
  }

  /**
    register new alias of `x` as name `t`.
  **/
  public function register( t:String,
                            x:Xml ):Void {
    name = t;
    elem = x.get( "elem" ).toUpperCase();
    switch( elem ) {
      case "ATOM":
        mybase = new WMAtom();
        mybase.copyFrom( WMDefaults.gl_Atom );
      case "BOND":
        mybase = new WMBond();
        mybase.copyFrom( WMDefaults.gl_Bond );
        mybase.rounded = WMDefaults.gl_BondRound;
        mybase.exclude = WMDefaults.gl_BondExclude;
        mybase.dashed = WMDefaults.gl_BondDashed;
      case "COIL":
        mybase = new WMRibbon( false );
        mybase.copyFrom( WMDefaults.gl_Coil );
      case "RIBBON":
        mybase = new WMRibbon( true );
        mybase.copyFrom( WMDefaults.gl_Ribbon );
        mybase.thickness = WMDefaults.gl_RibbonThickness;
        mybase.smoothing = WMDefaults.gl_RibbonSmoothing;
      case "LABEL":
        mybase = new WMLabel();
        mybase.copyFrom( WMDefaults.gl_Label );
        mybase.font = WMDefaults.gl_LabelFont;
        mybase.fontsize = WMDefaults.gl_LabelSize;
      case "SHAPE":
        mybase = new WMShape();
        mybase.copyFrom( WMDefaults.gl_Shape );
      case "OBJ3D":
        mybase = new WMObject3D();
        mybase.copyFrom( WMDefaults.gl_Object3D );
        mybase.type = WMDefaults.gl_Object3DType;
    }
    mybase.loadFromXmlOverwrite( x );
  }
}
