/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile$
 *
 *  $Revision$
 *
 *  last change: $Author$ $Date$
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#ifndef SC_PHONETIC_HXX
#define SC_PHONETIC_HXX

#include <vector>

#ifndef _STRING_HXX
#include <tools/string.hxx>
#endif

struct PhoneticPortion
{
    sal_uInt16          mncpa;
    sal_uInt16          mncpm;
    sal_uInt16          mnccm;
    explicit inline     PhoneticPortion() : mncpa( 0 ), mncpm( 0 ), mnccm( 0 ) {}
    explicit inline     PhoneticPortion( sal_uInt16 ncpa, sal_uInt16 ncpm, sal_uInt16 nccm ) :
                            mncpa( ncpa ), mncpm( ncpm ), mnccm ( nccm ) {}
};

inline bool operator==( const PhoneticPortion& rLeft, const PhoneticPortion& rRight )
{
    return
        ( rLeft.mncpa == rRight.mncpa ) &&
        ( rLeft.mncpm == rRight.mncpm ) &&
        ( rLeft.mnccm == rRight.mnccm );
}

typedef ::std::vector< PhoneticPortion > PhoneticPortionVec;

/*
 * A class for store Asian phonetic guide information.
 */
class ScPhonetic
{
public:
	ScPhonetic() : mnFontIdx( 0 ), mnAdditionalSettings( 0 ), mnRepeatedTotalLength ( 0 ) {}
	/** A constructor of ScPhonetic.
	 * @param nFontIdx font index.
	 * @param nAdditionalSettings an additional settings.
	 * @param nRepeatedTotalLength the number of repeated total length.
	 * @param rPhoneticString phonetic text.
	 * @param rPhoneticPortions the portion of phonetic text.
	 */
	ScPhonetic( sal_uInt16 nFontIdx, sal_uInt16 nAdditionalSettings,
					sal_uInt16 nRepeatedTotalLength, const String& rPhoneticString,
					const PhoneticPortionVec& rPhoneticPortions) :
		mnFontIdx( nFontIdx ), mnAdditionalSettings( nAdditionalSettings ),
		mnRepeatedTotalLength ( nRepeatedTotalLength ),
		maPhoneticString ( rPhoneticString ), maPhoneticPortions ( rPhoneticPortions ) {}
	virtual ~ScPhonetic();

	virtual int             operator==( const ScPhonetic& ) const;

	/** Returns True if no phonetic information. */
	inline BOOL IsEmpty() const { return maPhoneticString.Len() == 0; }
	/** Returns the text for phonetic information. */
	inline const String& GetString() const { return maPhoneticString; }
    /** Returns the portion for phonetic information. */
	inline const PhoneticPortionVec& GetPortions() const { return maPhoneticPortions; }
    /** Returns font index of phonetic information. */
	inline sal_uInt16 GetFontIndex() const { return mnFontIdx; }
    /** Returns additional settings of phonetic information. */
	inline sal_uInt16 GetAdditionalSettings() const { return mnAdditionalSettings; }
    /** Returns repeated length of phonetic information. */
	inline sal_uInt16 GetRepeatedTotalLength() const { return mnRepeatedTotalLength; }
    /** Returns the size of phonetic information. */
	inline sal_uInt32 GetSize() const { return 14 + ( maPhoneticString.Len() ? maPhoneticString.Len() * 2 : 2 ) + maPhoneticPortions.size() * 6; }

private:
	sal_uInt16         mnFontIdx;            /// Index to FONT record
	sal_uInt16         mnAdditionalSettings; /// Additional settings for the Asian phonetic text
	sal_uInt16         mnRepeatedTotalLength;
	String             maPhoneticString;
	PhoneticPortionVec maPhoneticPortions;
};

#endif
