/***************************************************************************
                          cewf.cpp  -  description
                             -------------------
    begin                : 2004ǯ  5  4  17:25:12 JST
    copyright            : (C) 2004 by Tomoaki Murakami
    email                : solarisphere@yahoo.co.jp
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <qregexp.h>
#include <qvaluelist.h>
#include "cvoxstormglobal.h"
#include "cewf.h"

/////////////////////////////////////////////////////////////////////////////

const double CEwf::PositiveMinValue = 0.00000000001;

/////////////////////////////////////////////////////////////////////////////

bool 
CEwf::isEwf(const QString& strFileName)
{
  QFile File(strFileName);
  QStringList strlLines;

  if ( File.open(IO_ReadOnly) ) {
    QTextStream Stream(&File);
    QString     str;
    while ( !Stream.atEnd() ) {
      if ( valueType(Stream.readLine()) == Invalid) {
        File.close();
        return false;
      }
    }
    File.close();
    return true;
  } else {
    return false;
  }
}

/////////////////////////////////////////////////////////////////////////////

CEwf::ValueType 
CEwf::valueType(const QString& strLine)
{
  // is comment?
  if ( strLine[0] == '#' ) {
    return Comment;
  }

  QString strTmp = strLine.simplifyWhiteSpace();
  strTmp.replace(QChar(' '), "");

  // is empty
  if ( (strLine == "") || (strLine == " ") ) {
    return Comment;
  }

  // is source?
  if ( strLine.startsWith("source=") ) {
    return Source;
  }

  // is offset?
  if ( strLine.startsWith("offset=") ) {
    return Offset;
  }

  // is start-position?
  if ( strLine.startsWith("start-position=") ) {
    return StartPosition;
  }

  // is offset?
  if ( strLine.startsWith("length=") ) {
    return Length;
  }

  // is offset?
  if ( strLine.startsWith("looping=") ) {
    return Looping;
  }

  return Invalid;
}

/////////////////////////////////////////////////////////////////////////////

QString
CEwf::value(const QString& strLine)
{
  QString strValue = strLine.section(QChar('='), -1);
  while ( strValue[0] == QChar(' ') ){
    strValue = strValue.right(strValue.length());
  }

  return strValue;
}

/////////////////////////////////////////////////////////////////////////////

CEwf::CEwf()
  : QObject(), 
    strFileName(""), 
    TimeOffset(), 
    TimeStartPosition(), 
    TimeLength(), 
    bLooping(false)
{
}

/////////////////////////////////////////////////////////////////////////////

CEwf::CEwf(const CEwf& EwfSrc)
  : QObject(),
    strFileName(EwfSrc.strFileName),
    TimeOffset(EwfSrc.TimeOffset),
    TimeStartPosition(EwfSrc.TimeStartPosition),
    TimeLength(EwfSrc.TimeLength),
    bLooping(EwfSrc.bLooping)
{
}

/////////////////////////////////////////////////////////////////////////////

CEwf::CEwf(const QString& strEwfFileName)
  : QObject(), 
    strFileName(""), 
    TimeOffset(), 
    TimeStartPosition(), 
    TimeLength(), 
    bLooping(false)
{
  // validate
  if ( !isEwf(strEwfFileName) ) {
    return;
  }

  QFile File(strEwfFileName);
  if ( File.open(IO_ReadOnly) ) {
    QTextStream Stream(&File);
    QString     str;
    while ( !Stream.atEnd() ) {
      QTime TimeTmp;
      str = Stream.readLine();
      switch ( valueType(str) ) {
      case Source:
        strFileName = value(str);
        break;
      case Offset:
        TimeOffset = 
          TimeTmp.addMSecs(static_cast<int>(QString(value(str)).toDouble() * 1000.0));
        break;
      case StartPosition:
        TimeStartPosition = 
          TimeTmp.addMSecs(static_cast<int>(QString(value(str)).toDouble() * 1000.0));
        break;
      case Length:
        TimeLength = 
          TimeTmp.addMSecs(static_cast<int>(QString(value(str)).toDouble() * 1000.0));
        break;
      case Looping:
        if ( QString(value(str)) == "true" ) {
          bLooping = true;
        } else {
          bLooping = false;
        }
      default:
        break;
      }
    }
  } else {
    return;
  }
}

/////////////////////////////////////////////////////////////////////////////

CEwf::CEwf(const CTrackInf& TrackInf)
{
  set(TrackInf);
}

/////////////////////////////////////////////////////////////////////////////

CEwf::~CEwf()
{
}

/////////////////////////////////////////////////////////////////////////////

CEwf&
CEwf::operator=(const CEwf& EwfSrc)
{
  strFileName       = EwfSrc.strFileName;
  TimeOffset        = EwfSrc.TimeOffset;
  TimeStartPosition = EwfSrc.TimeStartPosition;
  TimeLength        = EwfSrc.TimeLength;
  bLooping          = EwfSrc.bLooping;

  return *this;
}

/////////////////////////////////////////////////////////////////////////////

QString
CEwf::fileName() const
{
  return strFileName;
}

/////////////////////////////////////////////////////////////////////////////

QTime
CEwf::offset() const
{
  return TimeOffset;
}

/////////////////////////////////////////////////////////////////////////////

QTime
CEwf::startPosition() const
{
  return TimeStartPosition;
}

/////////////////////////////////////////////////////////////////////////////

double
CEwf::offsetInDouble() const
{
  QTime TimeTmp;
  return ((static_cast<double>(TimeTmp.msecsTo(TimeOffset))) / 1000.0);
}

/////////////////////////////////////////////////////////////////////////////

double 
CEwf::startPositionInDouble() const
{
  QTime TimeTmp;
  return ((static_cast<double>(TimeTmp.msecsTo(TimeStartPosition))) / 1000.0);
}

/////////////////////////////////////////////////////////////////////////////

double
CEwf::lengthInDouble() const
{
  QTime TimeTmp;
  return ((static_cast<double>(TimeTmp.msecsTo(TimeLength))) / 1000.0);
}

/////////////////////////////////////////////////////////////////////////////

bool
CEwf::looping() const
{
  return bLooping;
}

/////////////////////////////////////////////////////////////////////////////

bool 
CEwf::hasAdvantage() const
{
  if ( bLooping ) {
    return true;
  }

  if ( offsetInDouble() > PositiveMinValue ) {
    return true;
  }

  if ( startPositionInDouble() > PositiveMinValue ) {
    return true;
  }

  if ( lengthInDouble() > PositiveMinValue ) {
    return true;
  }

  return false;
}

/////////////////////////////////////////////////////////////////////////////

void
CEwf::set(const CTrackInf& TrackInf)
{
  strFileName       = TrackInf.strFileName;
  TimeOffset        = TrackInf.TimeOffset;
  TimeStartPosition = TrackInf.TimeStartPosition;
  TimeLength        = TrackInf.TimeLength;
  bLooping          = false;
}

/////////////////////////////////////////////////////////////////////////////

void
CEwf::set(const QString& strSrcFile,
          const QTime& TimeOffsetFromOrigin,
          const QTime& TimePosition, 
          const QTime& TimePlayLength,
          bool bDoLoop)
{
  this->strFileName       = strSrcFile;
  this->TimeOffset        = TimeOffsetFromOrigin;
  this->TimeStartPosition = TimePosition;
  this->TimeLength        = TimePlayLength;
  this->bLooping          = bDoLoop;
}

/////////////////////////////////////////////////////////////////////////////

bool
CEwf::writeEcasoundWaveFile(const QString& strEwf)
{
  /*
     source = test.wav
     offset = 5.0
     start-position = 2.0
     length = 3.0
     looping = true
  */

  QFile fileEwf(strEwf);

  if ( !fileEwf.open(IO_WriteOnly | IO_Translate) ) {
    return false;
  }

  QTextStream out(&fileEwf);
  out.setEncoding(QTextStream::UnicodeUTF8);

  // source
  // if source file name is empty, return false
  if ( strFileName == "" ) {
    return false;
  }
  out << "source = ";
  out << strFileName << "\n";

  // offset
  out << "offset = ";
  out << QString("%1").arg(offsetInDouble(), 0, 'f') << "\n";

  // start-position
  out << "start-position = ";
  out << QString("%1").arg(startPositionInDouble(), 0, 'f') << "\n";

  // length
  out << "length = ";
  out << QString("%1").arg(lengthInDouble(), 0, 'f') << "\n";

  // looping
  out << "looping = ";
  if ( bLooping ) {
    out << "true\n";
  } else {
    out << "false\n";
  }

  if ( fileEwf.status() != IO_Ok ) {
    return false;
  }

  return true;
}

/////////////////////////////////////////////////////////////////////////////

#include "cewf.moc"
