/***************************************************************************
                          cecaiamanager.cpp  -  description
                             -------------------
    begin                : 2004ǯ  5  4  17:36:12 JST
    copyright            : (C) 2004 by Tomoaki Murakami
    email                : solarisphere@yahoo.co.jp
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <vector>
#include <qfile.h>
#include <qfileinfo.h>
#include <qregexp.h>
#include <qvaluelist.h>
#include <ktempfile.h>
#include "cvoxstormglobal.h"
#include "cecaiamanager.h"
#include "cewf.h"

/////////////////////////////////////////////////////////////////////////////

const QString CEcaIAManager::DQ("\"");
const Status  CEcaIAManager::Running(0);
const Status  CEcaIAManager::NotStarted(1);
const Status  CEcaIAManager::Stopped(2);
const Status  CEcaIAManager::Finished(3);
const Status  CEcaIAManager::NotReady(-1);
const Status  CEcaIAManager::Error(-2);

/////////////////////////////////////////////////////////////////////////////

CEcaIAManager::CEcaIAManager()
{
  nTimerIDFinished = 0;
  nSecondPosition = 0;
  bPlaying = false;
  //! \todo 󥻥åȥå "TEMP" ơ
  //! (ÿ֤)νԤ
}

/////////////////////////////////////////////////////////////////////////////

CEcaIAManager::~CEcaIAManager()
{
  //! \todo 󥻥åȥå "TEMP" 롣
}

/////////////////////////////////////////////////////////////////////////////

bool
CEcaIAManager::isExisting(const QString& strTrackName) const
{
  if ( getIndexByName(strTrackName) != -1 ) {
    return true;
  } else {
    return false;
  }
}

/////////////////////////////////////////////////////////////////////////////

bool
CEcaIAManager::isPlaying()
{
  if ( bPlaying ) {
    return true;
  }

  if ( engineStatus() == CEcaIAManager::Running ) {
    return true;
  } else {
    return false;
  }
}

/////////////////////////////////////////////////////////////////////////////

bool
CEcaIAManager::isValidChainsetup()
{
  command("cs-is-valid", false);
  int nValid = EcaControler.last_integer();

  if ( nValid == 0 ) {
    return false;
  }
  return true;
}

/////////////////////////////////////////////////////////////////////////////

int
CEcaIAManager::numberOfInputs()
{
  command("ai-list", false);
  std::vector<std::string> vstrList = EcaControler.last_string_list();

  return vstrList.size();
}

/////////////////////////////////////////////////////////////////////////////

int 
CEcaIAManager::numberOfOutputs()
{
  command("ao-list", false);
  std::vector<std::string> vstrList = EcaControler.last_string_list();

  return vstrList.size();
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotCreateNewChainsetup(const QString& strChainsetup)
{
  slotReleaseCurrentChainsetup();

  nSecondPosition = 0;
  QString strStatus = "cs-add " + strChainsetup;
  command(strStatus);
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotReleaseCurrentChainsetup()
{
  vTrackChains.clear();
  slotStop();
  command("cs-disconnect");
  command("cs-remove");
}

/////////////////////////////////////////////////////////////////////////////

void 
CEcaIAManager::slotPlay(bool bRecord)
{
  switch ( engineStatus() ) {
  case NotReady:
  case Error:
    return;
  case Finished:
    slotSetPosition(0);
    break;
  default:
    break;
  }

  setIOToCurrentChainsetup(bRecord);
  slotSetPosition(this->nSecondPosition);

  if ( !this->isValidChainsetup() ) {
    return;
  }

  killTimers(); // kill all timers
  nTimerIDFinished = startTimer(500); // 500-milisecond timer

  bPlaying = true;

  command("cs-connect");
  command("start");
}

/////////////////////////////////////////////////////////////////////////////

void 
CEcaIAManager::slotStop()
{
  killTimers(); // kill all timers
  command("cs-disconnect");
  slotSetPosition(this->nSecondPosition);
  // need to reconnect? !!!!!!!!!!!!!!!!!!!!!!!!!!!!
  removeTmpFiles();
  bPlaying = false;
  emit finished();
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotPause()
{
  killTimers(); // kill all timers
  command("stop");
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotSetPosition(const QTime& Time)
{
  int nSec = 0;

  nSec += Time.hour() * 3600;
  nSec += Time.minute() * 60;
  nSec += Time.second();

  slotSetPosition(nSec);
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotSetPosition(int nSec)
{
  command(QString("cs-set-position %1").arg(nSec));

  if ( nSec == this->nSecondPosition ) {
    return;
  }

  this->nSecondPosition = nSec;

  int nHour   = nSec / 3600;
  emit positionChanged(QTime(nHour, 
                             (nSec - (nHour * 3600)) / 60,
                             (nSec - (nHour * 3600)) % 60));
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotRewind(int nSecond)
{
  if ( isPlaying() ) {
    command("cs-rewind " + QString("%1").arg(nSecond));
  } else {
    this->nSecondPosition -= nSecond;
    if ( nSecondPosition < 0 ) {
      this->nSecondPosition = 0;
    }
    int nHour   = this->nSecondPosition / 3600;
    emit positionChanged(QTime(nHour,
                               (this->nSecondPosition - (nHour * 3600)) / 60,
                               (this->nSecondPosition - (nHour * 3600)) % 60));
  }
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotForward(int nSecond)
{
  if ( isPlaying() ) {
    command("cs-forward " + QString("%1").arg(nSecond));
  } else {
    this->nSecondPosition += nSecond;
    int nHour   = this->nSecondPosition / 3600;
    emit positionChanged(QTime(nHour,
                               (this->nSecondPosition - (nHour * 3600)) / 60,
                               (this->nSecondPosition - (nHour * 3600)) % 60));
  }
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotAddChain(const CTrackInf& TrackInfAdd)
{
  // if no input or no output, not be added
  QFileInfo FileInfo(TrackInfAdd.strFileName);
  if ( TrackInfAdd.strFileName == "" ) {
    return;
  } else if ( (!TrackInfAdd.bRecPort && !FileInfo.exists() ) || 
              !CVoxStormGlobal::isSupportedFileType(TrackInfAdd.strFileName) ) {
    return;
  } else {
    if ( TrackInfAdd.bRecPort ) {
      if ( TrackInfAdd.strInputDevice == "" ) {
        return;
      }
    } else {
      if ( TrackInfAdd.strOutputDevice == "" ) {
        return;
      }
    }
  }

  // should be push to vector AT FIRST for after process (other slots or functions)
  vTrackChains.push_back(TrackInfAdd);

  // create chain
  QString strCommand = "c-add " + TrackInfAdd.strTrackName;
  command(strCommand);

  // select chain
  slotSelectChain(TrackInfAdd.strTrackName);

  // setup mute condition !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  // if already this name track is existing...???
  if ( TrackInfAdd.bMute ) {
    slotToggleMute(TrackInfAdd.strTrackName, TrackInfAdd.bMute);
  }

  // NOTE : don't change order after parameter (level, pan, ...)
  // CEcaIAManager object processes implicitly this order
  // setup level
  strCommand = "cop-add -ea:" + QString("%1").arg(TrackInfAdd.nLevel);
  command(strCommand);
  slotChangeLevel(TrackInfAdd.strTrackName, TrackInfAdd.nLevel);

  // setup pan
  strCommand = "cop-add -epp:" + QString("%1").arg(TrackInfAdd.nPan);
  command(strCommand);
  slotChangePan(TrackInfAdd.strTrackName, TrackInfAdd.nPan);
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotRemoveChain(const QString& strTrackName)
{
  // select chain
  slotSelectChain(strTrackName);

  QString strCommand = "c-remove " + strTrackName;
  command(strCommand);

  QValueVector<CTrackInf>::iterator it = vTrackChains.begin();
  while ( it != vTrackChains.end() ) {
    if ( it->strTrackName == strTrackName ) {
      vTrackChains.erase(it);
      break;
    }
    it++;
  }
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotRemoveChain(int nID)
{
  if ( (nID < 0) || (nID >= vTrackChains.size()) ) {
    return;
  }

  slotRemoveChain(vTrackChains[nID].strTrackName);
}

/////////////////////////////////////////////////////////////////////////////

void 
CEcaIAManager::slotRemoveAllChains()
{
  int nCount = vTrackChains.size();
  int i;
  for ( i = 0; i < nCount; i++ ) {
    slotRemoveChain(i);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotRemoveAllInputs()
{
  int nInput = numberOfInputs();
  int i;

  for ( i = 0; i < nInput; i++ ) {
    command("ai-index-select 1");
    command("ai-remove");
  }
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotRemoveAllOutputs()
{
  int nOutput = numberOfOutputs();
  int i;

  for ( i = 0; i < nOutput; i++ ) {
    command("ao-index-select 1");
    command("ao-remove");
  }
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotChangePlaybackDevice(const QString& strTrackName,
                                        const QString& strDevice)
{
  int nIndex;

  if ( (nIndex = this->getIndexByName(strTrackName)) == -1 ) {
    return;
  }

  vTrackChains[nIndex].strOutputDevice = strDevice;

  // setup I/O file (device setting will be done slotPlay)
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotChangeRecordingDevice(const QString& strTrackName,
                                         const QString& strDevice)
{
  int nIndex;

  if ( (nIndex = this->getIndexByName(strTrackName)) == -1 ) {
    return;
  }

  vTrackChains[nIndex].strInputDevice = strDevice;

  // setup I/O file (device setting will be done slotPlay)
}

/////////////////////////////////////////////////////////////////////////////

void 
CEcaIAManager::slotChangeFileName(
  const QString& strTrackName, 
  const QString& strFileName)
{
  int nIndex;

  if ( (nIndex = this->getIndexByName(strTrackName)) == -1 ) {
    return;
  }

  if ( strFileName == "" ) { // removed file name
    slotRemoveChain(nIndex);
  }

  vTrackChains[nIndex].strFileName = strFileName;
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotChangeLevel(const QString& strTrackName, int nLevel)
{
  int nIndex;

  if ( (nIndex = this->getIndexByName(strTrackName)) == -1 ) {
    return;
  }

  vTrackChains[nIndex].nLevel = nLevel;

  // select chain
  slotSelectChain(strTrackName);

  // select level operator
  // NOTE : parameter id is ALWAYS 1 and 1
  QString strCommand = "cop-select 1";
  command(strCommand);
  // select level operator parameter
  strCommand = "copp-select 1";
  command(strCommand);

  // set level
  strCommand = "copp-set " + QString("%1").arg(nLevel);
  command(strCommand);
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotChangePan(const QString& strTrackName, int nPan)
{
  int nIndex;

  if ( (nIndex = this->getIndexByName(strTrackName)) == -1 ) {
    return;
  }

  vTrackChains[nIndex].nPan = nPan;

  // select chain
  slotSelectChain(strTrackName);

  // select pan operator
  // NOTE : parameter id is ALWAYS 2 and 1
  QString strCommand = "cop-select 2";
  command(strCommand);
  // select pan operator parameter
  strCommand = "copp-select 1";
  command(strCommand);

  // set pan
  strCommand = "copp-set " + QString("%1").arg(nPan);
  command(strCommand);
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotToggleMute(const QString& strTrackName, bool bMute)
{
  int nIndex;

  if ( (nIndex = this->getIndexByName(strTrackName)) == -1 ) {
    return;
  }

  vTrackChains[nIndex].bMute = bMute;

  slotSelectChain(strTrackName);
  command("c-muting");
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotToggleRec(const QString& strTrackName, bool bRec)
{
  int nIndex;

  if ( (nIndex = getIndexByName(strTrackName)) == -1 ) {
    return;
  }

  vTrackChains[nIndex].bRecPort = bRec;
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotChangeStartPosition(
                 const QString& strTrackName,
                 const QTime& StartPosition)
{
  int nIndex;

  if ( (nIndex = getIndexByName(strTrackName)) == -1 ) {
    return;
  }

  vTrackChains[nIndex].TimeStartPosition = StartPosition;
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotChangeOffset(
                 const QString& strTrackName,
                 const QTime& Offset)
{
  int nIndex;

  if ( (nIndex = getIndexByName(strTrackName)) == -1 ) {
    return;
  }

  vTrackChains[nIndex].TimeOffset = Offset;
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotChangeLength(
                 const QString& strTrackName,
                 const QTime& Length)
{
  int nIndex;

  if ( (nIndex = getIndexByName(strTrackName)) == -1 ) {
    return;
  }

  vTrackChains[nIndex].TimeLength = Length;
}

/////////////////////////////////////////////////////////////////////////////

inline void
CEcaIAManager::slotSelectChain(const QString& strTrackName)
{
  QString strCommand = "c-select " + strTrackName;
  command(strCommand);
}

/////////////////////////////////////////////////////////////////////////////

int
CEcaIAManager::getIndexByName(const QString& strTrackName) const
{
  int i, nTrack;

  nTrack = vTrackChains.size();
  for ( i = 0; i < nTrack; i++ ) {
    if ( strTrackName == vTrackChains[i].strTrackName ) {
      return i;
    }
  }

  return -1;
}

/////////////////////////////////////////////////////////////////////////////

inline void
CEcaIAManager::command(const QString& strCommand, bool bOut)
{
  EcaControler.command(strCommand.ascii());
  if ( bOut ) {
    printf("  %s\n", strCommand.ascii());
    emit messageGenerated(strCommand + "\n");
  }
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::setIOToCurrentChainsetup(bool bRecord)
{
  prepareTracks();

  int nChains, i;
  nChains = vPrepareTrackChains.size();

  QString      strCommand;
  QStringList  strlInputDevices;
  QStringList  strlOutputDevices;
  QStringList  strlInputFiles;
  QStringList  strlOutputFiles;
  int          nResult;

  // setup I/O
  slotRemoveAllInputs();
  slotRemoveAllOutputs();
  for ( i = 0; i < nChains; i++ ) {
    if ( bRecord && vPrepareTrackChains[i].bRecPort ) { // recording
      // setup input (device)
      nResult = CVoxStormGlobal::findStringFromList(
                              vPrepareTrackChains[i].strInputDevice,
                              strlInputDevices);
      if ( nResult == -1 ) {
        strlInputDevices.append(vPrepareTrackChains[i].strInputDevice);
        command("ai-add " + vPrepareTrackChains[i].strInputDevice);
      }

      // setup output (file)
      if ( vPrepareTrackChains[i].strFileName != "" ) {
        nResult = CVoxStormGlobal::findStringFromList(
                              vPrepareTrackChains[i].strFileName, 
                              strlOutputFiles);
        if ( nResult == -1 ) {
          strlOutputFiles.append(vPrepareTrackChains[i].strFileName);
          command("ao-add " + DQ + vPrepareTrackChains[i].strFileName + DQ);
        }
      }
    } else {                                     // playback
      // setup input (file)
      QFileInfo FileInfo(vPrepareTrackChains[i].strFileName);
      //      if ( vPrepareTrackChains[i].strFileName != "" ) {
      if ( FileInfo.exists() ) {
        nResult = CVoxStormGlobal::findStringFromList(
                              vPrepareTrackChains[i].strFileName,
                              strlInputFiles);
        if ( nResult == -1 ) {
          strlInputFiles.append(vPrepareTrackChains[i].strFileName);
          command("ai-add " + DQ + vPrepareTrackChains[i].strFileName + DQ);
        }
      }

      // setup output (device)
      nResult = CVoxStormGlobal::findStringFromList(
                              vPrepareTrackChains[i].strOutputDevice,
                              strlOutputDevices);
      if ( nResult == -1 ) {
        strlOutputDevices.append(vPrepareTrackChains[i].strOutputDevice);
        command("ao-add " + vPrepareTrackChains[i].strOutputDevice);
      }
    }
  }

  QString  strSelection;
  QStringList::Iterator itIO;
  QValueVector<CTrackInf>::Iterator itTrack = vPrepareTrackChains.begin();

  // >>>> attach inputs
  // attach intput devices
  itIO = strlInputDevices.begin();
  while ( itIO != strlInputDevices.end() ){
    command("ai-select " + CVoxStormGlobal::fenceString(*itIO));

    strSelection = "";
    itTrack = vPrepareTrackChains.begin();
    while ( itTrack != vPrepareTrackChains.end() ) {
      if ( bRecord && (*itTrack).bRecPort ) {
        if ( (*itIO) == (*itTrack).strInputDevice ) {
          strSelection += (*itTrack).strTrackName + ",";
        }
      }
      itTrack++;
    }

    slotSelectChain(strSelection.left(strSelection.length() - 1));
    command("ai-attach");
    itIO++;
  }

  // attach intput files
  itIO = strlInputFiles.begin();
  while ( itIO != strlInputFiles.end() ){
    command("ai-select " + CVoxStormGlobal::fenceString(*itIO));

    strSelection = "";
    itTrack = vPrepareTrackChains.begin();
    while ( itTrack != vPrepareTrackChains.end() ) {
      if ( !bRecord || !(*itTrack).bRecPort ) {
        if ( (*itIO) == (*itTrack).strFileName ) {
          strSelection += (*itTrack).strTrackName + ",";
        }
      }
      itTrack++;
    }

    slotSelectChain(strSelection.left(strSelection.length() - 1));
    command("ai-attach");
    itIO++;
  }
  // <<<< attach intputs

  // >>>> attach oututs
  // attach output devices
  itIO = strlOutputDevices.begin();
  while ( itIO != strlOutputDevices.end() ){
    command("ao-select " + CVoxStormGlobal::fenceString(*itIO));

    strSelection = "";
    itTrack = vPrepareTrackChains.begin();
    while ( itTrack != vPrepareTrackChains.end() ) {
      if ( !bRecord || !(*itTrack).bRecPort ) {
        if ( (*itIO) == (*itTrack).strOutputDevice ) {
          strSelection += (*itTrack).strTrackName + ",";
        }
      }
      itTrack++;
    }

    slotSelectChain(strSelection.left(strSelection.length() - 1));
    command("ao-attach");
    itIO++;
  }

  // attach output files
  itIO = strlOutputFiles.begin();
  while ( itIO != strlOutputFiles.end() ){
    command("ao-select " + CVoxStormGlobal::fenceString(*itIO));

    strSelection = "";
    itTrack = vPrepareTrackChains.begin();
    while ( itTrack != vPrepareTrackChains.end() ) {
      if ( bRecord && (*itTrack).bRecPort ) {
        if ( (*itIO) == (*itTrack).strFileName ) {
          strSelection += (*itTrack).strTrackName + ",";
        }
      }
      itTrack++;
    }

    slotSelectChain(strSelection.left(strSelection.length() - 1));
    command("ao-attach");
    itIO++;
  }
  // <<<< attach outputs

  destroyPreparings();
}

/////////////////////////////////////////////////////////////////////////////

Status
CEcaIAManager::engineStatus()
{
  command("engine-status", false);
  QString strStatus = EcaControler.last_string().c_str();

  if ( strStatus == "running" ) {
    return Running;
  }
  if ( strStatus == "not started" ) {
    return NotStarted;
  }
  if ( strStatus == "stopped" ) {
    return Stopped;
  }
  if ( strStatus == "finished" ) {
    return Finished;
  }
  if ( strStatus == "not ready" ) {
    return NotReady;
  }
  return Error;
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::timerEvent(QTimerEvent* TimerEvent)
{
  if ( TimerEvent->timerId() == nTimerIDFinished ) {
    if ( engineStatus() == Finished ) {
      killTimer(nTimerIDFinished);
      removeTmpFiles();
      bPlaying = false;
      emit finished();
    }
  }
  QObject::timerEvent(TimerEvent);
}

/////////////////////////////////////////////////////////////////////////////

inline void
CEcaIAManager::writeEwf(const QString& strTrackName, const QString& strFileName)
{
  writeEwf(getIndexByName(strTrackName), strFileName);
}

/////////////////////////////////////////////////////////////////////////////

void 
CEcaIAManager::writeEwf(int nIndex, const QString& strFileName)
{
  if ( (nIndex < 0) || (nIndex >= vTrackChains.size()) ) {
    return;
  }

  QFile fileEwf(strFileName);
  fileEwf.open(IO_WriteOnly);

  QString strBuffer = "source = ";
  strBuffer += DQ + vTrackChains[nIndex].strFileName + DQ + "\n";
  fileEwf.writeBlock(strBuffer.ascii(), qstrlen(strBuffer));

  QTime Time = vTrackChains[nIndex].TimeOffset;
  strBuffer = "offset = ";
  strBuffer += QString("%1").arg(Time.secsTo(Time)) + ".0\n";
  fileEwf.writeBlock(strBuffer.ascii(), qstrlen(strBuffer));

  Time = vTrackChains[nIndex].TimeStartPosition;
  strBuffer = "start-position = ";
  strBuffer += QString("%1").arg(Time.secsTo(Time)) + ".0\n";
  fileEwf.writeBlock(strBuffer.ascii(), qstrlen(strBuffer));

  Time = vTrackChains[nIndex].TimeLength;
  strBuffer = "length = ";
  strBuffer += QString("%1").arg(Time.secsTo(Time)) + ".0\n";
  fileEwf.writeBlock(strBuffer.ascii(), qstrlen(strBuffer));

  strBuffer = "looping = false\n";
  fileEwf.writeBlock(strBuffer.ascii(), qstrlen(strBuffer));

  fileEwf.close();
}

/////////////////////////////////////////////////////////////////////////////

void 
CEcaIAManager::removeTmpFiles()
{
  QFile fileTmp;
  QStringList::iterator it = strlTmpFiles.begin();

  while ( it != strlTmpFiles.end() ) {
    fileTmp.setName(*it);
    fileTmp.remove();
    it++;
  }

  strlTmpFiles.clear();
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::prepareTracks()
{
  vPrepareTrackChains.clear();

  int nTracks = vTrackChains.size();
  int i;

  for ( i = 0; i < nTracks; i++ ) {
    QFileInfo FileInfo(vTrackChains[i].strFileName);
    bool bTest;
    bTest = vTrackChains[i].bRecPort;
    bTest = FileInfo.exists();
    if ( (!vTrackChains[i].bRecPort && !FileInfo.exists()) ||
         !CVoxStormGlobal::isSupportedFileType(vTrackChains[i].strFileName) ) {
      continue;
    }

    CEwf Ewf(vTrackChains[i]);
    if ( Ewf.hasAdvantage() && !vTrackChains[i].bRecPort ) {
      KTempFile TempFile(QString::null, ".ewf");
      CTrackInf TrackEwf = vTrackChains[i];

      TempFile.close();
      TrackEwf.strFileName = TempFile.name();
      Ewf.writeEcasoundWaveFile(TempFile.name());

      vPrepareTrackChains.push_back(TrackEwf);
      strlTmpFiles.push_back(TrackEwf.strFileName);
    } else {
      vPrepareTrackChains.push_back(vTrackChains[i]);
    }
  }
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::destroyPreparings()
{
  vPrepareTrackChains.clear();
}

/////////////////////////////////////////////////////////////////////////////

#include "cecaiamanager.moc"
