/*
    w32loader
    copyright (c) 1998-2010 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "w32ldr.h"
#include "image.h"
#include "advapi32.h"
#include "commctrl.h"
#include "commdlg.h"
#include "gdi32.h"
#include "kernel32.h"
#include "msvcrt.h"
#include "ntdll.h"
#include "ole32.h"
#include "oleaut32.h"
#include "rasapi32.h"
#include "rasdlg.h"
#include "shell32.h"
#include "user32.h"
#include "version.h"
#include "winmm.h"
#include "winsock.h"
#include "misc/fileio.h"
#include "misc/peimage.h"
#include "misc/profile.h"
#ifdef HAVE_SYS_MMAN_H
# include <sys/mman.h>
#endif /* HAVE_SYS_MMAN_H */
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif /* HAVE_UNISTD_H */


#define USE_DEBUG 1


#define DLL_PROCESS_DETACH 0
#define DLL_PROCESS_ATTACH 1


/******************************************************************************
*                                                                             *
* ja:モジュール関数群                                                         *
*                                                                             *
******************************************************************************/
typedef BOOL CALLBACK (*DllEntryProc)(HMODULE, DWORD, LPVOID);


static PeProcess *process = NULL;


/*  ja:既にロードされているモジュールを取得する
    file,モジュール名
     RET,モジュール,NULL:ロードされていない                                 */
gpointer
w32ldr_module_get_library (const gchar *file)
{
  if (!process)
    process = g_malloc0 (sizeof (PeProcess));
  return peimage_module_get_library (process, file);
}


/*  ja:システムイメージをロードする
      process,プロセス
         file,イメージ名
    user_data,ユーザデータ
          RET,イメージ,NULL:エラー                                          */
static guint8 *
w32ldr_module_callback_system (PeProcess *process,
                               const gchar *file,
                               gpointer     user_data)
{
  guint8 *image = NULL;
  gint i;
  struct {
    const gchar *name;
    W32LdrExport *exports;
  } osdll[] = {
    {"advapi32.dll", w32ldr_advapi32_exports},
    {"comctl32.dll", w32ldr_commctrl_exports},
    {"comdlg32.dll", w32ldr_commdlg_exports},
    {"gdi32.dll",    w32ldr_gdi32_exports},
    {"kernel32.dll", w32ldr_kernel32_exports},
    {"msvcrt.dll",   w32ldr_msvcrt_exports},
    {"ntdll.dll",    w32ldr_ntdll_exports},
    {"ole32.dll",    w32ldr_ole32_exports},
    {"oleaut32.dll", w32ldr_oleaut32_exports},
    {"rasapi32.dll", w32ldr_rasapi32_exports},
    {"rasdlg.dll",   w32ldr_rasdlg_exports},
    {"shell32.dll",  w32ldr_shell32_exports},
    {"user32.dll",   w32ldr_user32_exports},
    {"version.dll",  w32ldr_version_exports},
    {"winmm.dll",    w32ldr_winmm_exports},
    {"wsock32.dll",  w32ldr_winsock_exports},
    {NULL,           NULL}};

  for (i = 0; osdll[i].name; i++)
    if (g_ascii_strcasecmp (osdll[i].name, file) == 0)
      {
        /* ja:内蔵DLL */
        guint8 *p;
        gchar *q;
        gint j, num = 0;
        gint ordinal_min = G_MAXINT, ordinal_max = G_MININT;
        gsize leng, leng_head, leng_code, leng_data;
        LPDWORD lpdwSignature, lpdwFunction, lpdwName;
        LPWORD lpwOrdinal;
        ImageDosHeader *idh;
        ImageFileHeader *ifh;
        ImageOptionalHeader *ioh;
        ImageSectionHeader *ish;
        ImageExportDirectory *ied;
#if defined (HAVE_SYS_MMAN_H) && defined (HAVE_UNISTD_H)
        gint pagesize;
        guint addr;
#endif /* defined (HAVE_SYS_MMAN_H) && defined (HAVE_UNISTD_H) */
#ifdef USE_DEBUG
        gchar *r;
#endif /* USE_DEBUG */

        leng_head = sizeof (ImageDosHeader)
                  + sizeof (DWORD)
                  + sizeof (ImageFileHeader)
                  + sizeof (ImageOptionalHeader)
                  + sizeof (ImageSectionHeader);
        leng_code = 0;
        leng_data = sizeof (ImageExportDirectory)
                                                + g_strlen (osdll[i].name) + 1;
        for (j = 0; osdll[i].exports[j].func; j++)
          if (osdll[i].exports[j].implement >= 0)
            {
#ifdef USE_DEBUG
              leng_code += osdll[i].exports[j].implement > 0
                                        ? osdll[i].exports[j].implement : 30;
#else /* not USE_DEBUG */
              leng_code += osdll[i].exports[j].implement > 0
                                        ? osdll[i].exports[j].implement : 5;
#endif /* not USE_DEBUG */
              if (osdll[i].exports[j].name)
                {
                  num++;
                  leng_data += g_strlen (osdll[i].exports[j].name) + 1
                                            + sizeof (DWORD) + sizeof (WORD);
                }
              if (ordinal_min > osdll[i].exports[j].ordinal)
                ordinal_min = osdll[i].exports[j].ordinal;
              if (ordinal_max < osdll[i].exports[j].ordinal)
                ordinal_max = osdll[i].exports[j].ordinal;
            }
        leng_data += (ordinal_max - ordinal_min + 1) * sizeof (DWORD);
        leng = leng_head + leng_code + leng_data;
        image = g_malloc0 (leng);
#if defined (HAVE_SYS_MMAN_H) && defined (HAVE_UNISTD_H)
        pagesize = sysconf (_SC_PAGE_SIZE);
        addr = GPOINTER_TO_UINT (image) / pagesize * pagesize;
        mprotect (GUINT_TO_POINTER (addr),
            (GPOINTER_TO_UINT (image) + leng + pagesize - 1)
            / pagesize * pagesize - addr, PROT_READ | PROT_WRITE | PROT_EXEC);
#endif /* defined (HAVE_SYS_MMAN_H) && defined (HAVE_UNISTD_H) */
        /* ja:ポインタ */
        idh = (ImageDosHeader *)image;
        lpdwSignature = (LPDWORD)(idh + 1);
        ifh = (ImageFileHeader *)(lpdwSignature + 1);
        ioh = (ImageOptionalHeader *)(ifh + 1);
        ish = (ImageSectionHeader *)(ioh + 1);
        p = (guint8 *)(ish + 1);
        ied = (ImageExportDirectory *)(p + leng_code);
        lpdwFunction = (LPDWORD)(ied + 1);
        lpdwName = lpdwFunction + ordinal_max - ordinal_min + 1;
        lpwOrdinal = (LPWORD)(lpdwName + num);
        q = (gchar *)(lpwOrdinal + num);
#ifdef USE_DEBUG
        r = q;
#endif /* USE_DEBUG */
        /* ja:設定 */
        idh_set_magic (idh, PEIMAGE_DOS_SIGNATURE);
        idh_set_lfanew (idh, sizeof (ImageDosHeader));
        *lpdwSignature = PEIMAGE_NT_SIGNATURE;
        ifh_set_machine (ifh, PEIMAGE_FILE_MACHINE_I386);
        ifh_set_number_of_sections (ifh, 1);
        ifh_set_size_of_optional_header (ifh, sizeof (ImageOptionalHeader));
        ifh_set_characteristics (ifh, PEIMAGE_FILE_EXECUTABLE_IMAGE
                                    | PEIMAGE_FILE_LINE_NUMS_STRIPPED
                                    | PEIMAGE_FILE_LOCAL_SYMS_STRIPPED
                                    | PEIMAGE_FILE_32BIT_MACHINE
                                    | PEIMAGE_FILE_DLL);
        ioh_set_magic (ioh, PEIMAGE_NT_OPTIONAL_HDR_MAGIC);
        ioh_set_size_of_code (ioh, leng_code);
        ioh_set_size_of_initialized_data (ioh, leng_data);
        ioh_set_base_of_code (ioh, leng_head);
        ioh_set_base_of_data (ioh, leng_head + leng_code);
        ioh_set_image_base (ioh, 0x10000000);
        ioh_set_section_alignment (ioh, 1);
        ioh_set_file_alignment (ioh, 1);
        ioh_set_major_operating_system_version (ioh, 4);
        ioh_set_minor_operating_system_version (ioh, 0);
        ioh_set_major_subsystem_version (ioh, 4);
        ioh_set_minor_subsystem_version (ioh, 0);
        ioh_set_size_of_image (ioh, leng);
        ioh_set_size_of_headers (ioh, leng_head);
        ioh_set_subsystem (ioh, PEIMAGE_SUBSYSTEM_WINDOWS_GUI);
        ioh_set_size_of_stack_reserve (ioh, 0x100000);
        ioh_set_size_of_stack_commit (ioh, 0x1000);
        ioh_set_size_of_heap_reserve (ioh, 0x100000);
        ioh_set_size_of_heap_commit (ioh, 0x1000);
        ioh_set_number_of_rva_and_sizes (ioh,
                                        PEIMAGE_NUMBEROF_DIRECTORY_ENTRIES);
        ioh_set_data_directory_virtual_address (ioh,
                        PEIMAGE_DIRECTORY_ENTRY_EXPORT, leng_head + leng_code);
        ioh_set_data_directory_size (ioh,
                                    PEIMAGE_DIRECTORY_ENTRY_EXPORT, leng_data);
        ish_set_virtual_size (ish, leng_code + leng_data);
        ish_set_virtual_address (ish, leng_head);
        ish_set_size_of_raw_data (ish, leng_code + leng_data);
        ish_set_pointer_to_raw_data (ish, leng_head);
        ish_set_characteristics (ish, PEIMAGE_SCN_CNT_CODE
                                    | PEIMAGE_SCN_CNT_INITIALIZED_DATA
                                    | PEIMAGE_SCN_MEM_EXECUTE
                                    | PEIMAGE_SCN_MEM_READ);
        ied_set_name(ied, GPOINTER_TO_UINT (q) - GPOINTER_TO_UINT (image));
        ied_set_base (ied, ordinal_min);
        ied_set_number_of_functions (ied, ordinal_max - ordinal_min + 1);
        ied_set_number_of_names (ied, num);
        ied_set_address_of_functions (ied, GPOINTER_TO_UINT (lpdwFunction)
                                                - GPOINTER_TO_UINT (image));
        ied_set_address_of_names (ied, GPOINTER_TO_UINT (lpdwName)
                                                - GPOINTER_TO_UINT (image));
        ied_set_address_of_name_ordinals (ied, GPOINTER_TO_UINT (lpwOrdinal)
                                                - GPOINTER_TO_UINT (image));
        g_strcpy (q, osdll[i].name);
        q += g_strlen (osdll[i].name) + 1;
        for (j = 0; osdll[i].exports[j].func; j++)
          if (osdll[i].exports[j].implement >= 0)
            {
              gint index;

              index = osdll[i].exports[j].ordinal - ordinal_min;
              lpdwFunction[index] = GPOINTER_TO_UINT (p)
                                                    - GPOINTER_TO_UINT (image);
              if (osdll[i].exports[j].implement > 0)
                {
                  g_memmove (p, osdll[i].exports[j].func,
                                                osdll[i].exports[j].implement);
                  p += osdll[i].exports[j].implement;
                }
              else
                {
#ifdef USE_DEBUG
                  const static gchar *fmt = "%s,%s\n";

                  *p++ = 0x60;
                  *p++ = 0x68;
                  *(guint32 *)p = GPOINTER_TO_UINT (q);
                  p += 4;
                  *p++ = 0x68;
                  *(guint32 *)p = GPOINTER_TO_UINT (r);
                  p += 4;
                  *p++ = 0x68;
                  *(guint32 *)p = GPOINTER_TO_UINT (fmt);
                  p += 4;
                  *p++ = 0xe8;
                  *(guint32 *)p = GPOINTER_TO_UINT (g_print)
                                                - (GPOINTER_TO_UINT (p) + 4);
                  p += 4;
                  *p++ = 0x83;
                  *p++ = 0xc4;
                  *p++ = 0x0c;
                  *p++ = 0x61;
#endif /* USE_DEBUG */
                  *p++ = 0xe9;
                  *(guint32 *)p = GPOINTER_TO_UINT (osdll[i].exports[j].func)
                                                - (GPOINTER_TO_UINT (p) + 4);
                  p += 4;
                }
              if (osdll[i].exports[j].name)
                {
                  *lpdwName = GPOINTER_TO_UINT (q) - GPOINTER_TO_UINT (image);
                  g_strcpy (q, osdll[i].exports[j].name);
                  q += g_strlen (osdll[i].exports[j].name) + 1;
                  *lpwOrdinal = index;
                  lpdwName++;
                  lpwOrdinal++;
                }
            }
        break;
      }
  return image;
}


/*  ja:再配置の差を求める
      process,プロセス
        image,PEイメージ
    user_data,ユーザデータ
          RET,差                                                            */
static gint
w32ldr_module_callback_relocale (PeProcess    *process,
                                 const guint8 *image,
                                 gpointer      user_data)
{
  return GPOINTER_TO_UINT (image) - pe_ioh_get_image_base (image);
}


/*  ja:DllMainを呼び出す
      process,プロセス
        image,PEイメージ
       reason,TRUE:ロード,FALSE:解放
    user_data,ユーザデータ
          RET,DllMainの返値                                                 */
static gboolean
w32ldr_module_callback_entry (PeProcess      *process,
                              const guint8   *image,
                              const gboolean  reason,
                              gpointer        user_data)
{
#if defined (HAVE_SYS_MMAN_H) && defined (HAVE_UNISTD_H)
  gint i, pagesize;
  gssize length = 0;
  guint16 sections;
  guint addr;
  ImageSectionHeader *ish;

  pagesize = sysconf (_SC_PAGE_SIZE);
  sections = pe_ifh_get_number_of_sections (image);
  ish = pe_image_section_header (image);
  for (i = 0; i < sections; i++)
    {
      gssize leng;

      leng = ish_get_virtual_size (ish) + ish_get_virtual_address (ish);
      if (length < leng)
        length = leng;
      ish++;
    }
  addr = GPOINTER_TO_UINT (image) / pagesize * pagesize;
  mprotect (GUINT_TO_POINTER (addr),
            (GPOINTER_TO_UINT (image) + length + pagesize - 1)
            / pagesize * pagesize - addr, PROT_READ | PROT_WRITE | PROT_EXEC);
#endif /* defined (HAVE_SYS_MMAN_H) && defined (HAVE_UNISTD_H) */
  if (pe_ifh_get_characteristics (image) & IMAGE_FILE_DLL)
    {
      DllEntryProc DllMain;

      DllMain = peimage_file_get_func (image, "DllMain");
      if (!DllMain && pe_ioh_get_size_of_headers (image)
                            < pe_ioh_get_address_of_entry_point (image)
            && pe_ioh_get_address_of_entry_point (image)
                                            < pe_ioh_get_size_of_image (image))
        DllMain = (DllEntryProc)(image
                                + pe_ioh_get_address_of_entry_point (image));
      if (DllMain)
        return DllMain ((HANDLE)image, reason ? DLL_PROCESS_ATTACH
                                              : DLL_PROCESS_DETACH, NULL);
    }
  return TRUE;
}


/*  ja:モジュールをロードする
    file,モジュール名
    exec,TRUE:DllMainを実行する,FALSE:ロードのみ
     RET,モジュール,NULL:エラー                                             */
gpointer
w32ldr_module_load_library (const gchar    *file,
                            const gboolean  exec)
{
  if (!process)
    process = g_malloc0 (sizeof (PeProcess));
  return peimage_module_load_library (process,
                                    file,
                                    w32ldr_module_callback_system,
                                    w32ldr_module_callback_relocale,
                                    NULL,
                                    exec ? w32ldr_module_callback_entry : NULL,
                                    NULL);
}


/*  ja:モジュールを解放する
    memory,モジュール
       RET,TRUE:正常終了,FALSE:エラー                                       */
gboolean
w32ldr_module_free (gpointer memory)
{
  if (!process)
    process = g_malloc0 (sizeof (PeProcess));
  return peimage_module_free_library (process, memory,
                                        w32ldr_module_callback_entry, NULL);
}


/*  ja:モジュールの名前を取得する
    memory,モジュール
       RET,名前,NULL:エラー                                                 */
const gchar *
w32ldr_module_get_filename (gpointer memory)
{
  if (!process)
    process = g_malloc0 (sizeof (PeProcess));
  return peimage_module_get_filename (process, memory);

}


/******************************************************************************
*                                                                             *
* ja:文字列関数群                                                             *
*                                                                             *
******************************************************************************/
/*  ja:UTF-8←MultiByte
     mb,MultiByte
    RET,UTF-8,NULL:エラー                                                   */
gchar *
w32ldr_utf8_from_mb (const gchar *mb)
{
  gssize leng;
  gchar *utf8str;
  gunichar2 *utf16str;

  if (!mb)
    return NULL;
  leng = MultiByteToWideChar (CP_ACP, 0, mb, -1, NULL, 0);
  utf16str = g_malloc (leng * sizeof (gunichar2));
  MultiByteToWideChar (CP_ACP, 0, mb, -1, utf16str, leng);
  utf8str = g_utf16_to_utf8 (utf16str, -1, NULL, NULL, NULL);
  g_free (utf16str);
  return utf8str;
}


/*  ja:UTF-8→MultiByte
    utf8str,UTF-8
        RET,MultiByte,NULL:エラー                                           */
gchar *
w32ldr_utf8_to_mb (const gchar *utf8str)
{
  gssize leng;
  gchar *mb;
  gunichar2 *utf16str;

  if (!utf8str)
    return NULL;
  utf16str = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  if (!utf16str)
    return NULL;
  leng = WideCharToMultiByte (CP_ACP, 0, utf16str, -1, NULL, 0, NULL, NULL);
  mb = g_malloc (leng * sizeof (gchar));
  WideCharToMultiByte (CP_ACP, 0, utf16str, -1, mb, leng, NULL, NULL);
  g_free (utf16str);
  return mb;
}


/*  ja:ファイル名←MultiByte
     mb,MultiByte
    RET,ファイル名,NULL:エラー                                              */
gchar *
w32ldr_filename_from_mb (const gchar *mb)
{
  gssize leng;
  gchar *file, *utf8str, *str;
  gunichar2 *utf16str;
  gint i = 0;

  if (!mb)
    return NULL;
  str = g_strdup (mb);
  while (str[i] != '\0')
    if (IsDBCSLeadByteEx (CP_ACP, str[i]))
      {
        i += 2;
      }
    else
      {
        if (str[i] == '\\')
          str[i] = G_DIR_SEPARATOR;
        i++;
      }
  leng = MultiByteToWideChar (CP_ACP, 0, str, -1, NULL, 0);
  utf16str = g_malloc (leng * sizeof (gunichar2));
  MultiByteToWideChar (CP_ACP, 0, str, -1, utf16str, leng);
  g_free (str);
  utf8str = g_utf16_to_utf8 (utf16str, -1, NULL, NULL, NULL);
  g_free (utf16str);
  if (!utf8str)
    return NULL;
  if (g_ascii_isalpha (utf8str[i]) && utf8str[i + 1] == ':')
    g_memmove (utf8str, utf8str + 2,
                                    (g_strlen (utf8str) - 1) * sizeof (gchar));
  str = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
  file = peimage_dir_get_case_insensitive (str);
  g_free (str);
  return file;
}


/*  ja:ファイル名→MultiByte
    file,ファイル名
     RET,MultiByte,NULL:エラー                                              */
gchar *
w32ldr_filename_to_mb (const gchar *file)
{
  gssize leng;
  gchar *mb, *utf8str;
  gunichar2 *utf16str;
  gint i = 0;

  if (!file)
    return NULL;
  utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
  if (!utf8str)
    return NULL;
  if (G_IS_DIR_SEPARATOR (utf8str[0]) && !G_IS_DIR_SEPARATOR (utf8str[1]))
    {
      gchar *tmp;

      tmp = g_strconcat ("c:", utf8str, NULL);
      g_free (utf8str);
      utf8str = tmp;
    }
  utf16str = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  leng = WideCharToMultiByte (CP_ACP, 0, utf16str, -1, NULL, 0, NULL, NULL);
  mb = g_malloc (leng * sizeof (gchar));
  WideCharToMultiByte (CP_ACP, 0, utf16str, -1, (LPSTR)mb, leng, NULL, NULL);
  g_free (utf16str);
  while (mb[i] != '\0')
    if (IsDBCSLeadByteEx (CP_ACP, mb[i]))
      {
        i += 2;
      }
    else
      {
        if (G_IS_DIR_SEPARATOR (mb[i]))
          mb[i] = '\\';
        i++;
      }
  return mb;
}


/*  ja:ファイル名←WideChar
     wc,WideChar
    RET,ファイル名,NULL:エラー                                              */
gchar *
w32ldr_filename_from_wc (const gunichar2 *wc)
{
  gchar *file, *utf8str, *str;
  gint i;

  if (!wc)
    return NULL;
  utf8str = g_utf16_to_utf8 (wc, -1, NULL, NULL, NULL);
  for (i = 0; utf8str[i] != '\0'; i++)
    if (utf8str[i] == '\\')
      utf8str[i] = G_DIR_SEPARATOR;
  if (g_ascii_isalpha (utf8str[i]) && utf8str[i + 1] == ':')
    g_memmove (utf8str, utf8str + 2,
                                    (g_strlen (utf8str) - 1) * sizeof (gchar));
  str = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
  file = peimage_dir_get_case_insensitive (str);
  g_free (str);
  return file;
}


/*  ja:ファイル名→WideChar
    file,ファイル名
     RET,WideChar,NULL:エラー                                               */
gunichar2 *
w32ldr_filename_to_wc (const gchar *file)
{
  gchar *utf8str;
  gunichar2 *wc;
  gint i;

  utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
  if (!utf8str)
    return NULL;
  for (i = 0; utf8str[i] != '\0'; i++)
    if (G_IS_DIR_SEPARATOR (utf8str[i]))
      utf8str[i] = '\\';
  if (utf8str[0] == '\\')
    {
      gchar *tmp;

      tmp = g_strconcat ("c:", utf8str, NULL);
      g_free (utf8str);
      utf8str = tmp;
    }
  wc = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
  return wc;
}


/******************************************************************************
*                                                                             *
* ja:tchar関数群                                                              *
*                                                                             *
******************************************************************************/
/*  ja:ソースコード文字列をMultiByteに変換する
    str,ソースコード文字列
    RET,MultiByte                                                           */
gchar *
w32ldr_tchar_to_mb (gchar *str)
{
  gchar *mb = NULL, *charset, *utf8str;
  static GHashTable *ghash = NULL;

  if (!str)
    return NULL;
  if (ghash)
    {
      mb = g_hash_table_lookup (ghash, str);
      if (mb)
        return mb;
    }
  else
    {
      ghash = g_hash_table_new (g_str_hash, g_str_equal);
    }
  /* ja:ANSIコードページ */
  charset = g_strdup_printf ("CP%d", GetACP ());
  utf8str = g_convert (str, -1, "UTF-8", charset, NULL, NULL, NULL);
  g_free (charset);
  if (utf8str)
    {
      g_free (utf8str);
      mb = str;
    }
  /* en:UTF-8 */
  if (!mb)
    mb = g_utf8_validate (str, -1, NULL) ? w32ldr_utf8_to_mb (str) : NULL;
  /* ja:ロケール */
  if (!mb)
    {
      gchar *utf8str;

      utf8str = g_locale_to_utf8 (str, -1, NULL, NULL, NULL);
      mb = w32ldr_utf8_to_mb (utf8str);
      g_free (utf8str);
    }
  /* ja:文字符号化方式不明 */
  if (!mb)
    {
      gint i;

      mb = g_strdup (str);
      for (i = 0; mb[i] != '\0'; i++)
        mb[i] &= 0x7f;
    }
  g_hash_table_insert (ghash, str, mb);
  return mb;
}


/*  ja:ソースコード文字列をUTF-16に変換する
    str,ソースコード文字列
    RET,UTF-16                                                              */
gunichar2 *
w32ldr_tchar_to_wc (gchar *str)
{
  gchar *charset;
  gunichar2 *wc;
  static GHashTable *ghash = NULL;

  if (!str)
    return NULL;
  if (ghash)
    {
      wc = g_hash_table_lookup (ghash, str);
      if (wc)
        return wc;
    }
  else
    {
      ghash = g_hash_table_new (g_str_hash, g_str_equal);
    }
  /* ja:ANSIコードページ */
  charset = g_strdup_printf ("CP%d", GetACP ());
  wc = (gunichar2 *)g_convert (str, -1, "UTF-16LE", charset, NULL, NULL, NULL);
  g_free (charset);
  /* en:UTF-8 */
  if (!wc)
    wc = g_utf8_to_utf16 (str, -1, NULL, NULL, NULL);
  /* ja:ロケール */
  if (!wc)
    {
      gchar *utf8str;

      utf8str = g_locale_to_utf8 (str, -1, NULL, NULL, NULL);
      wc = utf8str ? g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL) : NULL;
      g_free (utf8str);
    }
  /* ja:文字符号化方式不明 */
  if (!wc)
    {
      gint i = 0;

      wc = g_malloc ((g_strlen (str) + 1) * sizeof (gunichar2));
      do
        wc[i] = str[i] & 0x7f;
      while (str[i++] != '\0');
    }
  g_hash_table_insert (ghash, str, wc);
  return wc;
}


/******************************************************************************
*                                                                             *
* ja:アトム関数群                                                             *
*                                                                             *
******************************************************************************/
#define ATOMBASE 0x0100
typedef struct _W32LdrAtom
{
  gint counter;
  gchar *name;
  guint16 atom;
} W32LdrAtom;
static GList *glist_atom = NULL;
static W32LdrAtom atom_table[] = {
    {0, "BUTTON",    0x0080},
    {0, "EDIT",      0x0081},
    {0, "STATIC",    0x0082},
    {0, "LISTBOX",   0x0083},
    {0, "SCROLLBAR", 0x0084},
    {0, "COMBOBOX",  0x0085},
    {0, NULL,        0x0000}};


/*  ja:アトムを追加する
    name,名前(UTF-8)
     RET,アトム,0:エラー                                                    */
guint16
w32ldr_atom_add_name (const gchar *name)
{
  if (GPOINTER_TO_UINT (name) > G_MAXUINT16)
    {
      guint16 atom = ATOMBASE - 1;
      gint i;
      GList *glist;
      W32LdrAtom *w32atom;

      for (i = 0; atom_table[i].name; i++)
        if (g_ascii_strcasecmp (atom_table[i].name, name) == 0)
          return atom_table[i].atom;
      for (glist = g_list_first (glist_atom); glist;
                                                glist = g_list_next (glist))
        {
          w32atom = glist->data;
          if (g_ascii_strcasecmp (w32atom->name, name) == 0)
            {
              w32atom->counter++;
              return w32atom->atom;
            }
          if (atom < w32atom->atom)
            atom = w32atom->atom;
        }
      w32atom = g_malloc (sizeof (W32LdrAtom));
      w32atom->name = g_strdup (name);
      w32atom->counter = 1;
      w32atom->atom = atom + 1;
      glist_atom = g_list_append (glist_atom, w32atom);
      return w32atom->atom;
    }
  else if (GPOINTER_TO_UINT (name) > 0)
    {
      gint i;
      guint16 atom;
      GList *glist;

      atom = GPOINTER_TO_UINT (name);
      for (i = 0; atom_table[i].name; i++)
        if (atom_table[i].atom == atom)
          return atom;
      for (glist = g_list_first (glist_atom); glist;
                                                glist = g_list_next (glist))
        {
          W32LdrAtom *w32atom;

          w32atom = glist->data;
          if (w32atom->atom == atom)
            {
              w32atom->counter++;
              return atom;
            }
        }
    }
  return 0;
}


/*  ja:アトムを追加する
    name,名前(MultiByte)
     RET,アトム,0:エラー                                                    */
guint16
w32ldr_atom_add_name_mb (const gchar *name)
{
  guint16 atom;

  if (GPOINTER_TO_UINT (name) > G_MAXUINT16)
    {
      gchar *utf8str;

      utf8str = w32ldr_utf8_from_mb (name);
      atom = w32ldr_atom_add_name (utf8str);
      g_free (utf8str);
    }
  else
    {
      atom = w32ldr_atom_add_name (name);
    }
  return atom;
}


/*  ja:アトムを追加する
    name,名前(WideChar)
     RET,アトム,0:エラー                                                    */
guint16
w32ldr_atom_add_name_wc (const gunichar2 *name)
{
  guint16 atom;

  if (GPOINTER_TO_UINT (name) > G_MAXUINT16)
    {
      gchar *utf8str;

      utf8str = g_utf16_to_utf8 (name, -1, NULL, NULL, NULL);
      atom = w32ldr_atom_add_name (utf8str);
      g_free (utf8str);
    }
  else
    {
      atom = w32ldr_atom_add_name ((const gchar *)name);
    }
  return atom;
}


/*  ja:アトムをアトムから削除する
    atom,アトム
     RET,TRUE:正常終了,FALSE:エラー                                         */
gboolean
w32ldr_atom_delete_atom (const guint16 atom)
{
  GList *glist;

  for (glist = g_list_first (glist_atom); glist; glist = g_list_next (glist))
    {
      W32LdrAtom *w32atom;

      w32atom = glist->data;
      if (w32atom->atom == atom)
        {
          if (--w32atom->counter <= 0)
            {
              glist_atom = g_list_remove (glist_atom, w32atom);
              g_free (w32atom->name);
              g_free (w32atom);
            }
          return TRUE;
        }
    }
  return FALSE;
}


/*  ja:アトムを名前から削除する
    name,名前(UTF-8)
     RET,TRUE:正常終了,FALSE:エラー                                         */
gboolean
w32ldr_atom_delete_name (const gchar *name)
{
  return w32ldr_atom_delete_atom (w32ldr_atom_find_name (name));
}


/*  ja:アトムを名前から削除する
    name,名前(MultiByte)
     RET,TRUE:正常終了,FALSE:エラー                                         */
gboolean
w32ldr_atom_delete_name_mb (const gchar *name)
{
  return w32ldr_atom_delete_atom (w32ldr_atom_find_name_mb (name));
}


/*  ja:アトムを名前から削除する
    name,名前(WideChar)
     RET,TRUE:正常終了,FALSE:エラー                                         */
gboolean
w32ldr_atom_delete_name_wc (const gunichar2 *name)
{
  return w32ldr_atom_delete_atom (w32ldr_atom_find_name_wc (name));
}


/*  ja:アトムを検索する
    name,名前(UTF-8)
     RET,アトム,0:エラー                                                    */
guint16
w32ldr_atom_find_name (const gchar *name)
{
  if (GPOINTER_TO_UINT (name) > G_MAXUINT16)
    {
      gint i;
      GList *glist;
      W32LdrAtom *w32atom;

      for (i = 0; atom_table[i].name; i++)
        if (g_ascii_strcasecmp (atom_table[i].name, name) == 0)
          return atom_table[i].atom;
      for (glist = g_list_first (glist_atom); glist;
                                                glist = g_list_next (glist))
        {
          w32atom = glist->data;
          if (g_ascii_strcasecmp (w32atom->name, name) == 0)
            return w32atom->atom;
        }
    }
  else if (GPOINTER_TO_UINT (name) > 0)
    {
      gint i;
      guint16 atom;
      GList *glist;

      atom = GPOINTER_TO_UINT (name);
      for (i = 0; atom_table[i].name; i++)
        if (atom_table[i].atom == atom)
          return atom;
      for (glist = g_list_first (glist_atom); glist;
                                                glist = g_list_next (glist))
        {
          W32LdrAtom *w32atom;

          w32atom = glist->data;
          if (w32atom->atom == atom)
            return atom;
        }
    }
  return 0;
}


/*  ja:アトムを追加する
    name,名前(MultiByte)
     RET,アトム,0:エラー                                                    */
guint16
w32ldr_atom_find_name_mb (const gchar *name)
{
  guint16 atom;

  if (GPOINTER_TO_UINT (name) > G_MAXUINT16)
    {
      gchar *utf8str;

      utf8str = w32ldr_utf8_from_mb (name);
      atom = w32ldr_atom_find_name (utf8str);
      g_free (utf8str);
    }
  else
    {
      atom = w32ldr_atom_find_name (name);
    }
  return atom;
}


/*  ja:アトムを検索する
    name,名前(WideChar)
     RET,アトム,0:エラー                                                    */
guint16
w32ldr_atom_find_name_wc (const gunichar2 *name)
{
  guint16 atom;

  if (GPOINTER_TO_UINT (name) > G_MAXUINT16)
    {
      gchar *utf8str;

      utf8str = g_utf16_to_utf8 (name, -1, NULL, NULL, NULL);
      atom = w32ldr_atom_find_name (utf8str);
      g_free (utf8str);
    }
  else
    {
      atom = w32ldr_atom_find_name ((const gchar *)name);
    }
  return atom;
}


/*  ja:アトムの名前を取得する
    atom,アトム
     RET,名前(UTF-8),NULL:エラー                                            */
gchar *
w32ldr_atom_get_name (const guint16 atom)
{
  gint i;
  GList *glist;

  for (i = 0; atom_table[i].name; i++)
    if (atom_table[i].atom == atom)
      return g_strdup (atom_table[i].name);
  for (glist = g_list_first (glist_atom); glist; glist = g_list_next (glist))
    {
      W32LdrAtom *w32atom;

      w32atom = glist->data;
      if (w32atom->atom == atom)
        return g_strdup (w32atom->name);
    }
  return NULL;
}


/*  ja:アトムの名前を取得する
    atom,アトム
     RET,名前(MultiByte),NULL:エラー                                        */
gchar *
w32ldr_atom_get_name_mb (const guint16 atom)
{
  gchar *utf8str, *mb;

  utf8str = w32ldr_atom_get_name (atom);
  mb = w32ldr_utf8_to_mb (utf8str);
  g_free (utf8str);
  return mb;
}


/*  ja:アトムの名前を取得する
    atom,アトム
     RET,名前(WideChar),NULL:エラー                                         */
gunichar2 *
w32ldr_atom_get_name_wc (const guint16 atom)
{
  gchar *utf8str;
  gunichar2 *wc;

  utf8str = w32ldr_atom_get_name (atom);
  wc = utf8str ? g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL) : NULL;
  g_free (utf8str);
  return wc;
}


/*  ja:名前から文字列を取得する
    name,名前(UTF-8)
     RET,文字列(UTF-8),NULL:エラー                                          */
gchar *
w32ldr_atom_get_string (const gchar *name)
{
  return GPOINTER_TO_UINT (name) > G_MAXUINT16 ? g_strdup (name)
                            : w32ldr_atom_get_name (GPOINTER_TO_UINT (name));
}


/*  ja:名前から文字列を取得する
    name,名前(UTF-8)
     RET,文字列(MultiByte),NULL:エラー                                      */
gchar *
w32ldr_atom_get_string_mb (const gchar *name)
{
  gchar *mb;

  if (GPOINTER_TO_UINT (name) > G_MAXUINT16)
    {
      mb = w32ldr_utf8_to_mb (name);
    }
  else
    {
      gchar *utf8str;

      utf8str = w32ldr_atom_get_name (GPOINTER_TO_UINT (name));
      mb = w32ldr_utf8_to_mb (name);
      g_free (utf8str);
    }
  return mb;
}


/*  ja:名前から文字列を取得する
    name,名前(UTF-8)
     RET,文字列(WideChar),NULL:エラー                                       */
gunichar2 *
w32ldr_atom_get_string_wc (const gchar *name)
{
  gunichar2 *wc;

  if (GPOINTER_TO_UINT (name) > G_MAXUINT16)
    {
      wc = g_utf8_to_utf16 (name, -1, NULL, NULL, NULL);
    }
  else
    {
      gchar *utf8str;

      utf8str = w32ldr_atom_get_name (GPOINTER_TO_UINT (name));
      wc = utf8str ? g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL) : NULL;
      g_free (utf8str);
    }
  return wc;
}
